/* ***************************************************************************************
* FILE:          RtDocRange.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtHighlighter is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once

#include <FeatStd/Util/Rtti.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>
#include <Widgets/2D/RichText/DocumentModel/RtDocElement.h>
#include <Widgets/2D/RichText/DocumentModel/RtDocument.h>

namespace hmibase {
namespace widget {
namespace richtext {

/**
 * This class represents a portion of a document.
 */
class DocRange
{
   public:
      FEATSTD_RTTI_DECLARATION();
      FEATSTD_TYPEDEF_SHARED_POINTER(DocRange);
      // The list type for marked elements.
      typedef FeatStd::Internal::Vector<DocElement::SharedPointer> DocElemList;

      DocRange();
      virtual ~DocRange();

      /**
       * Provides the number of atomic parts from the beginning of the first paragraph
       * item within marker range that are NOT part of the marker.
       *
       * @return The index of the first atomic part of the first paragraph item that
       * is contained in the range.
       */
      FeatStd::UInt32 GetStart() const;

      /**
       * Provides the number of atomic parts from the beginning of the last paragraph item
       * within marker range that ARE part of the marker. All subsequent atomic parts
       * are not part of the range.
       *
       * @return The index of the last atomic part from the beginning of the last paragraph item that
       * is contained in the range.
       */
      FeatStd::UInt32 GetEnd() const;

      /**
       * Provides all document elements within the range. The first item is may be
       * reduced from the beginning as indicated by 'GetStart', the last item within
       * the range may be reduced from the end as indicated by 'GetLimit'.
       *
       * @return The reference to the order list of document elements included in the range.
       */
      const DocElemList& GetElements() const;

      /**
       * Defines whether or not the range shall be visualized.
       *
       * @param show True to render the range, false otherwise.
       */

      void SetVisibility(bool show);

      /**
       * Determines whether or not the range shall be visualized.
       *
       * @return True (default) if the range shall be rendered, false otherwise.
       */
      bool GetVisibility() const;

   protected:

      /**
       * Defines the number of atomic parts from the beginning of the first paragraph item
       * within marker range that are NOT part of the marker.
       *
       * @param start   The index of the first atomic part from the beginning of the first
       *                paragraph item that is included in the range.
       */
      virtual void SetStart(FeatStd::UInt32 start);

      /**
       * Defines the number of atomic parts from the beginning of the last paragraph item within
       * marker range that are part of the range.
       *
       * @param end     The index of the last atomic part of the last paragraph item that is
       *                included in the range.
       */
      virtual void SetEnd(FeatStd::UInt32 end);

      /**
       * Adds the given document element to the end of the range.
       * @param element The element to add.
       */
      virtual void AddElement(const DocElement::SharedPointer& element);

      /**
       * Removes all document elements from the range.
       */
      virtual void ClearRange();

   private:
      friend class Marker;
      // Support smart references
      FEATSTD_SHARED_POINTER_DECLARATION();

      // the ordered list of document elements in the range
      DocElemList m_elems;
      // the start index, see GetStart
      FeatStd::UInt32 m_start;
      // the end index, see GetEnd
      FeatStd::UInt32 m_end;
      // true to visualize the range, false otherwise.
      bool m_visibility;
};


}
}


}
