/* ***************************************************************************************
* FILE:          RtDocAccessor.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtDocAccessor is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once

#include <Candera/System/Mathematics/Vector2.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>

#include <Widgets/2D/RichText/Engine/RtEngineModuleBase.h>
#include <Widgets/2D/RichText/DocumentModel/RtDocElement.h>
#include <Widgets/2D/RichText/DocumentModel/RtDocument.h>
#include <Widgets/2D/RichText/DocumentModel/RtChapter.h>
#include <Widgets/2D/RichText/DocumentModel/RtParagraph.h>

namespace hmibase {
namespace widget {
namespace richtext {

/**
 * This class provides basic access to the underlying rich text document
 * object model.
 */
class DocAccessor: public RichTextEngineModuleBase
{
      FEATSTD_TYPEDEF_BASE(RichTextEngineModuleBase);
   public:
      FEATSTD_RTTI_DECLARATION();
      FEATSTD_TYPEDEF_SHARED_POINTER(DocAccessor);

      /** Destructor */
      virtual ~DocAccessor();

      /**
       * Provides the reference to the underlying document.
       *
       * @return The underlying document.
       */
      virtual Document::SharedPointer GetDocument() = 0;

      /**
       * Provides access to the very first Chapter of the underlying document.
       * The hierarchy of chapters define the document outline.
       *
       * @return The reference to the very first chapter of the document.
       */
      virtual Chapter::SharedPointer GetOutline() = 0;

      /**
       * Provides the Chapter object to which the given element belongs to.
       *
       * @param element The element for which the Chapter object shall be retrieved.
       * @return The Chapter object that holds the given document element or
       *         an empty reference if the element is either invalid or does
       *         belong to the underlying document.
       */
      virtual Chapter::SharedPointer GetChapter(DocElement::SharedPointer element) = 0;

      /**
       * Provides the Paragraph object to which the given element belongs to.
       *
       * @param element The element for which the Paragraph object shall be retrieved.
       * @return The Paragraph object that holds the given document element or
       *         an empty reference if the element is either invalid,
       *         is not part of a paragraph, or does
       *         belong to the underlying document.
       */
      virtual Paragraph::SharedPointer GetParagraph(DocElement::SharedPointer element) = 0;

      /**
       * Provides the most atomic element of the underlying document
       * that matches the position relative to the upper left
       * corner of the document. The on-screen position is automatically
       * converted into document-relative position if parameter
       * 'screenCoordinates' is set to 'true'.
       * If the given position is outside the document content, an empty
       * reference is provided.
       *
       * @note: On-screen coordinates must be converted to document-related
       * coordinates using conversion functions provided by Viewport class
       * if the parameter 'screenCoordinates' is set to 'false'.
       *
       * @param posX The value of the horizontal coordinate in pixel
       *             of the document element to retrieve.
       * @param posY The value of the vertical coordinate in pixel
       *             of the document element to retrieve.
       * @param screenCoordinates True (default) if posX and posY
       *             are provided as on-screen coordinates, false if they
       *             are provided relative to the upper left corner of the
       *             document.
       * @return The document element at the given position or an empty
       * reference if no document element is placed at the given position.
       */
      virtual DocElement::SharedPointer GetElementAt(FeatStd::Int32 posX, FeatStd::Int32 posY, bool screenCoordinates) = 0;

      /**
       * Provides the most atomic element of the underlying document
       * that matches the position relative to the upper left
       * corner of the document. The on-screen position is automatically
       * converted into document-relative position if parameter
       * 'screenCoordinates' is set to 'true'.
       * If the given position is outside the document content, an empty
       * reference is provided.
       *
       * @note: On-screen coordinates must be converted to document-related
       * coordinates using conversion functions provided by Viewport class
       * if the parameter 'screenCoordinates' is set to 'false'.
       *
       * @param position The position in pixels of the document element to retrieve.
       * @param screenCoordinates True (default) if posX and posY
       *             are provided as on-screen coordinates, false if they
       *             are provided relative to the upper left corner of the
       *             document.
       * @return The document element at the given position or an empty
       * reference if no document element is placed at the given position.
       */
      virtual DocElement::SharedPointer GetElementAt(const Candera::Vector2& position, bool screenCoordinates) = 0;

   protected:

      ///Restricted constructor.
      DocAccessor();

   private:
};


} // namespace richtext
} // namespace widget
} // namespace hmibase
