/* ***************************************************************************************
* FILE:          RtHighlighterImpl.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtHighlighterImpl is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "RtHighlighterImpl.h"

#include <Candera/System/Mathematics/Rectangle.h>
#include <Candera/System/Mathematics/Vector2.h>
#include <Candera/TextEngine/Internal/FullBoundCodePointIterator.h>
#include <CanderaPlatform/Device/Common/Effects/TextBrushCache/BitmapTextBrushCache.h>

#include <Widgets/2D/RichText/Engine/RtEngine.h>
#include <Widgets/2D/RichText/DocumentModel/RtDocElementTraverser.h>
#include <Widgets/2D/RichText/Utils/RtCaseFolding.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_RICHTEXT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/RtHighlighterImpl.cpp.trc.h"
#endif

namespace hmibase {
namespace widget {
namespace richtext {

using FeatStd::UInt8;
using FeatStd::UInt16;
using FeatStd::Int32;
using FeatStd::UInt32;
using FeatStd::Float;

using Candera::Rectangle;
using Candera::Vector2;

using Candera::TextRendering::Internal::FullBoundCodePointIterator;

HighlighterImpl::IndexSearcher::IndexSearcher(const SearchIndexList& searchList, const HighlighterImpl& highlighter) :
   Base(),
   m_searchList(searchList),
   m_curIndex(0),
   m_processedCharCount(0),
   m_docRange(0),
   m_highlighter(highlighter)
{
}


HighlighterImpl::IndexSearcher::~IndexSearcher()
{
   if (m_docRange != 0)
   {
      // notify incomplete match at end of document
      m_docRange->SetEnd(m_processedCharCount);
      m_highlighter.OnMatch(DocRange::SharedPointer(m_docRange));
      m_docRange = 0;
   }
}


bool HighlighterImpl::IndexSearcher::Process(const DocElement& docElement, const Candera::Rectangle& /*effectiveRect*/)
{
   const Text* constTextElem = Candera::Dynamic_Cast<const Text*>(&docElement);
   if (0 != constTextElem)
   {
      Text* textElem = const_cast<Text*>(constTextElem);
      MatchDetectorAttribs detectorAttribs;
      detectorAttribs.m_isStartInElem = false;
      detectorAttribs.m_isElemAdded = false;
      detectorAttribs.m_remainingPartCnt = textElem->GetPartCount();
      detectorAttribs.m_elemPartCnt = textElem->GetPartCount();

      while ((m_curIndex < m_searchList.Size()) && (detectorAttribs.m_remainingPartCnt > 0))
      {
         detectorAttribs.m_isStartInElem = m_searchList[m_curIndex].m_startIdx < (m_processedCharCount + detectorAttribs.m_elemPartCnt);

         // no match in this element
         if ((0 == m_docRange) && (!detectorAttribs.m_isStartInElem))
         {
            detectorAttribs.m_remainingPartCnt = 0;
         }
         else
         {
            // there are potentially multiple matches in the element
            ProcessPotentialMatch(textElem, detectorAttribs);
         }
      }
      if (detectorAttribs.m_remainingPartCnt == 0)
      {
         m_processedCharCount += detectorAttribs.m_elemPartCnt;
      }
   }

   return (m_curIndex >= m_searchList.Size()); // abort if end of search list is reached
}


void HighlighterImpl::IndexSearcher::ProcessPotentialMatch(Text* textElem, MatchDetectorAttribs& detectorAttribs)
{
   if ((0 == m_docRange) && detectorAttribs.m_isStartInElem)
   {
      m_docRange = FEATSTD_NEW(HighlightDocRange);
      if (m_docRange != 0)
      {
         m_docRange->AddElement(Text::SharedPointer(textElem));
         int start = m_searchList[m_curIndex].m_startIdx - m_processedCharCount;

         m_docRange->SetStart(start);
         detectorAttribs.m_remainingPartCnt = ((detectorAttribs.m_elemPartCnt - start) - 1);
         detectorAttribs.m_isElemAdded  = true;

         if (!m_searchList[m_curIndex].IsRange())
         {
            UInt32 end = start;
            detectorAttribs.m_remainingPartCnt = FinalizeMatch(textElem->GetText().GetCString(), end, detectorAttribs.m_elemPartCnt);
         }
      }
   }

   if ((0 != m_docRange) && m_searchList[m_curIndex].IsRange())
   {
      if (!detectorAttribs.m_isElemAdded)
      {
         // could be the first one or a complete adjacent one, don't add the first one again
         m_docRange->AddElement(Text::SharedPointer(textElem));
         detectorAttribs.m_isElemAdded  = true;
      }

      if (m_searchList[m_curIndex].m_endIdx >= (m_processedCharCount + detectorAttribs.m_elemPartCnt))
      {
         // the complete remaining element is part of the match
         detectorAttribs.m_remainingPartCnt = 0;
      }
      else
      {
         // only the first part of the element is part of the match
         UInt32 end = m_searchList[m_curIndex].m_endIdx - m_processedCharCount;
         detectorAttribs.m_remainingPartCnt = FinalizeMatch(textElem->GetText().GetCString(), end, detectorAttribs.m_elemPartCnt);
      }
   }
}


UInt32 HighlighterImpl::IndexSearcher::FinalizeMatch(const FeatStd::TChar* /*text*/, UInt32 end, UInt32 elemPartCnt)
{
   m_docRange->SetEnd(end);
   UInt32 remainingCharCnt = ((elemPartCnt - end) - 1);
   m_highlighter.OnMatch(DocRange::SharedPointer(m_docRange));
   m_docRange = 0;
   m_curIndex++;
   return remainingCharCnt;
}


HighlighterImpl::TextSearcher::TextSearcher(SearchTextList& searchList, const HighlighterImpl& highlighter) :
   Base(),
   m_searchList(searchList),
   m_docRange(0),
   m_highlighter(highlighter)
{
}


HighlighterImpl::TextSearcher::~TextSearcher()
{
   if (m_docRange != 0)
   {
      // notify incomplete match at end of document
      m_highlighter.OnMatch(DocRange::SharedPointer(m_docRange));
      m_docRange = 0;
   }
}


bool HighlighterImpl::TextSearcher::Process(const DocElement& docElement, const Candera::Rectangle& /*effectiveRect*/)
{
   bool allDone = false;  // criterion for abort traversing
   const Text* constTextElem = Candera::Dynamic_Cast<const Text*>(&docElement);
   if (0 != constTextElem)
   {
      Text* textElem = const_cast<Text*>(constTextElem);

      for (UInt32 i = 0; i < m_searchList.Size(); i++)
      {
         bool enough = (m_searchList[i].m_maxOccurencies > 0) && (m_searchList[i].m_occurencies >= m_searchList[i].m_maxOccurencies);
         UInt32 matchIdx = 0;
         bool match = false;

         UInt32 textIdx = 0;
         const bool caseSensitive = m_searchList[i].m_caseSensitive;
         const FeatStd::TChar* mask = m_searchList[i].m_mask.GetCString();
         const FeatStd::TChar* text = textElem->GetText().GetCString();

         FullBoundCodePointIterator itText(text, -1);
         FullBoundCodePointIterator itMask(mask, -1);

         while ((!enough) && (*itMask != 0) && (*itText != 0))
         {
            // find candidate for a match
            while ((*itText != 0) && (!Match(*itMask, *itText, caseSensitive)))
            {
               itText.Advance();
               textIdx++;
            }
            // evaluate candidate further
            matchIdx = textIdx;
            while ((*itMask != 0) && (*itText != 0) && Match(*itMask, *itText, caseSensitive))
            {
               itMask.Advance();

               itText.Advance();
               textIdx++;
            }
            // finalize match detection
            match = (*itMask == 0);
            // create doc range for the match
            if (match)
            {
               m_docRange = FEATSTD_NEW(HighlightDocRange);
               m_docRange->AddElement(Text::SharedPointer(textElem));
               m_docRange->SetStart(matchIdx);
               m_docRange->SetEnd(textIdx - 1);
               m_searchList[i].m_occurencies++;
               m_highlighter.OnMatch(DocRange::SharedPointer(m_docRange));
               match = false;
               m_docRange = 0;
            }
            itMask = FullBoundCodePointIterator(mask, -1);
            enough = (m_searchList[i].m_maxOccurencies > 0) && (m_searchList[i].m_occurencies >= m_searchList[i].m_maxOccurencies);
            allDone = ((i == 0) && enough) || (allDone && enough);
         }
      }
   }
   return allDone;
}


bool HighlighterImpl::TextSearcher::Match(Candera::TextRendering::CodePoint c1, Candera::TextRendering::CodePoint c2, bool caseSensitive) const
{
   if (caseSensitive)
   {
      return c1 == c2;
   }
   else
   {
      return CaseFolding::ToLower(c1) == CaseFolding::ToLower(c2);
   }
}


FEATSTD_RTTI_DEFINITION(HighlighterImpl, Base);


HighlighterImpl::SharedPointer HighlighterImpl::Create(Engine* engine)
{
   HighlighterImpl::SharedPointer sp = HighlighterImpl::SharedPointer(FEATSTD_NEW(HighlighterImpl));
   if (!sp.PointsToNull())
   {
      sp->AttachEngine(engine);
   }
   return sp;
}


HighlighterImpl::HighlighterImpl() :
   Base()
{
}


HighlighterImpl::~HighlighterImpl()
{
}


void HighlighterImpl::MarkAll(const FeatStd::String& searchCriteria, ElementStyle::SharedPointer highlightStyle)
{
   Engine* engine = GetEngine();
   if (0 != engine)
   {
      m_highlightStyle = highlightStyle;

      SearchIndexList searchIndexList;
      SearchTextList searchTextList;

      if (!searchCriteria.IsEmpty())
      {
         ProcessCombinedCriteriaText(searchCriteria, searchIndexList, searchTextList);
      }

      // Completed criteria string parsing, now search the document.
      DoSearch(&searchIndexList, &searchTextList);
   }
}


void HighlighterImpl::ProcessCombinedCriteriaText(const FeatStd::String& searchCriteria, SearchIndexList& searchIndexList, SearchTextList& searchTextList)
{
   enum Mode
   {
      TextMode,
      IndexMode
   };
   IndexSearchAttribs searchAttribs(0, 0, false, false);
   const FeatStd::SizeType criteriaLength = searchCriteria.GetCharCount();
   FeatStd::TChar* textMask = FEATSTD_NEW_ARRAY(FeatStd::TChar, criteriaLength);
   FeatStd::SizeType textMaskLen = 0;
   Mode mode = IndexMode;
   const FeatStd::TChar* criteria = searchCriteria.GetCString();

   // parse search criteria, start with index mode
   for (FeatStd::UInt i = 0; (i < criteriaLength) && (criteria[i] != '\0'); i++)
   {
      // relaxed implementation of syntax, supports also ill-formed expressions
      // such as "1'ab'34-" to visualize even erroneous expressions.
      if (criteria[i] == '\'')
      {
         if (mode == IndexMode)
         {
            FinalizeSearchIndex(searchIndexList, searchAttribs);
         }
         else
         {
            FinalizeSearchIndex(searchTextList, textMask, textMaskLen);
         }
         mode = (mode == TextMode ? IndexMode : TextMode);
      }
      else if (criteria[i] == ',')
      {
         if (mode == IndexMode)
         {
            FinalizeSearchIndex(searchIndexList, searchAttribs);
         }
         else
         {
            FinalizeSearchIndex(searchTextList, textMask, textMaskLen);
         }
      }
      else if (mode == IndexMode)
      {
         CalcIndexSearchAttribs(criteria[i], searchAttribs);
      }
      else
      {
         // TextMode
         // append the character to the text mask
         // text mask array len is equal to length of criterion string,
         //therefore no overflow check needed
         textMask[textMaskLen] = criteria[i];
         textMaskLen++;
      }
   }

   // end of criterion string, finalize last token
   if (mode == IndexMode)
   {
      FinalizeSearchIndex(searchIndexList, searchAttribs);
   }
   else
   {
      FinalizeSearchIndex(searchTextList, textMask, textMaskLen);
   }
   FEATSTD_DELETE_ARRAY(textMask);
}


void HighlighterImpl::MarkRange(const FeatStd::String& range, ElementStyle::SharedPointer highlightStyle)
{
   Engine* engine = GetEngine();
   if (0 != engine)
   {
      m_highlightStyle = highlightStyle;
      IndexSearchAttribs searchAttribs(0, 0, false, false);

      SearchIndexList searchIndexList;

      if (!range.IsEmpty())
      {
         const FeatStd::SizeType criteriaLength = range.GetCharCount();
         const FeatStd::TChar* criteria = range.GetCString();

         // parse single indexes and ranges
         for (FeatStd::UInt i = 0; (i < criteriaLength) && (criteria[i] != '\0'); i++)
         {
            // relaxed implementation of syntax, supports also ill-formed expressions
            if (',' == criteria[i])
            {
               FinalizeSearchIndex(searchIndexList, searchAttribs);
            }
            else
            {
               CalcIndexSearchAttribs(criteria[i], searchAttribs);
            }
         }

         // end of criterion string, finalize last token
         FinalizeSearchIndex(searchIndexList, searchAttribs);
      }

      // Completed criteria string parsing, now search the document.
      DoSearch(&searchIndexList, 0);
   }
}


void HighlighterImpl::CalcIndexSearchAttribs(FeatStd::TChar criterion, IndexSearchAttribs& attribs)
{
   if ((criterion >= '0') && (criterion <= '9'))
   {
      attribs.m_isStartIdxPass = true; // a digit in indexMode -> at least a digit for the start index
      if (!attribs.m_isEndIdxPass)
      {
         // calculate start index
         attribs.m_startIdx = (attribs.m_startIdx * 10) + (static_cast<UInt32>(criterion) - static_cast<UInt32>('0'));
      }
      else
      {
         // calculate end index
         attribs.m_endIdx = (attribs.m_endIdx * 10) + (static_cast<UInt32>(criterion) - static_cast<UInt32>('0'));
      }
   }
   else
   {
      if (criterion >= '-')
      {
         // index range -> active end index pass
         attribs.m_isEndIdxPass = true;
      }
   }
   // ignore any other character
}


void HighlighterImpl::MarkText(const FeatStd::String& searchText, FeatStd::UInt32 maxOccurencies, bool caseSensitive, ElementStyle::SharedPointer highlightStyle)
{
   Engine* engine = GetEngine();
   if (0 != engine)
   {
      m_highlightStyle = highlightStyle;

      SearchTextList searchTextList;

      if (!searchText.IsEmpty())
      {
         SearchText searchNode(searchText, maxOccurencies, caseSensitive);
         (void)searchTextList.Add(searchNode);
      }

      // Completed criteria string parsing, now search the document.
      DoSearch(0, &searchTextList);
   }
}


class HighlighterClearAllTraverser : public DocElementTraverser
{
   protected:
      virtual bool Process(const DocElement& docElement, const Candera::Rectangle& /*effectiveRect*/) override
      {
         DocElement& docElem = const_cast<DocElement&>(docElement);
         docElem.SetSubstituteElement(DocElement::SharedPointer());

         Text* text = const_cast<Text*>(Candera::Dynamic_Cast<const Text*>(&docElement));
         if (text != 0)
         {
            text->SetHighlightData(Text::HighlightData::SharedPointer());
         }
         return false;
      }
};


void HighlighterImpl::ClearAll()
{
   Engine* engine = GetEngine();
   if (engine != 0)
   {
      DocAccessor::SharedPointer docAccessor = engine->GetDocAccessor();
      if (docAccessor != 0)
      {
         Document::SharedPointer doc = docAccessor->GetDocument();
         if (doc != 0)
         {
            HighlighterClearAllTraverser traverser;
            (void)doc->Process(traverser, Candera::Rectangle());
         }
      }
   }
}


void HighlighterImpl::Update()
{
   ClearAll();

   switch (m_data.m_mode)
   {
      case Mask:
         MarkText(m_data.m_text, (m_data.m_allOccurrences ? 0 : 1), m_data.m_caseSensitive, m_data.m_style);
         break;

      case Range:
         MarkRange(m_data.m_text, m_data.m_style);
         break;

      default:
         FEATSTD_DEBUG_FAIL();
         break;
   }

   Engine* engine = GetEngine();
   if (engine != 0)
   {
      RichTextRenderer::SharedPointer renderer = engine->GetRenderer();
      if (renderer != 0)
      {
         renderer->Invalidate();
      }
   }
}


void HighlighterImpl::OnMatch(DocRange::SharedPointer match) const
{
   if (match != 0)
   {
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Highlighter matched: elems: %3d, idx from %3d to %3d", match->GetElements().Size(), match->GetStart(),
                          match->GetEnd()));
      const DocRange::DocElemList& list = match->GetElements();
      for (DocRange::DocElemList::ConstIterator it(list.ConstBegin()); it != list.ConstEnd(); ++it)
      {
         Text* text = const_cast<Text*>(Candera::Dynamic_Cast<const Text*>(&(*(*it))));
         // no need to highlight text (mainly spaces) that is marked to be skipped (e.g. not rendered because they are at the begin of the line)
         if ((text != 0) && (!text->GetSkip()))
         {
            Text::HighlightData::SharedPointer highlightData = text->GetHighlightData();
            if (highlightData == 0)
            {
               highlightData = Text::HighlightData::SharedPointer(FEATSTD_NEW(Text::HighlightData));
               text->SetHighlightData(highlightData);
            }
            if (highlightData != 0)
            {
               FeatStd::UInt32 length = text->GetPartCount();
               FeatStd::UInt32 start = (it == list.ConstBegin() ? match->GetStart() : 0);
               FeatStd::UInt32 end = ((it + 1) == list.ConstEnd() ? match->GetEnd() : length - 1);
               if (end > length - 1)
               {
                  end = length - 1;
               }
               ApplyHighlighting(highlightData, start, end, text);
            }
         }
      }
   }
   else
   {
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Highlighter matched: MATCH IS NULL\n"));
   }
}


void HighlighterImpl::ApplyHighlighting(Text::HighlightData::SharedPointer highlightData, UInt32 start, UInt32 end, const Text* text) const
{
   Candera::Color color;
   if (m_highlightStyle->GetParameter(StyleParameterBase::TextColor, color))
   {
      highlightData->m_textColor = color;
      (void)highlightData->m_rangeList.Add(Text::HighlightRange(start, end));
   }

   if ((text != 0) && (m_highlightStyle->GetParameter(StyleParameterBase::BackgroundColor, color)))
   {
      const Candera::Rectangle& rect = text->GetRect();
      Candera::Rectangle highlightRect = text->GetPartBoundaries(UInt16(start));
      highlightRect.SetTop(0);
      highlightRect.SetHeight(rect.GetHeight());
      if (end != start)
      {
         Candera::Rectangle endRect = text->GetPartBoundaries(UInt16(end));
         endRect.SetTop(0);
         endRect.SetHeight(rect.GetHeight());
         highlightRect.Union(endRect);
      }
      (void)highlightData->m_boxList.Add(highlightRect);
      highlightData->m_backgroundColor = color;
   }
}


void HighlighterImpl::FinalizeSearchIndex(SearchIndexList& searchList, IndexSearchAttribs& attribs) const
{
   if (attribs.m_isStartIdxPass)
   {
      SearchIndex searchIndex(attribs.m_startIdx, (attribs.m_isEndIdxPass ? Candera::Math::Maximum(attribs.m_startIdx, attribs.m_endIdx) : attribs.m_startIdx));
      FeatStd::SizeType insertIdx = searchList.Size();
      // insertion sort, ordering search indexes by startIndex ascending
      while ((insertIdx > 0) && (searchList[insertIdx - 1].m_startIdx > attribs.m_startIdx))
      {
         insertIdx--;
      }

      bool combined = false;
      if (insertIdx > 0)
      {
         if ((searchList[insertIdx - 1].IsRange() && ((searchList[insertIdx - 1].m_endIdx + 1) >= attribs.m_startIdx))
               || ((!searchList[insertIdx - 1].IsRange()) && ((searchList[insertIdx - 1].m_startIdx + 1) == attribs.m_startIdx)))
         {
            // the first search index in the list having a smaller start index (the index before the current one)
            // has an overlapping or adjacent end index
            // -> combine search indexes
            searchList[insertIdx - 1].m_endIdx =
               Candera::Math::Maximum(
                  (searchIndex.IsRange() ? attribs.m_endIdx : attribs.m_startIdx),
                  searchList[insertIdx - 1].m_endIdx);
            combined = true;
         }
         else
         {
            if ((searchIndex.IsRange() && ((attribs.m_endIdx + 1) >= searchList[insertIdx].m_startIdx))
                  || ((!searchIndex.IsRange()) && ((attribs.m_startIdx + 1) == searchList[insertIdx].m_startIdx)))
            {
               // the current search index has an overlapping or adjacent start or end index
               // -> combine search indexes
               searchList[insertIdx].m_startIdx =
                  Candera::Math::Minimum(attribs.m_startIdx, searchList[insertIdx].m_startIdx);
               searchList[insertIdx].m_endIdx =
                  Candera::Math::Maximum(
                     (searchIndex.IsRange() ? attribs.m_endIdx : attribs.m_startIdx),
                     searchList[insertIdx].m_endIdx);
               combined = true;
            }
         }
      }
      else
      {
         if ((searchList.Size() > 0) &&
               // insert index is zero
               ((searchIndex.IsRange() && ((searchIndex.m_endIdx + 1) >= searchList[insertIdx].m_startIdx))
                || ((!searchIndex.IsRange()) && ((searchIndex.m_startIdx + 1) == searchList[insertIdx].m_startIdx))))
         {
            // the first search index in the list having a small start index has an overlapping or adjacent end index
            // -> combine search indexes
            searchList[insertIdx].m_startIdx = searchIndex.m_startIdx;
            if (searchIndex.IsRange() || searchList[insertIdx].IsRange())
            {
               searchList[insertIdx].m_endIdx = Candera::Math::Maximum(searchIndex.m_endIdx, searchList[insertIdx].m_endIdx);
            }
            combined = true;
         }
      }

      if (!combined)
      {
         if (insertIdx < searchList.Size())
         {
            (void)searchList.Insert(insertIdx, searchIndex);
         }
         else
         {
            (void)searchList.Add(searchIndex);
         }
      }
   }
   // reset values
   attribs.m_startIdx = 0;
   attribs.m_endIdx = 0;
   attribs.m_isStartIdxPass = false;
   attribs.m_isEndIdxPass = false;
}


void HighlighterImpl::FinalizeSearchIndex(SearchTextList& searchList,
      FeatStd::TChar* textMask, FeatStd::SizeType& textMaskLength) const
{
   if (textMaskLength > 0)
   {
      textMask[textMaskLength] = '\0';
      FeatStd::String searchText(textMask, textMaskLength);
      SearchText searchNode(searchText, 0, false);
      (void)searchList.Add(searchNode);
   }

   // reset values
   textMaskLength = 0;
}


void HighlighterImpl::DoSearch(const SearchIndexList* searchIndexList, SearchTextList* searchTextList)
{
   Engine* engine = GetEngine();
   if (0 != engine)
   {
      DocAccessor::SharedPointer docAccessor = engine->GetDocAccessor();
      if (docAccessor != 0)
      {
         Document::SharedPointer doc = docAccessor->GetDocument();
         if (doc != 0)
         {
            Candera::Rectangle effectiveRect;
            if ((0 != searchIndexList) && (searchIndexList->Size() > 0))
            {
               IndexSearcher searcher((*searchIndexList), (*this));
               (void)doc->Process(searcher, effectiveRect);
            }
            if ((0 != searchTextList) && (searchTextList->Size() > 0))
            {
               TextSearcher searcher((*searchTextList), (*this));
               (void)doc->Process(searcher, effectiveRect);
            }
         }
      }
   }
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
