/* ***************************************************************************************
* FILE:          RtDocAccessorImpl.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtDocAccessorImpl is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once

#include <Candera/System/Mathematics/Vector2.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>

#include <Widgets/2D/RichText/Engine/RtMarkups.h>
#include <Widgets/2D/RichText/Engine/RtDocAccessor.h>
#include <Widgets/2D/RichText/DocumentModel/RtDocElement.h>
#include <Widgets/2D/RichText/DocumentModel/RtDocument.h>
#include <Widgets/2D/RichText/DocumentModel/RtChapter.h>
#include <Widgets/2D/RichText/DocumentModel/RtParagraph.h>

namespace hmibase {
namespace widget {
namespace richtext {

class Engine;

/**
 * Implementation of the DocAccessor interface.
 * This class is not intended being used by clients directly.
 */
class DocAccessorImpl: public DocAccessor
{
      FEATSTD_TYPEDEF_BASE(DocAccessor);
   public:
      FEATSTD_RTTI_DECLARATION();
      FEATSTD_TYPEDEF_SHARED_POINTER(DocAccessorImpl);

      static DocAccessorImpl::SharedPointer Create(Engine* engine);
      virtual ~DocAccessorImpl();

      void SetDocument(Markup::Enum markup, Document::SharedPointer document);
      virtual Document::SharedPointer GetDocument() override;
      virtual Chapter::SharedPointer GetOutline() override;
      virtual Chapter::SharedPointer GetChapter(DocElement::SharedPointer element) override;
      virtual Paragraph::SharedPointer GetParagraph(DocElement::SharedPointer element) override;

      virtual DocElement::SharedPointer GetElementAt(FeatStd::Int32 posX, FeatStd::Int32 posY, bool screenCoordinates) override;
      virtual DocElement::SharedPointer GetElementAt(const Candera::Vector2& position, bool screenCoordinates) override;

   protected:
      DocAccessorImpl();

   private:
      class ElementLocator : public DocElementTraverser
      {
            FEATSTD_TYPEDEF_BASE(DocElementTraverser);
         public:
            ElementLocator(const Candera::Vector2& targetPoint);

            DocElement::SharedPointer GetHitElement() const;

         protected:
            virtual bool Process(const DocElement& docElement, const Candera::Rectangle& effectiveRect) override;

         private:
            // The point in document space for which the element shall be located
            Candera::Vector2 m_targetPoint;
            // The bounding rect of the hit element.
            Candera::Rectangle m_hitRect;
            // The element with the smallest bounding rectangle that intersects with the target point.
            DocElement::SharedPointer m_hitElement;

            FEATSTD_MAKE_CLASS_UNCOPYABLE(ElementLocator);
      };

      // Gets the first element of T type in the object hierarchy of the given element
      // The element is returned if it matches the type already.
      // If no element of the requested type is found in the hierarchy, an empty
      // reference is returned.
      template<typename T>
      typename T::SharedPointer FindInHierarchy(DocElement::SharedPointer element)
      {
         T* target = 0;
         if (element != 0)
         {
            if (element->IsTypeOf(T::GetTypeId()))
            {
               target = Candera::Dynamic_Cast<T*>(&(*element));
            }
            else
            {
               DocElement* parent = element->GetParent();
               while ((parent != 0) && (target == 0))
               {
                  if (parent->IsTypeOf(T::GetTypeId()))
                  {
                     target = Candera::Dynamic_Cast<T*>(&(*element));
                  }
                  parent = parent->GetParent();
               }
            }
         }
         return typename T::SharedPointer(target);
      };

      // The markup language of the original source document.
      Markup::Enum m_srcDocMarkup;
      // The document model maintained by this engine instance.
      Document::SharedPointer m_document;

      FEATSTD_MAKE_CLASS_UNCOPYABLE(DocAccessorImpl);
};


inline
DocElement::SharedPointer DocAccessorImpl::ElementLocator::GetHitElement() const
{
   return m_hitElement;
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
