/* ***************************************************************************************
* FILE:          RtText.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtText is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "RtText.h"
#include <Candera/TextEngine/Font.h>
#include <hmibase/util/Macros.h>
#if defined(USE_MINIMAL_TEXTRENDERER)
#include <Candera/TextEngine/MaximalPreprocessingContext.h>
#else
#include <Candera/TextEngine/AlignTextMeasureContext.h>
#endif

namespace hmibase {
namespace widget {
namespace richtext {

FEATSTD_RTTI_DEFINITION(Text, Base);

Text::Text(const FeatStd::String& text /*= std::string()*/) :
   Base(),
   m_text(text)
{
}


Text::~Text()
{
}


FeatStd::UInt16 Text::GetPartCount() const
{
   FeatStd::UInt16 count = 0;
   if (m_textContext != 0)
   {
      const TruncationTextContext& textContext = static_cast<const TruncationTextContext&>(*m_textContext);
      for (TruncationTextContext::TruncateGlyphDataIterator it = textContext.GetTruncateGlyphDataIterator(); it.IsValid(); ++it)
      {
         count++;
      }
   }

   return count;
}


Candera::Vector2 Text::GetPartPosition(FeatStd::UInt16 partIdx) const
{
   Candera::Vector2 partPosition(GetRect().GetWidth(), 0.0F); // in case of index clipping

   if (m_textContext != 0)
   {
      FeatStd::UInt16 count = 0;
      const TruncationTextContext& textContext = static_cast<const TruncationTextContext&>(*m_textContext);
      for (TruncationTextContext::TruncateGlyphDataIterator it = textContext.GetTruncateGlyphDataIterator(); it.IsValid(); ++it)
      {
         if (count++ == partIdx)
         {
            Candera::TextRendering::PixelPosition2D pos = (*it).GetPosition();
            partPosition.SetX(FeatStd::Float(pos.x));
            partPosition.SetY(FeatStd::Float(pos.y));
            break;
         }
      }
   }

   return partPosition;
}


Candera::Rectangle Text::GetPartBoundaries(FeatStd::UInt16 partIdx) const
{
   Candera::Rectangle partRect(GetRect().GetWidth(), 0.0F, 0.0F, 0.0F); // in case of index clipping

   if (m_textContext != 0)
   {
      const TruncationTextContext& textContext = static_cast<const TruncationTextContext&>(*m_textContext);
      Candera::TextRendering::TextPosition characterPos = textContext.GetCharacterPosition(partIdx);

      for (TruncationTextContext::TruncateGlyphDataIterator it = textContext.GetTruncateGlyphDataIterator(); it.IsValid(); ++it)
      {
         if (characterPos == (*it).GetCharacterPosition())
         {
            Candera::TextRendering::TextRect textRect = (*it).GetRect();
            Candera::TextRendering::PixelPosition2D pos = (*it).GetPosition();
            partRect.SetLeft(FeatStd::Float(pos.x - textRect.GetLeft()));
            partRect.SetTop(FeatStd::Float(pos.y - textRect.GetTop()));
            partRect.SetWidth(FeatStd::Float((*it).GetAdvance().width));
            partRect.SetHeight(FeatStd::Float((*it).GetAdvance().height));
            break;
         }
      }
   }

   return partRect;
}


#if defined (RICHTEXT_LOGGING_ENABLED)
void Text::Log(std::ostream& stream, const Candera::Rectangle& effectiveRect, bool repeatLastLine /* = false */,
               StreamIterationData& data /* = StreamIterationData::default */) const
{
   data.name = "Text";
   Base::Log(stream, effectiveRect, repeatLastLine, data);

   RICHTEXT_LOG(stream, " '" << m_text.GetCString() << "'");
   Candera::TextRendering::SharedStyle::SharedPointer style;
   if (GetStyleParameter(StyleParameterBase::FontStyle, style))
   {
      RICHTEXT_LOG(stream, " size:" << style->GetDefaultFont().GetHeight());
   }
   RICHTEXT_LOG(stream, std::endl);
}


#endif


bool Text::HighlightData::IsInRange(FeatStd::UInt32 index) const
{
   bool isInRange = false;

   Text::HighlightRange::List::ConstIterator it(m_rangeList.ConstBegin());
   while ((!isInRange) && (it != m_rangeList.ConstEnd()))
   {
      const Text::HighlightRange& range = (*it);
      isInRange = (index >= range.m_start) && (index <= range.m_end);
      ++it;
   }

   return isInRange;
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
