/* ***************************************************************************************
* FILE:          RtDocHelper.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtDocHelper is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "RtDocHelper.h"
#include <Candera/TextEngine/Font.h>
#include <Candera/TextEngine/TextRenderContext.h>
#include <FeatStd/Platform/PerfCounter.h>
#include <FeatStd/Util/BinarySearch.h>
#include <Widgets/2D/RichText/DocumentModel/RtDocElement.h>
#include <vector>

namespace hmibase {
namespace widget {
namespace richtext {

using namespace Candera::TextRendering;


CustomStyleCache::Key::Key(const FeatStd::Char* faceName, FeatStd::Int32 height) :
   m_faceName(faceName),
   m_height(height)
{
}


FeatStd::Int CustomStyleCache::Key::Compare(const Key& left, const CustomStyleCache::SharedPointer& right)
{
#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
   FeatStd::Int32 strResult = left.m_faceName.Compare(right->m_key.m_faceName);
#else
   FeatStd::Int32 strResult = ::strcmp(left.m_faceName.GetCString(), right->m_key.m_faceName.GetCString());
#endif
   return
      ((strResult < 0) || ((strResult == 0) && (left.m_height < right->m_key.m_height))) ? -1
      : ((strResult == 0) && (left.m_height == right->m_key.m_height)) ? 0
      : +1;
}


CustomStyleCache* CustomStyleCache::Get(const FeatStd::Char* faceName, FeatStd::Int32 height)
{
   FEATSTD_SYNCED_STATIC_OBJECT(Cache, s_styleCache);

   CustomStyleCache* cacheEntry = 0;

   Key key(faceName, height);
   const FeatStd::SizeType itemCount = s_styleCache.Size();
   FeatStd::SizeType insertAt;
   FeatStd::SizeType foundAt = FeatStd::Internal::BinarySearch<CustomStyleCache::Key>(key, s_styleCache, itemCount, &insertAt);
   if (foundAt < itemCount)
   {
      cacheEntry = s_styleCache[foundAt].GetPointerToSharedInstance();
   }
   else
   {
      cacheEntry = FEATSTD_NEW(CustomStyleCache)();
      if (cacheEntry != 0 && !cacheEntry->Init(faceName, height))
      {
         FEATSTD_SAFE_DELETE(cacheEntry);
      }
      else
      {
         s_styleCache.Insert(insertAt, SharedPointer(cacheEntry));
      }
   }

   return cacheEntry;
}


CustomStyleCache::CustomStyleCache() :
   m_key("", 0),
   m_style()
{
}


bool CustomStyleCache::Init(const FeatStd::Char* faceName, FeatStd::Int32 height)
{
   bool success = false;

   m_key = Key(faceName, height);
   Font font;
   if (font.Setup(faceName, FeatStd::Int16(height)))
   {
      m_style = SharedStyle::Create();
      if (m_style != 0)
      {
         m_style->SetDefaultFont(font);
         success = true;
      }
   }

   return success;
}


#if defined(RICHTEXT_LOGGING_ENABLED)
void RemoveLastLine(std::ostream& stream)
{
   std::ostringstream* oss = dynamic_cast<std::ostringstream*>(&stream);
   if (oss != 0)
   {
      std::string s((*oss).str());
      std::stringstream ss(s);

      std::vector<std::string> lines;
      std::string line;
      while (std::getline(ss, line))
      {
         lines.push_back(line);
      }

      // reset stream to begin
      stream.seekp(0);

      // fill stream again with all separated lines except the last one - ToDo: find more efficient way
      for (size_t i = 0; i < lines.size() - 1; i++)
      {
         stream << lines[i] << std::endl;
      }
   }
}


StreamTraverser::StreamTraverser(std::ostream& stream, StreamIterationData& data /* = StreamIterationData::defaultData */) :
   m_stream(stream),
   m_data(data)
{
}


bool StreamTraverser::Process(const class DocElement& object, const Candera::Rectangle& effectiveRect)
{
   if (m_data.deep)
   {
      m_data.indent = GetLevel() * 2;
   }
   object.Log(m_stream, effectiveRect, false, m_data);
   return false;
}


StreamIterationData StreamIterationData::defaultData;
#endif
} // namespace richtext
} // namespace widget
} // namespace hmibase
