/* ***************************************************************************************
* FILE:          RtDocElement.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtDocElement is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once

#include <Candera/System/Mathematics/Rectangle.h>
#include <Widgets/2D/RichText/DocumentModel/RtDocHelper.h>
#include <Widgets/2D/RichText/StyleProvider/RtElementStyleAccessor.h>

namespace hmibase {
namespace widget {
namespace richtext {

class DocElementTraverser;
class Document;


/** The base class for document elements. */
class DocElement
{
   public:
      FEATSTD_RTTI_DECLARATION();
      FEATSTD_TYPEDEF_SHARED_POINTER(DocElement);

      typedef FeatStd::Internal::Vector<Candera::Rectangle> BoxList;

      DocElement();
      virtual ~DocElement();

      void SetSkip(bool skip);
      bool GetSkip() const;

      void SetParent(DocElement* parent);
      DocElement* GetParent() const;

      void SetRect(const Candera::Rectangle& rect);
      const Candera::Rectangle& GetRect() const;

      template <typename T>
      bool AddStyleParameter(StyleParameterBase::Name name, const T& value);

      template <typename T>
      bool GetStyleParameter(StyleParameterBase::Name name, T& value) const;

      template <typename ParentType, typename ChildType>
      ChildType* AddChildType();

      virtual bool Process(DocElementTraverser& traverser, const Candera::Rectangle& effectiveRect) const;

      template <typename ChildType>
      bool ProcessChildList(const FeatStd::Internal::Vector<typename ChildType::SharedPointer>& list, DocElementTraverser& traverser,
                            const Candera::Rectangle& effectiveRect) const;

      void AddBackgroundBox(const Candera::Rectangle& box);
      const BoxList& GetBackgroundBoxList() const;
      void SetSubstituteElement(const DocElement::SharedPointer& substitute);
      const DocElement::SharedPointer& GetSubstituteElement() const;

#if defined (RICHTEXT_LOGGING_ENABLED)
      Document* GetDocument();
      virtual void Log(std::ostream& stream, const Candera::Rectangle& effectiveRect, bool repeatLastLine = false,
                       StreamIterationData& data = StreamIterationData::defaultData) const;
#endif

   protected:

   private:
#if defined (RICHTEXT_LOGGING_ENABLED)
      friend std::ostream& operator <<(std::ostream& out, const DocElement& obj);
#endif

      bool m_skip;
      DocElement* m_parent;
      DocElement::SharedPointer m_substitute;  // used for rendering a document with manipulated structure without destroying the original structure;
      Candera::Rectangle m_rect;
      ElementStyle::SharedPointer m_style;
      BoxList m_backgroundBoxList;

      FEATSTD_SHARED_POINTER_DECLARATION();

      StyleParameterBase::SharedPointer GetStyleParameter(StyleParameterBase::Name name) const;
};


inline
void DocElement::SetSkip(bool skip)
{
   m_skip = skip;
}


inline
bool DocElement::GetSkip() const
{
   return m_skip;
}


inline void DocElement::SetParent(DocElement* parent)
{
   m_parent = parent;
}


inline hmibase::widget::richtext::DocElement* DocElement::GetParent() const
{
   return m_parent;
}


inline const Candera::Rectangle& DocElement::GetRect() const
{
   return m_rect;
}


template <typename T>
bool DocElement::GetStyleParameter(StyleParameterBase::Name name, T& value) const
{
   bool found = false;

   StyleParameterBase::SharedPointer param = GetStyleParameter(name);
   if (param != 0)
   {
      const TypedStyleParameter<T>* p = dynamic_cast<const TypedStyleParameter<T>*>(param.GetPointerToSharedInstance());
      FEATSTD_DEBUG_ASSERT(p != 0);
      if (p != 0)
      {
         value = p->GetValue();
         found = true;
      }
   }

   return found;
}


template <typename T>
bool DocElement::AddStyleParameter(StyleParameterBase::Name name, const T& value)
{
   if (m_style == 0)
   {
      m_style = ElementStyleAccessor::GetInstance().CreateStyle("");
   }

   TypedStyleParameter<T>* param = FEATSTD_NEW(TypedStyleParameter<T>)(name, value);
   return m_style->AddParameter(StyleParameterBase::SharedPointer(param));
}


template <typename ParentType, typename ChildType>
ChildType* DocElement::AddChildType()
{
   ChildType* child = 0;
   ParentType* parent = dynamic_cast<ParentType*>(this);
   if (parent != 0)
   {
      child = FEATSTD_NEW(ChildType)();
      parent->AddChild(typename ChildType::SharedPointer(child));
   }
   return child;
}


template <typename ChildType>
bool DocElement::ProcessChildList(const FeatStd::Internal::Vector<typename ChildType::SharedPointer>& list, DocElementTraverser& traverser,
                                  const Candera::Rectangle& effectiveRect) const
{
   bool abort = false;
   traverser.IncLevel();
   for (typename FeatStd::Internal::Vector<typename ChildType::SharedPointer>::ConstIterator it(list.ConstBegin()); !abort
         && (it != list.ConstEnd()); ++it)
   {
      const typename ChildType::SharedPointer& child = (*it);
      abort = (*child).Process(traverser, traverser.GetEffeciveRect(*child, effectiveRect));
   }
   traverser.DecLevel();
   return abort;
}


inline
void DocElement::AddBackgroundBox(const Candera::Rectangle& box)
{
   (void)m_backgroundBoxList.Add(box);
}


inline
const DocElement::BoxList& DocElement::GetBackgroundBoxList() const
{
   return m_backgroundBoxList;
}


inline
void DocElement::SetSubstituteElement(const DocElement::SharedPointer& substitute)
{
   m_substitute = substitute;
   if (m_substitute != 0)
   {
      FEATSTD_DEBUG_ASSERT(m_substitute->GetParent() == 0);
      m_substitute->SetParent(m_parent);
   }
}


inline
const DocElement::SharedPointer& DocElement::GetSubstituteElement() const
{
   return m_substitute;
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
