/* ***************************************************************************************
* FILE:          RtDocElement.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtDocElement is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "RtDocElement.h"

#if defined(RICHTEXT_LOGGING_ENABLED)
#include "RtDocument.h"
#include <sstream>
#include <iomanip>
#endif

namespace hmibase {
namespace widget {
namespace richtext {


#if defined (RICHTEXT_LOGGING_ENABLED)
std::ostream& operator <<(std::ostream& stream, const DocElement& obj)
{
   StreamIterationData data;
   data.deep = true;

   StreamTraverser traverser(stream, data);

   RICHTEXT_LOG(stream, std::endl);

   obj.Process(traverser, Candera::Rectangle());
   return stream;
}


#endif

FEATSTD_RTTI_BASECLASS_DEFINITION(DocElement);

DocElement::DocElement() :
   m_skip(false),
   m_parent(0),
   m_rect(0.0F, 0.0F, -1.0F, -1.0F)
{
}


DocElement::~DocElement()
{
}


void DocElement::SetRect(const Candera::Rectangle& rect)
{
   m_rect = rect;
   RICHTEXT_LOG_DOCELEMENT(this, true);
}


bool DocElement::Process(DocElementTraverser& traverser, const Candera::Rectangle& effectiveRect) const
{
   return traverser.Process(*this, effectiveRect);
}


StyleParameterBase::SharedPointer DocElement::GetStyleParameter(StyleParameterBase::Name name) const
{
   StyleParameterBase::SharedPointer param;
   if (m_style != 0)
   {
      param = m_style->GetParameter(name);
   }

   if (param == 0)
   {
      DocElement* parent = GetParent();
      if (parent != 0)
      {
         param = parent->GetStyleParameter(name);
      }
   }

   return param;
}


#if defined (RICHTEXT_LOGGING_ENABLED)

#define LOG_RECT(rect) \
                "(" << rect.GetLeft() << ", " << rect.GetTop() << ", " << rect.GetWidth() << ", " << rect.GetHeight() << ")";

hmibase::widget::richtext::Document* DocElement::GetDocument()
{
   Document* doc = 0;
   DocElement* parent = this;
   while (parent != 0)
   {
      doc = Candera::Dynamic_Cast<Document*>(parent);
      parent = parent->GetParent();
   }
   return doc;
}


void DocElement::Log(std::ostream& stream, const Candera::Rectangle& effectiveRect, bool repeatLastLine /* = false */,
                     StreamIterationData& data /* = StreamIterationData::default */) const
{
   if (repeatLastLine)
   {
      RemoveLastLine(stream);
   }

   if (!data.deep)
   {
      data.indent = 0;
      DocElement* parent = m_parent;
      while (parent != 0)
      {
         data.indent += 2;
         parent = parent->m_parent;
      }
   }

   std::ostringstream position;
   position << " @" << LOG_RECT(m_rect);
   if (data.deep)
   {
      position << " -> " LOG_RECT(effectiveRect);
   }

   RICHTEXT_LOG(stream, std::setw(data.indent) << "" << data.name << " 0x" << m_parent << " ->" << " 0x" << this << position.str() << ":");
}


#endif
} // namespace richtext
} // namespace widget
} // namespace hmibase
