/* ***************************************************************************************
* FILE:          RtChapter.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtChapter is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "RtChapter.h"
#include "RtDocument.h"

namespace hmibase {
namespace widget {
namespace richtext {

FEATSTD_RTTI_DEFINITION(Chapter, Base);

Chapter::Chapter() :
   Base()
{
}


void Chapter::AddTitle(const Paragraph::SharedPointer& title)
{
   FEATSTD_DEBUG_ASSERT(title->GetParent() == 0);
   m_title = title;
   title->SetParent(this);
   RICHTEXT_LOG_DOCELEMENT(title);
}


void Chapter::AddChild(const Paragraph::SharedPointer& paragraph)
{
   FEATSTD_DEBUG_ASSERT(paragraph->GetParent() == 0);
   (void)m_paragraphs.Add(paragraph);
   paragraph->SetParent(this);
   RICHTEXT_LOG_DOCELEMENT(paragraph);
}


void Chapter::AddChild(const Chapter::SharedPointer& chapter)
{
   FEATSTD_DEBUG_ASSERT(chapter->GetParent() == 0);
   (void)m_chapters.Add(chapter);
   chapter->SetParent(this);
   RICHTEXT_LOG_DOCELEMENT(chapter);
}


FeatStd::Int Chapter::GetNestedLevel() const
{
   FeatStd::Int nestedLevel = -1;
   Chapter* parent = dynamic_cast<Chapter*>(GetParent());
   while (parent != 0)
   {
      nestedLevel++;
      parent = dynamic_cast<Chapter*>(parent->GetParent());
   }

   return nestedLevel;
}


ParagraphItem::SharedPointer Chapter::GetFirstLeafParagraphItem()
{
   ParagraphItem::SharedPointer leafItem;
   if (m_title != 0)
   {
      leafItem = m_title->GetFirstLeafParagraphItem();
   }
   else
   {
      if (!m_paragraphs.Empty())
      {
         leafItem = ((*m_paragraphs.Begin())->GetFirstLeafParagraphItem());
      }
   }

   return leafItem;
}


ParagraphItem::SharedPointer Chapter::GetLastLeafParagraphItem()
{
   ParagraphItem::SharedPointer leafItem;
   if (!m_paragraphs.Empty())
   {
      leafItem = ((*m_paragraphs.End())->GetLastLeafParagraphItem());
   }
   else
   {
      if (m_title != 0)
      {
         leafItem = m_title->GetLastLeafParagraphItem();
      }
   }

   return leafItem;
}


bool Chapter::Process(DocElementTraverser& traverser, const Candera::Rectangle& effectiveRect) const
{
   bool terminate = Base::Process(traverser, effectiveRect);
   if (m_title != 0)
   {
      traverser.IncLevel();
      terminate = (!terminate) && (*m_title).Process(traverser, traverser.GetEffeciveRect(*m_title, effectiveRect));
      traverser.DecLevel();
   }

   terminate = (!terminate) && ProcessChildList<Paragraph>(m_paragraphs, traverser, effectiveRect);
   terminate = (!terminate) && ProcessChildList<Chapter>(m_chapters, traverser, effectiveRect);
   return terminate;
}


#if defined (RICHTEXT_LOGGING_ENABLED)
void Chapter::StreamChapterHierarchy(std::ostream& stream) const
{
   const Chapter* parent = dynamic_cast<const Chapter*>(GetParent());
   if (parent != 0)
   {
      const ChapterList& siblings = parent->m_chapters;

      int level = 1;
      if (!siblings.Empty())
      {
         for (ChapterList::ConstIterator it(siblings.ConstBegin()); it != siblings.ConstEnd(); ++it)
         {
            const Chapter::SharedPointer& ref = *it;
            const Chapter* chapter = &(*ref);
            if (chapter == this)
            {
               break;
            }
            level++;
         }

         parent->StreamChapterHierarchy(stream);
      }

      RICHTEXT_LOG(stream, level << ".");
   }
}


void Chapter::Log(std::ostream& stream, const Candera::Rectangle& effectiveRect, bool repeatLastLine /* = false */,
                  StreamIterationData& data /* = StreamIterationData::default */) const
{
   data.name = "Chapter";
   Base::Log(stream, effectiveRect, repeatLastLine, data);

   RICHTEXT_LOG(stream, " ");
   StreamChapterHierarchy(stream);
   RICHTEXT_LOG(stream, std::endl);
}


#endif
} // namespace richtext
} // namespace widget
} // namespace hmibase
