/* ***************************************************************************************
* FILE:          RtBlock.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtBlock is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once

#include <Widgets/2D/RichText/DocumentModel/RtParagraphItem.h>

namespace hmibase {
namespace widget {
namespace richtext {

/** A paragraph item used as container for multiple other paragraph items that form a logical unit within a paragraph. */
class Block : public ParagraphItem
{
      FEATSTD_TYPEDEF_BASE(ParagraphItem);

   public:
      FEATSTD_RTTI_DECLARATION();
      FEATSTD_TYPEDEF_SHARED_POINTER(Block);

      Block();

      virtual void AddChild(const ParagraphItem::SharedPointer& item);

      /** Provides the number of atomic parts - letters or images - of this block item.
          @return The number of letters or images of this block including all children. */
      virtual FeatStd::UInt16 GetPartCount() const override;

      /** Retrieves the position the 'partIdx'th part of this block.
          @param partIdx  The index of the part of which the position is retrieved. The index is clipped to 'GetPartCount'.
          @return         The position of the left upper corner of the addressed atomic part. */
      virtual Candera::Vector2 GetPartPosition(FeatStd::UInt16 partIdx) const override;

      virtual ParagraphItem::SharedPointer GetFirstLeafParagraphItem() override;
      virtual ParagraphItem::SharedPointer GetLastLeafParagraphItem() override;

      virtual bool Process(DocElementTraverser& traverser, const Candera::Rectangle& effectiveRect) const override;
      virtual FeatStd::UInt16 GetItemCount() const override;

#if defined (RICHTEXT_LOGGING_ENABLED)
      virtual void Log(std::ostream& stream, const Candera::Rectangle& effectiveRect, bool repeatLastLine = false,
                       StreamIterationData& data = StreamIterationData::defaultData) const override;
#endif
   private:
      class PartCounter : public DocElementTraverser
      {
            FEATSTD_TYPEDEF_BASE(DocElementTraverser);
         public:
            PartCounter();
            FeatStd::UInt16 GetPartCount() const;

         protected:
            virtual bool Process(const DocElement& docElement, const Candera::Rectangle& effectiveRect) override;

         private:
            // The number of parts.
            FeatStd::UInt16 m_partCount;
      };

      class PartMeasurer : public DocElementTraverser
      {
            FEATSTD_TYPEDEF_BASE(DocElementTraverser);
         public:
            PartMeasurer(FeatStd::UInt16 partIdx);
            Candera::Vector2 GetPartPosition() const;

         protected:
            virtual bool Process(const DocElement& docElement, const Candera::Rectangle& effectiveRect) override;

         private:
            // The index of the addressed part.
            FeatStd::UInt16 m_partIdx;
            // The number of processed parts.
            FeatStd::UInt16 m_processedPartCnt;
            // The position of the addressed part.
            Candera::Vector2 m_partPosition;
      };

      typedef FeatStd::Internal::Vector<ParagraphItem::SharedPointer> ItemList;

      ItemList m_items;
};


inline
FeatStd::UInt16 Block::GetItemCount() const
{
   return static_cast<FeatStd::UInt16>(m_items.Size());
}


inline
FeatStd::UInt16 Block::PartCounter::GetPartCount() const
{
   return m_partCount;
}


inline
Candera::Vector2 Block::PartMeasurer::GetPartPosition() const
{
   return m_partPosition;
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
