/* ***************************************************************************************
* FILE:          RtBlock.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtBlock is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "RtBlock.h"
#include "RtText.h"
#include "RtDocument.h"

namespace hmibase {
namespace widget {
namespace richtext {


Block::PartCounter::PartCounter() :
   Base(),
   m_partCount(0)
{
}


bool Block::PartCounter::Process(const DocElement& docElement, const Candera::Rectangle& /* effectiveRect */)
{
   const ParagraphItem* item = Candera::Dynamic_Cast<const ParagraphItem*>(&docElement);
   if (item != 0)
   {
      m_partCount = static_cast<FeatStd::UInt16>(m_partCount + item->GetPartCount());
   }
   return false;
}


Block::PartMeasurer::PartMeasurer(FeatStd::UInt16 partIdx) :
   Base(),
   m_partIdx(partIdx),
   m_processedPartCnt(0),
   m_partPosition()
{
}


bool Block::PartMeasurer::Process(const DocElement& docElement, const Candera::Rectangle& /* effectiveRect */)
{
   bool terminate = false;
   const ParagraphItem* item = Candera::Dynamic_Cast<const ParagraphItem*>(&docElement);
   if (item != 0)
   {
      FeatStd::UInt16 cnt = item->GetPartCount();
      if (m_partIdx < m_processedPartCnt + cnt)
      {
         m_partPosition = item->GetPartPosition(static_cast<FeatStd::UInt16>(m_partIdx - m_processedPartCnt));
         terminate = true;
      }
      else
      {
         m_processedPartCnt = static_cast<FeatStd::UInt16>(m_processedPartCnt + cnt);
      }
   }
   return terminate;
}


FEATSTD_RTTI_DEFINITION(Block, Base);

Block::Block() :
   Base()
{
}


void Block::AddChild(const ParagraphItem::SharedPointer& item)
{
   FEATSTD_DEBUG_ASSERT(item->GetParent() == 0);
   (void)m_items.Add(item);
   item->SetParent(this);
   RICHTEXT_LOG_DOCELEMENT(item);
}


FeatStd::UInt16 Block::GetPartCount() const
{
   PartCounter partCounter;
   Candera::Rectangle rect;
   (void)ProcessChildList<ParagraphItem>(m_items, partCounter, rect);
   return partCounter.GetPartCount();
}


Candera::Vector2 Block::GetPartPosition(FeatStd::UInt16 partIdx) const
{
   Candera::Vector2 partPosition(0.0F, 0.0F);
   PartMeasurer partMeasurer(partIdx);
   Candera::Rectangle rect;
   (void)ProcessChildList<ParagraphItem>(m_items, partMeasurer, rect);
   return partMeasurer.GetPartPosition();
}


ParagraphItem::SharedPointer Block::GetFirstLeafParagraphItem()
{
   ParagraphItem::SharedPointer leafItem;
   if (!m_items.Empty())
   {
      leafItem = ((*m_items.Begin())->GetFirstLeafParagraphItem());
   }
   // does not return this, because a block is never a leaf although its subclasses may be.
   return leafItem;
}


ParagraphItem::SharedPointer Block::GetLastLeafParagraphItem()
{
   ParagraphItem::SharedPointer leafItem;
   if (!m_items.Empty())
   {
      leafItem = ((*m_items.End())->GetLastLeafParagraphItem());
   }
   // does not return this, because a block is never a leaf although its subclasses may be.
   return leafItem;
}


bool Block::Process(DocElementTraverser& traverser, const Candera::Rectangle& effectiveRect) const
{
   bool terminate = Base::Process(traverser, effectiveRect);
   if (!terminate)
   {
      terminate = ProcessChildList<ParagraphItem>(m_items, traverser, effectiveRect);
   }
   return terminate;
}


#if defined (RICHTEXT_LOGGING_ENABLED)
void Block::Log(std::ostream& stream, const Candera::Rectangle& effectiveRect, bool repeatLastLine /* = false */,
                StreamIterationData& data /* = StreamIterationData::default */) const
{
   data.name = "Block";
   Base::Log(stream, effectiveRect, repeatLastLine, data);
   RICHTEXT_LOG(stream, std::endl);
}


#endif
}


} // namespace widget
} // namespace hmibase
