/* ***************************************************************************************
* FILE:          RtUrlResolver.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtUrlResolver is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once

#include <Candera/Engine2D/Core/BitmapImage2D.h>
#include <Widgets/2D/RichText/Utils/RtTextInStream.h>

namespace hmibase {
namespace widget {
namespace richtext {

typedef const FeatStd::TChar* Url;

/** The basic interface for classes resolving URLs. */
class UrlResolver
{
   public:
      /** Encapsulates dedicated URL segments. */
      struct UrlSegments
      {
         /// If not 0, the scheme for the given URL is set.
         FeatStd::String scheme;
         /// If not 0, the authority for the given URL is set.
         FeatStd::String authority;
         /// If not 0, the path for the given URL is set
         FeatStd::String path;
      };
      /** Splits the given URL into its components scheme:authority/path.
          @param  baseUrl          Denominates the absolute part of the complete URL and may be a null-object if the resourceUrl parameter
                                   is already absolute.
          @param  resourceUrl      Denominates the location of the resource relative to the baseUrl or the absolute location.
          @param[out] urlSegements Provides the segements of the resource URL. */
      static void SplitUrl(Url baseUrl, Url resourceUrl, UrlSegments& urlSegments);

      /** Determines whether or not the given URL scheme is supported by this URL resolver.
          The URL scheme is the first segment of the URL, e.g. 'scheme' in URL 'scheme://domain.tld/path/resource#anchor?attrib=value'.

          @return     True is returned, if the URL scheme is supported and URLs of that scheme can be resolved, false otherwise. */
      bool IsUrlSchemeSupported(FeatStd::String scheme);

      /** Provides the bitmap resource in native render-able format addressed by the given URL.
          @param  baseUrl         Denominates the absolute part of the complete URL and may be a null-object if the resourceUrl parameter
                                  is already absolute.
          @param  resourceUrl     Denominates the location of the resource relative to the baseUrl or the absolute location.
          @return                 The shared pointer to the Bitmap object addressed by the URLs if it can be accessed and loaded,
                                  otherwise a null-object is returned. */
      virtual Candera::Bitmap::SharedPointer GetBitmap(Url baseUrl, Url resourceUrl) = 0;

      /** Provides the text input stream to the resource addressed by the given URL.
          @param  baseUrl         Denominates the absolute part of the complete URL and may be a null-object if the resourceUrl parameter
                                  is already absolute.
          @param  resourceUrl     Denominates the location of the resource relative to the baseUrl or the absolute location.
          @return                 The TextInStream object associated with the resource if it can be accessed, otherwise a null-object
                                  is returned. */
      virtual TextInStream::SharedPointer GetTextResource(Url baseUrl, Url resourceUrl) = 0;

      virtual Candera::TextRendering::SharedStyle::SharedPointer GetTextStyle(Url baseUrl, Url resourceUrl) = 0;
};


} // namespace richtext
} // namespace widget
} // namespace hmibase
