/* ***************************************************************************************
* FILE:          RtContentProvider.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtContentProvider.cpp is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once

#include <Candera/Engine2D/Core/BitmapImage2D.h>
#include <Widgets/2D/RichText/ContentProvider/RtUrlResolver.h>

namespace hmibase {
namespace widget {
namespace richtext {

/** Provides the requested content or an according null-object, e.g. an empty Color, a 0-pointer, an invalid shared pointer,
    or similar. This is accomplished by looking up the first URL resolver instance capable resolving the given URL and by delegating
    the resource resolution to that URL resolver. If resource resolution fails, look up of URL resolution support within the remaining
    URL resolvers is continued and the URL resolution is tried again, if another suitable URL resolver is is detected.

    Resolution of resources specified via resource attributes rather than URLs, such as colors or fonts, is performed directly by this
    implementation and is not delegated to UrlResolvers. */
class ContentProvider
{
   public:
      /** @return     The ContentProvider singleton instance. */
      static ContentProvider& GetInstance();

      /** Provides the bitmap resource in native render-able format addressed by the given URL.
          @param baseUrl          Denominates the absolute part of the complete URL and may be a null-object if the resourceUrl parameter
                                  is already absolute.
          @param resourceUrl      Denominates the location of the resource relative to the baseUrl or the absolute location.
          @return                 The shared pointer to the Bitmap object addressed by the URLs if it can be accessed and loaded,
                                  otherwise a null-object is returned. */
      Candera::Bitmap::SharedPointer GetBitmap(Url baseUrl, Url resourceUrl) const;

      /** Provides the text input stream to the resource addressed by the given URL.
          @param  baseUrl         Denominates the absolute part of the complete URL and may be a null-object if the resourceUrl parameter
                                  is already absolute.
          @param  resourceUrl     Denominates the location of the resource relative to the baseUrl or the absolute location.
          @return                 The TextInStream object associated with the resource if it can be accessed, otherwise a null-object
                                  is returned. */
      TextInStream::SharedPointer GetTextResource(Url baseUrl, Url resourceUrl) const;

      Candera::TextRendering::SharedStyle::SharedPointer GetTextStyle(Url baseUrl, Url resourceUrl) const;

   private:
      ContentProvider();
};


} // namespace richtext
} // namespace widget
} // namespace hmibase
