/* ***************************************************************************************
* FILE:          PageEditStrategy.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PageEditStrategy.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"
#include "PageEditStrategy.h"
#include "PageEditController2D.h"
#include "PageEditManager.h"
#include "PageEditWidget2D.h"

#include <Trace/ToString.h>
#include <Widgets/2D/Gizmo/GizmoWidget2D.h>
#include <Widgets/2D/Gizmo/GizmoController2D.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_PAGEEDIT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/PageEditStrategy.cpp.trc.h"
#endif


namespace hmibase {
namespace widget {
namespace pageedit {


/*****************************************************************************/
bool DefaultPageEditStrategy::SlideRows(Session& session, CellGrid& cellGrid, CellIndexType direction, CellIndexType column, CellIndexVector& rowSlideDelta, CellRectMap& newCellRectMap)
{
   Page::SharedPointer crtPage = session._Album.GetPage(Cell(column, 0));

   CellRect crtPageRect;
   if (!crtPage.PointsToNull())
   {
      crtPageRect.TopLeft = session._Album.GetPageOffset(crtPage);
      crtPageRect.Span = crtPage->CellCount;
   }

   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "SlideRows direction=%d column=%d crtPage=%25s crtPageRect=%25s rowDelta=%s",
                       direction,
                       column,
                       HMIBASE_TO_STRING(crtPage.PointsToNull() ? PageIdType() : crtPage->Id),
                       HMIBASE_TO_STRING(crtPageRect),
                       HMIBASE_TO_STRING(rowSlideDelta)));

   if (crtPage.PointsToNull() || (direction == 0) || (static_cast<CellIndexType>(rowSlideDelta.size()) != cellGrid.GetRowCount()) || (column < 0) || (cellGrid.GetColumnCount() <= column))
   {
      return false;
   }

   for (size_t rowIndex = 0; rowIndex < rowSlideDelta.size(); ++rowIndex)
   {
      Cell cell(column, static_cast<CellIndexType>(rowIndex));
      if (cellGrid.ContainsCell(cell))
      {
         PageItemIndex itemIndex = cellGrid.GetCellData(cell);
         Cell itemSpan = cellGrid.GetCellSpan(cell);
         CellIndexType minMaxDelta = (direction < 0) ? GetRangeMinValue(rowSlideDelta, rowIndex, itemSpan.Row) : GetRangeMaxValue(rowSlideDelta, rowIndex, itemSpan.Row);
         CellRect itemNewCellRect(cell + Cell(minMaxDelta, 0), itemSpan);

         //occupied cell => propagate delta
         if (!cellGrid.IsCellEmpty(cell))
         {
            if (itemSpan.Row > 0)
            {
               if (cell == cellGrid.FindTopLeftCell(itemIndex))
               {
                  //item exceeds the page => increase delta to move it to next page
                  if (!crtPageRect.Contains(itemNewCellRect))
                  {
                     if ((direction * minMaxDelta) < itemSpan.Column)
                     {
                        ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "SlideRows move item to next page"));
                        minMaxDelta = direction * itemSpan.Column;
                        itemNewCellRect.TopLeft.Column = column + minMaxDelta;
                     }
                  }

                  if (itemNewCellRect != CellRect(cell, itemSpan))
                  {
                     newCellRectMap[itemIndex] = itemNewCellRect;

                     ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "SlideRows move itemIndex=%20s %13s => %s",
                                         HMIBASE_TO_STRING(itemIndex),
                                         HMIBASE_TO_STRING(CellRect(cell, itemSpan)),
                                         HMIBASE_TO_STRING(itemNewCellRect)));
                  }
                  else
                  {
                     //nothing to do
                  }
               }
               //adjust delta for all item rows to min/max
               SetRangeValue(rowSlideDelta, rowIndex, itemSpan.Row, minMaxDelta);
               rowIndex += static_cast<size_t>(itemSpan.Row - 1);
            }
         }
         //empty cell => attenuate non zero delta
         else
         {
            if ((rowSlideDelta[rowIndex] * direction) > 0)
            {
               rowSlideDelta[rowIndex] -= direction;
            }
         }
      }
   }

   //delta for all rows exhausted => column move was successful
   CellIndexType minMaxDelta = (direction) < 0 ? GetRangeMinValue(rowSlideDelta, 0, rowSlideDelta.size()) : GetRangeMaxValue(rowSlideDelta, 0, rowSlideDelta.size());
   if (minMaxDelta == 0)
   {
      return true;
   }

   column += direction;
   //try next column
   if ((column >= 0) && (column < cellGrid.GetColumnCount()))
   {
      return SlideRows(session, cellGrid, direction, column, rowSlideDelta, newCellRectMap);
   }
   //first or last column => column move failed
   else
   {
      return false;
   }
}


/*****************************************************************************/
bool DefaultPageEditStrategy::SlideColumns(Session& session, CellGrid& cellGrid, CellIndexType direction, CellIndexType row, CellIndexVector& columnSlideDelta, CellRectMap& newCellRectMap)
{
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "SlideColumns direction=%d row=%d columnDelta=%s",
                       direction,
                       row,
                       HMIBASE_TO_STRING(columnSlideDelta)));

   if ((direction == 0) || (static_cast<CellIndexType>(columnSlideDelta.size()) != cellGrid.GetColumnCount()) || (row < 0) || (cellGrid.GetRowCount() <= row))
   {
      return false;
   }

   for (size_t columnIndex = 0; columnIndex < columnSlideDelta.size(); ++columnIndex)
   {
      Cell cell(static_cast<CellIndexType>(columnIndex), row);
      if ((columnSlideDelta[columnIndex] != 0) && cellGrid.ContainsCell(cell))
      {
         //occupied cell => propagate delta
         if (!cellGrid.IsCellEmpty(cell))
         {
            Cell span = cellGrid.GetCellSpan(cell);
            if (span.Column > 0)
            {
               //adjust delta for all item column to min/max
               CellIndexType minMaxDelta = (direction < 0) ? GetRangeMinValue(columnSlideDelta, columnIndex, span.Column) : GetRangeMaxValue(columnSlideDelta, columnIndex, span.Column);
               SetRangeValue(columnSlideDelta, columnIndex, span.Column, minMaxDelta);
               columnIndex += static_cast<size_t>(span.Column - 1);

               //if top left item cell also collect the offset
               PageItemIndex itemIndex = cellGrid.GetCellData(cell);
               CellRect newCellRect(cell + Cell(0, minMaxDelta), cellGrid.GetCellSpan(cell));
               if ((cell == cellGrid.FindTopLeftCell(itemIndex)) && (newCellRect != CellRect(cell, span)))
               {
                  newCellRectMap[itemIndex] = newCellRect;

                  ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "SlideColumns move itemIndex=%20s %13s => %s",
                                      HMIBASE_TO_STRING(itemIndex),
                                      HMIBASE_TO_STRING(CellRect(cell, span)),
                                      HMIBASE_TO_STRING(newCellRect)));
               }
            }
         }
         //empty cell => attenuate non zero delta
         else
         {
            if ((columnSlideDelta[columnIndex] * direction) > 0)
            {
               columnSlideDelta[columnIndex] -= direction;
            }
         }
      }
   }

   //delta for all columns exhausted => row move was successful
   CellIndexType minMaxDelta = (direction) < 0 ? GetRangeMinValue(columnSlideDelta, 0, columnSlideDelta.size()) : GetRangeMaxValue(columnSlideDelta, 0, columnSlideDelta.size());
   if (minMaxDelta == 0)
   {
      return true;
   }

   row += direction;
   //try next row
   if ((row >= 0) && (row < cellGrid.GetRowCount()))
   {
      return SlideColumns(session, cellGrid, direction, row, columnSlideDelta, newCellRectMap);
   }
   //first or last row => row move failed
   else
   {
      return false;
   }
}


/*****************************************************************************/
static Item* GetFirstFreeItem(Page& page)
{
   for (size_t index = 0; index < page.Items.size(); ++index)
   {
      Item& item = page.Items[index];
      if (item.EditingCells.Span.IsZero())
      {
         return &item;
      }
   }
   return NULL;
}


/*****************************************************************************/
bool DefaultPageEditStrategy::RearrangeItems(Session& session, CellRectMap& newCellRectMap)
{
   if (session.SourcePage.PointsToNull())
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "RearrangeItems page is null!"));
      return false;
   }

   Album& album = session._Album;
   Page& srcPage = session.SourcePage.GetSharedInstance();
   Item* srcItem = srcPage.GetItem(srcPage.GetItemIndex(session.SourceItemId));
   if (srcItem == NULL)
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "RearrangeItems editing item is null! %s",
                         HMIBASE_TO_STRING(srcPage)));
      return false;
   }

   bool result = false;

   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "RearrangeItems album=%s",
                       HMIBASE_TO_STRING(session._Album.GetId())));
   if (PageEditManager::GetInstance().IsTraceEnabled())
   {
      PageEditManager::GetInstance().PrintBigText(HMIBASE_TO_STRING(session.AlbumGrid));
   }

   CellGrid& albumGrid = session.AlbumGrid;

   //reset cell data in album grid for the source item because that space is available for moving the source item
   CellRect srcItemCells(srcItem->Cells);
   srcItemCells.TopLeft += album.GetPageOffset(session.SourcePage);
   albumGrid.ResetCellData(srcItemCells);

   CellRect destEditingCells(session.DestinationPageEditingCells);
   destEditingCells.TopLeft += album.GetPageOffset(session.DestinationPage);

   //CellRectVector EditingItemCellRectHistory;
   //EditingItemCellRectHistory.push_back(editingItem->EditingCells);
   //result = RearrangeItemsImpl(cellGrid, /*page.*/EditingItemCellRectHistory, page.EditingCells, newCellRectMap);

   if (albumGrid.IsCellRectEmpty(destEditingCells))
   {
      result = true;
   }
   else
   {
      if (session.DestinationItemRegion.Horizontal == CellRegion::HLeft)
      {
         result = true;

         CellIndexType direction = 1;
         CellIndexType column = destEditingCells.TopLeft.Column;

         CellIndexVector rowSlideDelta;
         rowSlideDelta.resize(albumGrid.GetRowCount());
         SetRangeValue(rowSlideDelta, static_cast<size_t>(destEditingCells.TopLeft.Row), static_cast<size_t>(destEditingCells.Span.Row), direction * destEditingCells.Span.Column);

         result = SlideRows(session, albumGrid, direction, column, rowSlideDelta, newCellRectMap);
      }
      else if (session.DestinationItemRegion.Horizontal == CellRegion::HRight)
      {
         result = true;

         CellIndexType direction = -1;
         CellIndexType column = destEditingCells.GetBottomRight().Column;

         CellIndexVector rowSlideDelta;
         rowSlideDelta.resize(albumGrid.GetRowCount());
         SetRangeValue(rowSlideDelta, static_cast<size_t>(destEditingCells.TopLeft.Row), static_cast<size_t>(destEditingCells.Span.Row), direction * destEditingCells.Span.Column);

         result = SlideRows(session, albumGrid, direction, column, rowSlideDelta, newCellRectMap);
      }

      if (!result && ((session.DestinationItemRegion.Vertical == CellRegion::VTop) || (session.DestinationItemRegion.Vertical == CellRegion::VBottom)))
      {
         result = true;

         CellIndexType direction = (session.DestinationItemRegion.Vertical == CellRegion::VTop) ? 1 : -1;

         CellIndexVector columnSlideDelta;
         columnSlideDelta.resize(albumGrid.GetColumnCount());
         SetRangeValue(columnSlideDelta, static_cast<size_t>(destEditingCells.TopLeft.Column), static_cast<size_t>(destEditingCells.Span.Column), direction * destEditingCells.Span.Row);

         CellIndexType row = destEditingCells.TopLeft.Row;
         result = SlideColumns(session, albumGrid, direction, row, columnSlideDelta, newCellRectMap);
      }
   }

   if (result)
   {
      PageItemIndex pageItemIndex(album.GetPageIndex(session.SourcePage), srcPage.GetItemIndex(session.SourceItemId));
      albumGrid.SetCellData(destEditingCells, pageItemIndex);
      newCellRectMap[pageItemIndex] = destEditingCells;

      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "RearrangeItems successful"));
      if (PageEditManager::GetInstance().IsTraceEnabled())
      {
         PageEditManager::GetInstance().PrintBigText(HMIBASE_TO_STRING(newCellRectMap));
      }

//      /*page.*/EditingItemCellRectHistory.push_back(page.EditingCells);
   }
   else
   {
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "RearrangeItems Failed to move row or column."));
      newCellRectMap.clear();
   }

   return result;
}


/*****************************************************************************/
bool DefaultPageEditStrategy::ValidateNewBounds(Session& session, CellRectMap& newCellRectMap)
{
   Album& album(session._Album);
   CellGrid albumGrid;
   session._Album.InitAlbumGrid(albumGrid);

   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "ValidateNewBounds initial album grid"));
   if (PageEditManager::GetInstance().IsTraceEnabled())
   {
      PageEditManager::GetInstance().PrintBigText(HMIBASE_TO_STRING(albumGrid));
   }

   //reset the cells for moved items
   for (CellRectMap::const_iterator it = newCellRectMap.begin(); it != newCellRectMap.end(); ++it)
   {
      PageItemIndex itemIndex(it->first);
      CellRect newCellRect(it->second);

      Page::SharedPointer page = album.GetPage(static_cast<size_t>(itemIndex.PageIndex));
      if (!page.PointsToNull())
      {
         Item* item = page->GetItem(itemIndex.ItemIndex);
         if (item != NULL)
         {
            CellRect itemCells(item->Cells);
            itemCells.TopLeft += album.GetPageOffset(page);
            albumGrid.ResetCellData(itemCells);
         }
      }
   }

   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "ValidateNewBounds intermediate album grid"));
   if (PageEditManager::GetInstance().IsTraceEnabled())
   {
      PageEditManager::GetInstance().PrintBigText(HMIBASE_TO_STRING(albumGrid));
   }

   for (CellRectMap::const_iterator it = newCellRectMap.begin(); it != newCellRectMap.end(); ++it)
   {
      PageItemIndex itemIndex(it->first);
      CellRect newCellRect(it->second);

      Page::SharedPointer newPage = album.GetPage(newCellRect.TopLeft);
      if (newPage.PointsToNull())
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ValidateNewBounds albumCell=%12s not found in album %s",
                            HMIBASE_TO_STRING(newCellRect.TopLeft),
                            HMIBASE_TO_STRING(album.GetId())));
         return false;
      }

      if (album.GetPage(newCellRect.TopLeft) != album.GetPage(newCellRect.GetBottomRight()))
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ValidateNewBounds albumCell=%12s is not in one page in album %s",
                            HMIBASE_TO_STRING(newCellRect),
                            HMIBASE_TO_STRING(album.GetId())));
         return false;
      }

      if (!albumGrid.IsCellRectEmpty(newCellRect))
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ValidateNewBounds albumCell=%12s is not free in album %s",
                            HMIBASE_TO_STRING(newCellRect),
                            HMIBASE_TO_STRING(album.GetId())));
         return false;
      }

      albumGrid.SetCellData(newCellRect, itemIndex);
   }

   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "ValidateNewBounds final album grid"));
   if (PageEditManager::GetInstance().IsTraceEnabled())
   {
      PageEditManager::GetInstance().PrintBigText(HMIBASE_TO_STRING(albumGrid));
   }

   return true;
}


/*****************************************************************************/
bool DefaultPageEditStrategy::SaveNewBounds(Session& session, const CellRectMap& newCellRectMap)
{
   bool changed = false;
   Album& album(session._Album);

   //first, mark as available in their original page the items which were moved to another page
   for (CellRectMap::const_iterator it = newCellRectMap.begin(); it != newCellRectMap.end(); ++it)
   {
      PageItemIndex itemIndex(it->first);
      CellRect newCellRect(it->second);

      Page::SharedPointer newPage = album.GetPage(newCellRect.TopLeft);
      if (newPage.PointsToNull())
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "SaveBounds albumCell=%12s not found in album %s",
                            HMIBASE_TO_STRING(newCellRect.TopLeft),
                            HMIBASE_TO_STRING(album.GetId())));
         return false;
      }

      Page::SharedPointer page = album.GetPage(static_cast<size_t>(itemIndex.PageIndex));
      if (page.PointsToNull())
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "SaveBounds pageIndex=%d not found in album %s",
                            itemIndex.PageIndex,
                            HMIBASE_TO_STRING(album.GetId())));
         return false;
      }

      Item* item = page->GetItem(itemIndex.ItemIndex);
      if (item == NULL)
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "SaveBounds itemIndex=%d not found in page %s",
                            itemIndex.ItemIndex,
                            HMIBASE_TO_STRING(page.GetSharedInstance())));
         return false;
      }

      //item moved to a new page
      if (newPage != page)
      {
         //reset cells
         item->EditingCells = CellRect();

         changed = true;
      }
      //item moved within the same page
      else
      {
         newCellRect.TopLeft -= album.GetPageOffset(page);

         if (item->EditingCells != newCellRect)
         {
            ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "SaveBounds index=%d oldCells=%13s newCells=%13s page=%25s item=%s",
                                itemIndex.ItemIndex,
                                HMIBASE_TO_STRING(item->Cells),
                                HMIBASE_TO_STRING(newCellRect),
                                HMIBASE_TO_STRING(page->Id),
                                HMIBASE_TO_STRING(item->Id)));

            item->EditingCells = newCellRect;
            changed = true;
         }
      }
   }

   //second, set new cells for items moved to new pages
   for (CellRectMap::const_iterator it = newCellRectMap.begin(); it != newCellRectMap.end(); ++it)
   {
      PageItemIndex itemIndex(it->first);
      CellRect newCellRect(it->second);

      Page::SharedPointer newPage = album.GetPage(newCellRect.TopLeft);
      if (newPage.PointsToNull())
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "SaveBounds albumCell=%12s not found in album %s",
                            HMIBASE_TO_STRING(newCellRect.TopLeft),
                            HMIBASE_TO_STRING(album.GetId())));
         return false;
      }

      Page::SharedPointer page = album.GetPage(static_cast<size_t>(itemIndex.PageIndex));
      if (page.PointsToNull())
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "SaveBounds pageIndex=%d not found in album %s",
                            itemIndex.PageIndex,
                            HMIBASE_TO_STRING(album.GetId())));
         return false;
      }

      Item* item = page->GetItem(itemIndex.ItemIndex);
      if (item == NULL)
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "SaveBounds itemIndex=%d not found in page %s",
                            itemIndex.ItemIndex,
                            HMIBASE_TO_STRING(page.GetSharedInstance())));
         return false;
      }

      //item moved to a new page
      if (newPage != page)
      {
         Item* newItem = GetFirstFreeItem(newPage.GetSharedInstance());
         if (newItem == NULL)
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "SaveBounds No available free item in page %s",
                               HMIBASE_TO_STRING(newPage.GetSharedInstance())));
            return false;
         }

         newCellRect.TopLeft -= album.GetPageOffset(newPage);
         newItem->EditingCells = newCellRect;
         newItem->ExternalId = item->Id;

         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "SaveBounds oldIndex=%12s oldCells=%13s oldPage=%25s oldItem=%s =>",
                             HMIBASE_TO_STRING(itemIndex),
                             HMIBASE_TO_STRING(item->Cells),
                             HMIBASE_TO_STRING(page->Id),
                             HMIBASE_TO_STRING(item->Id)));

         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "=> newCells=%13s newPage=%25s newItem=%s",
                             HMIBASE_TO_STRING(newCellRect),
                             HMIBASE_TO_STRING(newPage->Id),
                             HMIBASE_TO_STRING(newItem->Id)));

         changed = true;
      }
   }

   return changed;
}


/*****************************************************************************/
CellIndexType DefaultPageEditStrategy::GetRangeMinValue(const CellIndexVector& v, size_t index, size_t count)
{
   CellIndexType min = 0;
   for (size_t i = index; (i < v.size()) && (i < index + count); ++i)
   {
      if (v[i] < min)
      {
         min = v[i];
      }
   }
   return min;
}


/*****************************************************************************/
CellIndexType DefaultPageEditStrategy::GetRangeMaxValue(const CellIndexVector& v, size_t index, size_t count)
{
   CellIndexType max = 0;
   for (size_t i = index; (i < v.size()) && (i < index + count); ++i)
   {
      if (v[i] > max)
      {
         max = v[i];
      }
   }
   return max;
}


/*****************************************************************************/
void DefaultPageEditStrategy::SetRangeValue(CellIndexVector& v, size_t index, size_t count, CellIndexType value)
{
   for (size_t i = index; (i < v.size()) && (i < index + count); ++i)
   {
      v[i] = value;
   }
}


}
}


}
