/* ***************************************************************************************
* FILE:          PageEditData.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PageEditData.h is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#pragma once

#include <AppUtils/ObjectGuard.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>
#include <Trace/IOSExtensions.h>

#include "PageEditCell.h"
#include "PageEditWidget2D.h"

namespace hmibase {
namespace widget {
namespace gizmo {
class GizmoWidget2D;
}


namespace pageedit {

typedef hmibase::widget::gizmo::GizmoWidget2D ItemWidget;
typedef hmibase::util::ObjectAccessor<ItemWidget> ItemWidgetAccessor;

class PageEditWidget2D;
typedef hmibase::util::ObjectAccessor<PageEditWidget2D> PageWidgetAccessor;

typedef ::FeatStd::UInt32 AlbumId;
typedef std::string ItemId;
typedef std::string PageId;


/*****************************************************************************/
struct Item
{
   ItemId Id;
   ItemWidgetAccessor WidgetAccessor;
   CellRect Cells;

   //item dragged on top of this item
   ItemId ExternalId;

   //cells occupied during editing
   CellRect EditingCells;
};


HMIBASE_STREAM_INSERTION_OP(Item)


/*****************************************************************************/
struct Page
{
   CANDERA_SHARED_POINTER_DECLARATION();
   FEATSTD_TYPEDEF_SHARED_POINTER(Page);

   PageId Id;
   PageWidgetAccessor WidgetAccessor;

   bool CellConfigValid;
   Cell CellCount;
   Candera::Vector2 CellSize;
   Candera::Vector2 PageSize;

   bool ItemsValid;
   typedef ::std::vector<Item> ItemsType;
   ItemsType Items;

   bool IsEditing;

   Page(const PageId& name = PageId());

   static SharedPointer Create(const PageId& name);

   ItemIndexType GetItemIndex(const ItemId& name) const;
   ItemIndexType GetItemIndex(const ItemWidget* widget) const;
   ItemIndexType GetItemIndex(const Cell& editingCell) const;

   Item* GetItem(ItemIndexType index);
   const Item* GetItem(ItemIndexType index) const;

   bool AreItemAccessorsValid() const;
   void ResetEditingInfo();
   void InitItemEditingCells();
};


HMIBASE_STREAM_INSERTION_OP(Page)


/*****************************************************************************/
class Album
{
   public:
      typedef ::std::map<PageId, Page::SharedPointer> PagesType;

      Album(const AlbumId& id);

      const AlbumId& GetId() const;

      const PagesType& GetPages() const;
      PagesType& GetPages();

      size_t GetPageCount() const;
      Page::SharedPointer GetPage(size_t index) const;

      Page::SharedPointer GetPage(const PageId& pageId) const;
      Page::SharedPointer AddNewPage(const PageId& pageId);
      void RemovePage(const PageId& pageId);

      Page::SharedPointer GetPage(const Cell& albumCell) const;

      Cell GetCellCount() const;
      Cell GetPageOffset(Page::SharedPointer page) const;
      PageIndexType GetPageIndex(Page::SharedPointer page) const;

      void InitAlbumGrid(CellGrid& albumGrid) const;

   private:
      Album(const Album&);
      Album& operator=(const Album&);

      AlbumId _id;
      PagesType _pages;
};


HMIBASE_STREAM_INSERTION_OP(Page)


/*****************************************************************************/
struct Session
{
      Album& _Album;
      CellGrid AlbumGrid;

      //page where drag starts
      Page::SharedPointer SourcePage;

      //item which is being dragged
      ItemId SourceItemId;
      Cell SourceItemSpan;
      Cell SourceItemTouchedCell;

      //page where drag ends
      Page::SharedPointer DestinationPage;
      Candera::Vector2 DestinationPageNormalizedTouchedCoordinates;
      Cell DestinationPageTouchedCell;
      CellRect DestinationPageEditingCells;

      //item on which drag ends
      ItemId DestinationItemId;
      CellRegion DestinationItemRegion;

      Session(Album& album);

   private:
      Session(Session&);
      Session& operator=(Session&);
};


HMIBASE_STREAM_INSERTION_OP(Session)
}


}
}
