/* ***************************************************************************************
* FILE:          OPSWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  OPSWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "OPSWidget2D.h"

/*************************OPS SCENE NODE STRUCTURE****************

_______OPS
      |_____OPS_4CH
      |      |_________CAR_IMAGES
      |      |          |--------Rapid_Rear
      |      |          |--------Superb_rear
      |      |          |--------Super_Combi_rear....etc and last is trailer img
      |      |
      |      |_________SEGMENTS
      |                 |______SENSORS_ERROR
      |                 |        |------Sensor_error_warning
      |                 |
      |                 |______REAR_LEFT_OUTSIDE
      |                 |        |______RED
      |                 |        |     |----4 or more Bitmap nodes
      |                 |        |
      |                 |        |______ORANGE
      |                 |        |     |----4 or more Bitmap nodes
      |                 |        |
      |                 |        |______GREY
      |                 |              |----4 or more Bitmap nodes
      |                 |
      |                 |______REAR_LEFT_MIDDLE
      |                 |        |______RED
      |                 |        |     |----11 or more Bitmap nodes
      |                 |        |
      |                 |        |______ORANGE
      |                 |        |     |----11 or more Bitmap nodes
      |                 |        |
      |                 |        |______GREY
      |                 |              |----11 or more Bitmap nodes
      |                 |
      |                 |
      |                 |______REAR_RIGHT_MIDDLE(Similar to REAR_LEFT_MIDDLE)
      |                 |
      |                 |______REAR_RIGHT_OUTSIDE(Similar to REAR_LEFT_OUTSIDE)
      |
      |
      |______OPS_8CH_GROUP
      |        |
      |        |__OPS_8CH_FABIA
      |        |     |
      |        |     |________CAR_IMAGES
      |        |     |        |---------Fabia car Image Bitmap node
      |        |     |        |---------Trailer Image Bitmap node
      |        |     |
      |        |     |________SEGMENTS
      |        |              |
      |        |              |___SENSORS_ERROR
      |        |              |     |----------sensor error warning img front
      |        |              |     |----------sensor error warning img rear
      |        |              |
      |        |              |___FRONT_LEFT_OUTSIDE
      |        |              |        |_____RED(Similar to above red node)
      |        |              |        |_____ORANGE(SSimilar to above Orage node)
      |        |              |        |_____GREY(Similar to above Grey node)
      |        |              |        |_____BACKGROUND
      |        |              |              |-----Error background
      |        |              |              |-----Normal background
      |        |              |
      |        |              |___(Similarly other 7 Sector nodes follow the FRONT_LEFT_OUTSIDE structure)
      |        |              |
      |        |              |______RTA
      |        |                    |_____RTA_LEFT
      |        |                    |        |-----4 RTA warning imgs follow one by one on Bitmap node
      |        |                    |
      |        |                    |_____RTA_RIGHT
      |        |                             |-----4 RTA warning imgs follow one by one on Bitmap node
      |        |
      |        |
      |        |__OPS_8CH_OCTAVIA (Same structure of OPS_8CH_FABIA)
      |        |
      |        |
      |        |__SIMILARLY FOR OTHER 8CH CAR VARIANTS..........
      |
      |
      |
      |
      |_____OPS_16_CH_GROUP (Similar node structure of OPS_8CH_GROUP)


NOTE: For OPS mini mode OPS_4CH Node and its sub nodes will not be present


***********************************************************/

using namespace Candera;
using namespace std;

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_OPS
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/OPSWidget2D.cpp.trc.h"
#endif


//namespace UnderConstruction {

CGI_WIDGET_RTTI_DEFINITION(OPSWidget2D);

#define BLINK_ANIMATION_REPEAT_COUNT 0

OPSWidget2D::OPSWidget2D():
   m_u8CurrentSector(0),
   m_bOPSWidgetTypeUpdate(false),
   m_bDisplayModeUpdate(false),
   m_bTrailerConnected(false),
   m_bVehicleType(false),
   m_bSensorsAvailable(false),
   m_bSensorsSegmentValue(false),
   m_bSegmentPreDefinedDistance(false),
   m_bStatusValueUpdate(false),
   m_bLeftRTASensorLevel(false),
   m_bRightRTASensorLevel(false),
   m_bLeftRTADistanceValue(false),
   m_bRightRTADistanceValue(false),
   m_bSensorFieldHatched(false),
   m_bSensorFieldScanned(false),
   m_bSensorFieldBlink(false),
   m_bBlinkIntervalValue(false),
   m_enPreviousDismode(enRearOnly),
   m_bImgRemoved(false),
   m_u8StartIxFrontSen(0),
   m_u8EndIxFrontSen(0),
   m_u8StartIxRearSen(0),
   m_u8EndIxRearSen(0),
   m_u8StartIxLeftSen(0),
   m_u8EndIxLeftSen(0),
   m_u8StartIxRightSen(0),
   m_u8EndIxRightSen(0),
   _isInvalid(true),
   m_u8MapLastDisplayedSegInfo(),
   _animationHelperInterface(NULL)
   //m_u8aLastDisplayedSegInfo()
{
   _listEventHandler.m_pWidget = this;
}


OPSWidget2D::~OPSWidget2D()
{
   m_u8CurrentSector = 0;
   m_lstbSensorData.Clear();
   //m_lstbSensorsAvailProperty(),
   m_lstu16SegValData.Clear();
   //m_lstu16SegValProperty(),
   m_lstu16PreValData.Clear();
   //m_lstu16PreValProperty(),
   m_lstu16StatusData.Clear();
   m_bOPSWidgetTypeUpdate = false;
   m_bDisplayModeUpdate = false;
   m_bTrailerConnected = false;
   m_bVehicleType = false;
   m_bSensorsAvailable = false;
   m_bSensorsSegmentValue = false;
   m_bSegmentPreDefinedDistance = false;
   m_bStatusValueUpdate = false;
   m_bLeftRTASensorLevel = false;
   m_bLeftRTASensorLevel = false;
   m_bRightRTADistanceValue = false;
   m_bRightRTADistanceValue = false;
   m_enPreviousDismode = enFrontAndRear;
   m_bImgRemoved = false;
   m_u8StartIxFrontSen = 0;
   m_u8EndIxFrontSen = 0;
   m_u8StartIxRearSen = 0;
   m_u8EndIxRearSen = 0;
   m_u8StartIxLeftSen = 0;
   m_u8EndIxLeftSen = 0;
   m_u8StartIxRightSen = 0;
   m_u8EndIxRightSen = 0;
   m_u8MapLastDisplayedSegInfo.clear();

   if (_animationHelperInterface != 0)
   {
      FEATSTD_DELETE(_animationHelperInterface);
   }
   _animationHelperInterface = NULL;
}


/****************************************************************************
*    Function    : Init
*    Description : Initializes the widget so that all referred resource can be
*                                resolved
*    Parameters  : assetFactory Asset provider to resolve all referred resources
*    Return      : void
****************************************************************************/
void OPSWidget2D::InitWidget()
{
   Base::InitWidget();

   for (UInt8 u8In = 0; u8In < 16; u8In++)
   {
      m_lstbSensorData.Insert(u8In, false);
   }

   for (UInt8 u8In = 0; u8In < 16; u8In++)
   {
      m_lstbSensorHatchedData.Insert(u8In, false);
   }

   for (UInt8 u8In = 0; u8In < 16; u8In++)
   {
      m_lstu16SegValData.Insert(u8In, 0);
   }

   for (UInt8 u8In = 0; u8In < 16; u8In++)
   {
      m_lstu16PreValData.Insert(u8In, 0);
   }

   for (UInt8 u8In = 0; u8In < 16; u8In++)
   {
      m_lstu16StatusData.Insert(u8In, 0);
   }
   for (UInt8 u8In = 0; u8In < 8; u8In++)
   {
      m_lstbSensorScannedData.Insert(u8In, true);
   }
   for (UInt8 u8In = 0; u8In < 16; u8In++)
   {
      m_lstbSensorBlinkData.Insert(u8In, false);
   }
   for (UInt8 u8In = 0; u8In < 16; u8In++)
   {
      m_lstu16BlinkIntervalData.Insert(u8In, 0);
   }
   //Animation
   _animationHelperInterface = FEATSTD_NEW(AnimationHelperInterface);
}


/****************************************************************************
*    Function    : Update1
*    Description : overrides WidgetBase::Update
*    Parameters  : void
*    Return      : void
****************************************************************************/
void OPSWidget2D::Update1()
{
   if (m_bSensorFieldBlink)
   {
      vHandleSensorFieldBlink();
      m_bSensorFieldBlink = false;
      _isInvalid = true;
   }
   if (m_bBlinkIntervalValue)
   {
      vHandleBlinkInterval();
      m_bBlinkIntervalValue = false;
      _isInvalid = true;
   }
   if (m_bSensorsAvailable)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : m_bSensorsAvailable update"));
      vRequestSensorsAvailability();
      m_bSensorsAvailable = false;
      _isInvalid = true;
   }

   if (m_bSensorsSegmentValue)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : m_bSensorsSegmentValue update"));
      vRequestsensorssegmentvalue();
      m_bSensorsSegmentValue = false;
      _isInvalid = true;
   }

   if (m_bSegmentPreDefinedDistance)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : m_bSegmentPreDefinedDistance update"));
      vRequestSegmentPreDefinedDistance();
      m_bSegmentPreDefinedDistance = false;
      _isInvalid = true;
   }

   if (m_bStatusValueUpdate)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : m_bStatusValueUpdate update"));
      vRequestStatusValuelist();
      m_bStatusValueUpdate = false;
      _isInvalid = true;
   }

   if (m_bDisplayModeUpdate)
   {
      vHandleOPSDisplayModeUpdate();
      m_bDisplayModeUpdate = false;
      _isInvalid = true;
   }

   if (m_bOPSWidgetTypeUpdate)
   {
      // vHandleOPSWidgetTypeUpdate();
      m_bOPSWidgetTypeUpdate = false;
      _isInvalid = true;
   }
   if (m_bSensorFieldHatched)
   {
      vHandleSensorFieldHatched();
      m_bSensorFieldHatched = false;
      _isInvalid = true;
   }

   if (m_bSensorFieldScanned)
   {
      vHandleSensorFieldScanned();
      m_bSensorFieldScanned = false;
      _isInvalid = true;
   }
}


/****************************************************************************
*    Function    : Update2
*    Description : overrides WidgetBase::Update
*    Parameters  : void
*    Return      : void
****************************************************************************/
void OPSWidget2D::Update2()
{
   if (m_bTrailerConnected)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : m_bTrailerConnected update"));
      vHandleTrailerConnectedUpdate();
      m_bTrailerConnected = false;
      _isInvalid = true;
   }

   if (m_bVehicleType)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : m_bVehicleType update"));
      vHandleVehicleTypeUpdate();
      m_bVehicleType = false;
      _isInvalid = true;
   }

   if (m_bLeftRTASensorLevel)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : m_bLeftRTASensorLevel update"));
      vHandleLeftRTAUpdate();
      m_bLeftRTASensorLevel = false;
      _isInvalid = true;
   }

   if (m_bRightRTASensorLevel)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : m_bRightRTASensorLevel update"));
      vHandleRightRTAUpdate();
      m_bRightRTASensorLevel = false;
      _isInvalid = true;
   }

   if (m_bLeftRTADistanceValue)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : m_bLeftRTADistanceValue update"));
      vHandleLeftRTAUpdate();
      m_bLeftRTADistanceValue = false;
      _isInvalid = true;
   }

   if (m_bRightRTADistanceValue)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : m_bRightRTADistanceValue update"));
      vHandleRightRTAUpdate();
      m_bRightRTADistanceValue = false;
      _isInvalid = true;
   }
}


/****************************************************************************
*    Function    : Update
*    Description : overrides WidgetBase::Update
*    Parameters  : void
*    Return      : void
****************************************************************************/
void OPSWidget2D::Update()
{
   Update1();
   Update2();
   Base::Update();

   if (_isInvalid || ((_animationHelperInterface != 0) && (_animationHelperInterface->isAnimationOnAnyPlayerRunning())))

   {
      Invalidate();
      _isInvalid = false;
   }
}


/******************************************************************************
*    Function    : OnChanged1
*    Description : For any property update this fuction would be called
*    Parameters  : propertyId
*    Return      : void
******************************************************************************/
void OPSWidget2D::OnChanged1(::Courier::UInt32 propertyId)
{
   switch (propertyId)
   {
      case OPSWidgetTypePropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : OPSWidgetTypePropertyId update"));
         m_bOPSWidgetTypeUpdate = true;
         //vHandleOPSWidgetTypeUpdate();
         break;

      case OPSDisplayModePropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : OPSDisplayModePropertyId update"));
         m_bDisplayModeUpdate = true;
         break;

      case TrailerConnectedPropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : TrailerConnectedPropertyId update"));
         m_bTrailerConnected = true;
         break;

      case VehicleTypePropertyId :
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : VehicleTypePropertyId update"));
         m_bVehicleType = true;
         break;

      case SensorsAvailablePropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : SensorsAvailablePropertyId update"));
         m_bSensorsAvailable = true;
         //vRequestSensorsAvailability();
         break;

      case SensorsSegmentValuesPropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : SensorsSegmentValuesPropertyId update"));
         m_bSensorsSegmentValue = true;
         //vRequestsensorssegmentvalue();
         break;

      case SegmentPreDefinedDistanceValuesPropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : SegmentPreDefinedDistanceValuesPropertyId update"));
         m_bSegmentPreDefinedDistance = true;
         //vRequestSegmentPreDefinedDistance();
         break;

      case SensorFieldHatchedPropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : SensorFieldhatchedPropertyId update"));
         m_bSensorFieldHatched = true;
         break;
      case SensorFieldScannedPropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : SensorFieldScannedPropertyId update"));
         m_bSensorFieldScanned = true;
         break;
      case SensorFieldBlinkingPropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : SensorFieldBlinkingPropertyId update"));
         m_bSensorFieldBlink = true;
         break;
      case BlinkIntervalPropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : BlinkIntervalPropertyId update"));
         // vHandleBlinkInterval();
         m_bBlinkIntervalValue = true;
         break;
      default:
         break;
   }
}


/******************************************************************************
*    Function    : OnChanged2
*    Description : For any property update this fuction would be called
*    Parameters  : propertyId
*    Return      : void
******************************************************************************/
void OPSWidget2D::OnChanged2(::Courier::UInt32 propertyId)
{
   switch (propertyId)
   {
      case StatusValuesPropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : StatusValuesPropertyId update"));
         m_bStatusValueUpdate = true;
         //vRequestStatusValuelist();
         break;

      case LeftRTAStatuslevelPropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : LeftRTAStatuslevelPropertyId update"));
         m_bLeftRTASensorLevel = true;
         break;

      case RightRTAStatuslevelPropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : RightRTAStatuslevelPropertyId update"));
         m_bRightRTASensorLevel = true;
         break;

      case LeftRTADistanceValuePropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : LeftRTADistanceValuePropertyId update"));
         m_bLeftRTADistanceValue = true;
         break;

      case RightRTADistanceValuePropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : LeftRTADistanceValuePropertyId update"));
         m_bRightRTADistanceValue = true;
         break;

      default:
         break;
   }
}


/******************************************************************************
*    Function    : OnChanged
*    Description : For any property update this fuction would be called
*    Parameters  : propertyId
*    Return      : void
******************************************************************************/
void OPSWidget2D::OnChanged(::Courier::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   OnChanged1(propertyId);
   OnChanged2(propertyId);
}


/****************************************************************************
*    Function    : vSetImage
*    Description : changes the attached to a BitmapBrush by unloading the
*                  previously attached image and uploading the new one.
*    Parameters  : pRenderNode  Pointer to the render node which holds the BitmapBrush
*                  copImageName The name of the image to be attached.
*              bool  -> True(enable), false(disable)
*    Return      : void
****************************************************************************/
void OPSWidget2D::vSetImage(RenderNode* pRenderNode, bool bFlag)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : SET IMAGE with bFlag = %d", bFlag));

   if (pRenderNode)
   {
      pRenderNode->SetRenderingEnabled(bFlag);
   }
   _isInvalid = true;
   triggerUpdate();
}


/****************************************************************************
*    Function    : vHandleOPSWidgetTypeUpdate
*    Description : Handler for OPS Display based on Sector or Segment
*    Parameters  : none
*    Return      : void
****************************************************************************/
void OPSWidget2D::vHandleOPSWidgetTypeUpdate()
{
   vRemoveAllImgs();
   vDisplayCarImg();
   m_lstbSensorData.Clear();
   m_lstu16SegValData.Clear();
}


/****************************************************************************
*    Function    : vHandleOPSDisplayModeUpdate
*    Description : Handler for OPS Display mode
*    Parameters  : none
*    Return      : void
****************************************************************************/
void OPSWidget2D::vHandleOPSDisplayModeUpdate()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vHandleOPSDisplayModeUpdate"));
   enOPSDisplayMode enDisMode = (enOPSDisplayMode)GetOPSDisplayMode();

   //if(m_enPreviousDismode!=0)
   vRemoveAllImgs();
   m_enPreviousDismode = enDisMode;
   vDisplayCarImg();
   vSetSensorIndex(enDisMode);
   // m_lstbSensorData.Clear();
   // m_lstu16SegValData.Clear();

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : enDisMode [%d]", enDisMode));

   switch (enDisMode)
   {
      case enFrontAndRear:
         vDisplayFrontSensorSectorImgs();
         vDisplayRearSensorSectorImgs();
         break;

      case enAround360:
         vDisplayFrontSensorSectorImgs();
         vDisplayRearSensorSectorImgs();
         vDisplayLeftSensorSectorImgs();
         vDisplayRightSensorSectorImgs();
         break;

      case enRearOnly:
         vDisplayRearSensorSectorImgs();
         break;

      default:
         break;
   }
}


/****************************************************************************
*    Function    : vHandleTrailerConnectedUpdate
*    Description : Handler for trailer connection
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::vHandleTrailerConnectedUpdate()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vHandleTrailerConnectedUpdate"));
   vDisplayRearSensorSectorImgs();
}


/****************************************************************************
*    Function    : vHandleVehicleTypeUpdate
*    Description : Handler for OPS vehicle type
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::vHandleVehicleTypeUpdate()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vHandleVehicleTypeUpdate"));
   //vRemoveAllImgs();
   vDisplayCarImg();
}


/****************************************************************************
*    Function    : vHandleLeftRTAUpdate for left sensor
*    Description : Handler for RTA update based on sensor state and dist.
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::vHandleLeftRTAUpdate()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vHandleLeftRTAUpdate"));
   UInt8 u8SensorState = GetLeftRTAStatuslevel();
   UInt8 u8DistVal = GetLeftRTADistanceValue();
   vDisplayFnsforRTA(u8SensorState, u8DistVal, false);
}


/****************************************************************************
*    Function    : vHandleRightRTAUpdate
*    Description : Handler for RTA update based on sensor state and dist.
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::vHandleRightRTAUpdate()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vHandleRightRTAUpdate"));
   UInt8 u8SensorState = GetRightRTAStatuslevel();
   UInt8 u8DistVal = GetRightRTADistanceValue();
   vDisplayFnsforRTA(u8SensorState, u8DistVal, true);
}


/****************************************************************************
*    Function    : vHandleSensorsAvailableUpdate
*    Description : Handler for OPS sensor changes
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::vHandleSensorsAvailableUpdate(Candera::UInt8 u8SecNo)
{
   bool bIsSectorAvailable = m_lstbSensorData[u8SecNo];
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vHandleSensorsAvailableUpdate [%d],[%d]", u8SecNo, bIsSectorAvailable));

   //If sensor available then check the sensor for the corresponding sector
   if (bIsSectorAvailable)
   {
      if (u8SecNo >= m_u8StartIxFrontSen && u8SecNo <= m_u8EndIxFrontSen)
      {
         vDisplayFrontSensorSectorImgs();
      }
      else if (u8SecNo >= m_u8StartIxRearSen && u8SecNo <= m_u8EndIxRearSen)
      {
         vDisplayRearSensorSectorImgs();
      }
      else if (u8SecNo >= m_u8StartIxLeftSen && u8SecNo <= m_u8EndIxLeftSen)
      {
         vDisplayLeftSensorSectorImgs();
      }
      else if (u8SecNo >= m_u8StartIxRightSen && u8SecNo <= m_u8EndIxRightSen)
      {
         vDisplayRightSensorSectorImgs();
      }
   }
}


/****************************************************************************
*    Function    : vHandleSensorsSegmentValuesUpdate
*    Description : Handler for OPS segment value changes
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::vHandleSensorsSegmentValuesUpdate()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vHandleSensorsSegmentValuesUpdate"));
   vDisplaySegmentImg();
}


/****************************************************************************
*    Function    : vRequestSensorsAvailability
*    Description : Requests sensors availability from list, sent OnSensorsAvailabilityListEvent
*    Parameters  : void
*    Return      : void
****************************************************************************/
void OPSWidget2D::vRequestSensorsAvailability()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vRequestSensorsAvailability"));
   SensorsAvailableType SensorsAvailableList = GetSensorsAvailable();
   UInt32 u32Cnt = static_cast<UInt32>(SensorsAvailableList.Count());
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : u32Cnt = %d", u32Cnt));
   SensorsAvailableList.SetEventCallback(&_listEventHandler, &ListEventHandler::OnSensorsAvailabilityListEvent);
   SensorsAvailableList.Request(0, u32Cnt);
}


/****************************************************************************
*    Function    : vOnSensorsAvailabilityListEvent
*    Description : Call back for sensor availability
*    Parameters  : const Courier::ListEvent &
*    Return      : void
****************************************************************************/
void OPSWidget2D::ListEventHandler::OnSensorsAvailabilityListEvent(const Courier::ListEvent& SensorListEvent)
{
   if (m_pWidget == 0)
   {
      return;
   }

   bool l_bSensorData = false;
   const bool* l_pSensorData = SensorListEvent.GetItem<bool>();

   if (l_pSensorData != 0)
   {
      l_bSensorData = *l_pSensorData;
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D:OnSensorsAvailability [%19s] received %d: newIndex=%u, oldIndex=%u, l_bSensorData=%d ",
                       m_pWidget->GetLegacyName(),
                       SensorListEvent.EventType(),
                       SensorListEvent.NewIndex(),
                       SensorListEvent.OldIndex(),
                       l_bSensorData
                      ));
   bool l_bResult = true;

   switch (SensorListEvent.EventType())
   {
      // list item delivery bay - the order requested items come in is not specified
      case Courier::ListEventType::RequestedItem:
      {
         //m_lstbSensorData.Insert(((UInt16)SensorListEvent.NewIndex()), (bool)l_bSensorData);
         m_pWidget->vAddNewSensorData(m_pWidget->m_lstbSensorData, (UInt16)SensorListEvent.NewIndex(), l_bSensorData);
         //ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(),"OPSWidget2D : m_lstbSensorData.Insert & size %d", m_lstbSensorData.Size()));
      }
      break;

      // change & add event handling
      case Courier::ListEventType::ModifiedItem:
      case Courier::ListEventType::AddedItem:
      {
         m_pWidget->vAddNewSensorData(m_pWidget->m_lstbSensorData, (UInt16)SensorListEvent.NewIndex(), l_bSensorData);

         // insert item to local list
         if (!SensorListEvent.HasDataItem())
         {
            // if the event does not carry the modified data item, create a request for it
            l_bResult = m_pWidget->m_lstbSensorsAvailProperty.Request(SensorListEvent.NewIndex(), 1);
         }
      }
      break;

      case Courier::ListEventType::RefreshList:
      case Courier::ListEventType::ItemCountChanged:
      {
         l_bResult = m_pWidget->m_lstbSensorsAvailProperty.Request(0, m_pWidget->m_lstbSensorsAvailProperty.Count());
      }
      break;

      default:
      {
      }
      break;
   }

   if (!l_bResult)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: [%s]. OnSensorsAvaila l_bResult = false", m_pWidget->GetLegacyName()));
   }
}


/****************************************************************************
*    Function    : vAddNewSensorData
*    Description : Add function for the new sensor
*    Parameters  : u16Index, bool DataEntry
*    Return      : bool
****************************************************************************/
void OPSWidget2D::vAddNewSensorData(FeatStd::Internal::Vector<bool>& Sendata, FeatStd::UInt16 u16Index, const bool& DataEntry)
{
   bool l_bIsDataStored = true;
   UInt32 l_NumberOfItemsInList = static_cast<UInt32>(Sendata.Size());
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: vAddNewSensorData[%19s] - u16Index = [%d], and DataEntry = [%d]",
                       GetLegacyName(), u16Index, DataEntry));

   if (l_NumberOfItemsInList <= u16Index)
   {
      // resize vector to store the index value at given index
      Candera::UInt32 l_Index = l_NumberOfItemsInList;

      for (; l_Index < u16Index; ++l_Index)
      {
         l_bIsDataStored = Sendata.Insert(static_cast<Int>(l_Index), DataEntry);
      }

      l_bIsDataStored = (l_bIsDataStored == true) ? Sendata.Insert(u16Index, DataEntry) : false;
   }
   else
   {
      Sendata[u16Index] = DataEntry;
   }

   if (!m_lstbSensorData.Empty() && u16Index < m_lstbSensorData.Size())
   {
      UInt32 l_NewSize = static_cast<UInt32>(Sendata.Size());
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: l_NewSize = [%d]", l_NewSize));

      if (l_NewSize)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: m_lstbSensorData[%d] = [%d]", u16Index, Sendata[u16Index]));
      }

      m_u8CurrentSector = (UInt8)u16Index;

      if (m_lstbSensorData[u16Index])
      {
         vHandleSensorsSegmentValuesUpdate();
      }
      else
      {
         vRemoveSectorNodeImgs(m_u8CurrentSector);
      }
   }

   if (false == l_bIsDataStored)
   {
      ETG_TRACE_SYS_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: [%s]. Adding new data failed."
                         "Invalid index or memory reallocation failed\n",
                         GetLegacyName()
                        ));
   }
}


/****************************************************************************
*    Function    : vRequestsensorssegmentvalue
*    Description : Requests sensors segment value from list, sent OnSensorsAvailabilityListEvent
*    Parameters  : void
*    Return      : void
****************************************************************************/
void OPSWidget2D::vRequestsensorssegmentvalue()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vRequestsensorssegmentvalue"));
   SensorsSegmentValuesType SegmentvalueList = GetSensorsSegmentValues();
   UInt32 u32Cnt = static_cast<UInt32>(SegmentvalueList.Count());
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : u32Cnt = %d", u32Cnt));
   SegmentvalueList.SetEventCallback(&_listEventHandler, &ListEventHandler::OnSegmentUpdateListEvent);
   SegmentvalueList.Request(0, u32Cnt);
}


/****************************************************************************
*    Function    : vOnSegmentUpdateListEvent
*    Description : Call back fns for OPS segment value changes
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::ListEventHandler::OnSegmentUpdateListEvent(const Courier::ListEvent& SegValListEvent)
{
   if (m_pWidget == 0)
   {
      return;
   }

   Courier::UInt16 l_u16SegData = 0;
   const Courier::UInt16* l_pu16SegData = SegValListEvent.GetItem<Courier::UInt16>();

   if (l_pu16SegData != 0)
   {
      // if the list event carries a list item
      l_u16SegData = *l_pu16SegData;
   }
   else
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: [%s]. l_u16SegData = NULL", m_pWidget->GetLegacyName()));
      //l_u16SegData = 10;
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D:OnSegmentUpdate [%19s] received %d: newIndex=%u, oldIndex=%u, l_u16SegData=%d ",
                       m_pWidget->GetLegacyName(),
                       SegValListEvent.EventType(),
                       SegValListEvent.NewIndex(),
                       SegValListEvent.OldIndex(),
                       l_u16SegData
                      ));
   bool l_bResult = true;

   switch (SegValListEvent.EventType())
   {
      // list item delivery bay - the order requested items come in is not specified
      case Courier::ListEventType::RequestedItem:
      {
         //m_lstu16SegValData.Insert(((UInt16)SegValListEvent.NewIndex()), l_u16SegData);
         m_pWidget->vAddNewSegmnetData(m_pWidget->m_lstu16SegValData, (UInt16)SegValListEvent.NewIndex(), l_u16SegData);
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : m_lstu16SegValData.Insert & size %d", m_pWidget->m_lstu16SegValData.Size()));
      }
      break;

      // add & change event handling
      case Courier::ListEventType::ModifiedItem:
      case Courier::ListEventType::AddedItem:
      {
         m_pWidget->vAddNewSegmnetData(m_pWidget->m_lstu16SegValData, (UInt16)SegValListEvent.NewIndex(), l_u16SegData);

         // insert item to local list
         if (!SegValListEvent.HasDataItem())
         {
            // if the event does not carry the modified data item, create a request for it
            l_bResult = m_pWidget->m_lstu16SegValProperty.Request(SegValListEvent.NewIndex(), 1);
         }
      }
      break;

      case Courier::ListEventType::RefreshList:
      case Courier::ListEventType::ItemCountChanged:
      {
         l_bResult = m_pWidget->m_lstu16SegValProperty.Request(0, m_pWidget->m_lstu16SegValProperty.Count());
      }
      break;

      default:
      {
      }
      break;
   }

   if (!l_bResult)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: [%s]. OnSegment l_bResult = false ", m_pWidget->GetLegacyName()));
   }
}


void OPSWidget2D::vHandleSensorFieldScanned()
{
   SensorFieldScannedType ScannedvalueList = GetSensorFieldScanned();
   UInt32 u32Cnt = static_cast<UInt32>(ScannedvalueList.Count());
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : u32Cnt = %d", u32Cnt));
   ScannedvalueList.SetEventCallback(&_listEventHandler, &ListEventHandler::OnScannedUpdateListEvent);
   ScannedvalueList.Request(0, u32Cnt);
}


void OPSWidget2D::ListEventHandler::OnScannedUpdateListEvent(const Courier::ListEvent& ScannedValListEvent)
{
   if (m_pWidget == 0)
   {
      return;
   }

   bool l_bScanned = false;
   const bool* l_pbScannedData = ScannedValListEvent.GetItem<bool>();
   if (l_pbScannedData != NULL)
   {
      l_bScanned = *l_pbScannedData;
   }
   switch (ScannedValListEvent.EventType())
   {
      case Courier::ListEventType::RequestedItem:
      {
         m_pWidget->vAddNewScannedData(m_pWidget->m_lstbSensorScannedData, (UInt16)ScannedValListEvent.NewIndex(), l_bScanned);
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : m_lstbSensorScannedData.Insert & size %d", m_pWidget->m_lstbSensorScannedData.Size()));
      }
      break;
      case Courier::ListEventType::ModifiedItem:
      case Courier::ListEventType::AddedItem:
      {
         m_pWidget->vAddNewScannedData(m_pWidget->m_lstbSensorScannedData, (UInt16)ScannedValListEvent.NewIndex(), l_bScanned);
         if (!ScannedValListEvent.HasDataItem())
         {
            m_pWidget->m_lstbScannedValProperty.Request(ScannedValListEvent.NewIndex(), 1);
         }
      }
      break;
      case Courier::ListEventType::RefreshList:
      case Courier::ListEventType::ItemCountChanged:
      {
         m_pWidget->m_lstbScannedValProperty.Request(0, m_pWidget->m_lstbScannedValProperty.Count());
      }
      break;
      default:
      {
      }
      break;
   }
   m_pWidget->vHandleSensorsSegmentValuesUpdate();
}


void OPSWidget2D::vHandleSensorFieldHatched()
{
   SensorFieldHatchedType HatchedvalueList = GetSensorFieldHatched();
   UInt32 u32Cnt = static_cast<UInt32>(HatchedvalueList.Count());
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : u32Cnt = %d", u32Cnt));
   HatchedvalueList.SetEventCallback(&_listEventHandler, &ListEventHandler::OnHatchedUpdateListEvent);
   HatchedvalueList.Request(0, u32Cnt);
}


void OPSWidget2D::ListEventHandler::OnHatchedUpdateListEvent(const Courier::ListEvent& HatchedValListEvent)
{
   if (m_pWidget == 0)
   {
      return;
   }

   bool l_bHatched = false;
   const bool* l_pbHatchedData = HatchedValListEvent.GetItem<bool> ();
   if (l_pbHatchedData != NULL)
   {
      l_bHatched = *l_pbHatchedData;
   }
   switch (HatchedValListEvent.EventType())
   {
      case Courier::ListEventType::RequestedItem:
      {
         m_pWidget->vAddNewHatchedData(m_pWidget->m_lstbSensorHatchedData, (UInt16)HatchedValListEvent.NewIndex(), l_bHatched);
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : m_lstbSensorHatchedData.Insert & size %d", m_pWidget->m_lstbSensorHatchedData.Size()));
      }
      break;
      case Courier::ListEventType::ModifiedItem:
      case Courier::ListEventType::AddedItem:
      {
         m_pWidget->vAddNewHatchedData(m_pWidget->m_lstbSensorHatchedData, (UInt16)HatchedValListEvent.NewIndex(), l_bHatched);
         if (!HatchedValListEvent.HasDataItem())
         {
            m_pWidget->m_lstbHatchedValProperty.Request(HatchedValListEvent.NewIndex(), 1);
         }
      }
      break;
      case Courier::ListEventType::RefreshList:
      case Courier::ListEventType::ItemCountChanged:
      {
         m_pWidget->m_lstbHatchedValProperty.Request(0, m_pWidget->m_lstbHatchedValProperty.Count());
      }
      break;
      default:
      {
      }
      break;
   }
   m_pWidget->vHandleSensorsSegmentValuesUpdate();
}


/****************************************************************************
*    Function    : bAddNewSegmnetData
*    Description :Add fns for OPS segment value changes
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::vAddNewSegmnetData(FeatStd::Internal::Vector<Courier::UInt16>& Segdata, FeatStd::UInt16 u16Index, const Courier::UInt16& DataEntry)
{
   bool l_bIsDataStored = true;
   UInt32 l_NumberOfItemsInList = static_cast<UInt32>(Segdata.Size());
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: vAddNewSegmnetData[%19s] - l_NumberOfItemsInList = [%d]u16Index = [%d], and DataEntry = [%d]",
                       GetLegacyName(), l_NumberOfItemsInList, u16Index, DataEntry));

   if (l_NumberOfItemsInList <= u16Index)
   {
      // resize vector to store the index value at given index
      Candera::UInt32 l_Index = l_NumberOfItemsInList;

      for (; l_Index < u16Index; ++l_Index)
      {
         l_bIsDataStored = Segdata.Insert(static_cast<Int>(l_Index), DataEntry);
      }
      l_bIsDataStored = (l_bIsDataStored == true) ? Segdata.Insert(u16Index, DataEntry) : false;
   }
   else
   {
      Segdata[u16Index] = DataEntry;
   }

   if (!m_lstbSensorData.Empty() && u16Index < m_lstbSensorData.Size())
   {
      UInt32 l_NewSize = static_cast<UInt32>(Segdata.Size());
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: l_NewSize = [%d]", l_NewSize));

      if (l_NewSize)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: m_lstbSensorData[%d] = [%d]", u16Index, m_lstbSensorData[u16Index]));
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: Segdata[%d] = [%d]", u16Index, Segdata[u16Index]));
      }

      m_u8CurrentSector = (UInt8)u16Index;

      if (m_lstbSensorData[u16Index])
      {
         vHandleSensorsSegmentValuesUpdate();
      }
      else
      {
         vRemoveSectorNodeImgs(m_u8CurrentSector);
      }
   }

   if (false == l_bIsDataStored)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: [%s]. Adding new data failed."
                          "Invalid index or memory reallocation failed\n",
                          GetLegacyName()
                         ));
   }
}


void OPSWidget2D::vAddNewScannedData(FeatStd::Internal::Vector<bool>& Segdata, FeatStd::UInt16 u16Index, const bool& DataEntry)
{
   bool l_bIsDataStored = true;
   UInt32 l_NumberOfItemsInList = static_cast<UInt32>(Segdata.Size());
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: vAddNewSegmnetData[%19s] - l_NumberOfItemsInList = [%d]u16Index = [%d], and DataEntry = [%d]",
                       GetLegacyName(), l_NumberOfItemsInList, u16Index, DataEntry));
   if (l_NumberOfItemsInList <= u16Index)
   {
      Candera::UInt32 l_Index = l_NumberOfItemsInList;
      for (; l_Index < u16Index; ++l_Index)
      {
         l_bIsDataStored = Segdata.Insert(static_cast<Int>(l_Index), DataEntry);
      }
      l_bIsDataStored = (l_bIsDataStored == true) ? Segdata.Insert(u16Index, DataEntry) : false;
   }
   else
   {
      Segdata[u16Index] = DataEntry;
   }
   if (!m_lstbSensorScannedData.Empty() && u16Index < m_lstbSensorScannedData.Size())
   {
      UInt32 l_NewSize = static_cast<UInt32>(Segdata.Size());
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: l_NewSize = [%d]", l_NewSize));

      if (l_NewSize)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: m_lstbSensorScannedData[%d] = [%d]", u16Index, m_lstbSensorScannedData[u16Index]));
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: Segdata[%d] = [%d]", u16Index, Segdata[u16Index]));
      }

      m_u8CurrentSector = static_cast<UInt8>(u16Index + 8);

      if (m_lstbSensorScannedData[u16Index])
      {
         vHandleSensorsSegmentValuesUpdate();
      }
      else
      {
         vRemoveSectorNodeImgs(m_u8CurrentSector);
         if (m_lstbSensorData[m_u8CurrentSector])
         {
            vDisplayBackgroundImg(m_u8CurrentSector, m_lstbSensorHatchedData[m_u8CurrentSector], true, DataEntry);
         }
      }
   }

   if (false == l_bIsDataStored)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: [%s]. Adding new data failed."
                          "Invalid index or memory reallocation failed\n",
                          GetLegacyName()
                         ));
   }
}


void OPSWidget2D::vAddNewHatchedData(FeatStd::Internal::Vector<bool>& Segdata, FeatStd::UInt16 u16Index, const bool& DataEntry)
{
   bool l_bIsDataStored = true;
   UInt32 l_NumberOfItemsInList = static_cast<UInt32>(Segdata.Size());
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: vAddNewSegmnetData[%19s] - l_NumberOfItemsInList = [%d]u16Index = [%d], and DataEntry = [%d]",
                       GetLegacyName(), l_NumberOfItemsInList, u16Index, DataEntry));
   if (l_NumberOfItemsInList <= u16Index)
   {
      Candera::UInt32 l_Index = l_NumberOfItemsInList;
      for (; l_Index < u16Index; ++l_Index)
      {
         l_bIsDataStored = Segdata.Insert(static_cast<Int>(l_Index), DataEntry);
      }
      l_bIsDataStored = (l_bIsDataStored == true) ? Segdata.Insert(u16Index, DataEntry) : false;
   }
   else
   {
      Segdata[u16Index] = DataEntry;
   }
   if (!m_lstbSensorHatchedData.Empty() && u16Index < m_lstbSensorHatchedData.Size())
   {
      UInt32 l_NewSize = static_cast<UInt32>(Segdata.Size());
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: l_NewSize = [%d]", l_NewSize));

      if (l_NewSize)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: m_lstbSensorScannedData[%d] = [%d]", u16Index, m_lstbSensorHatchedData[u16Index]));
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: Segdata[%d] = [%d]", u16Index, Segdata[u16Index]));
      }

      m_u8CurrentSector = (UInt8)u16Index;

      if (m_lstbSensorHatchedData[u16Index])
      {
         vHandleSensorsSegmentValuesUpdate();
      }
      else
      {
         vRemoveSectorNodeImgs(m_u8CurrentSector);
      }
   }

   if (false == l_bIsDataStored)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: [%s]. Adding new data failed."
                          "Invalid index or memory reallocation failed\n",
                          GetLegacyName()
                         ));
   }
}


/****************************************************************************
*    Function    : vRequestSegmentPreDefinedDistance
*    Description : Requests sensors segment pre defined value from list,
*                  sent OnSensorsAvailabilityListEvent
*    Parameters  : void
*    Return      : void
****************************************************************************/
void OPSWidget2D::vRequestSegmentPreDefinedDistance()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vRequestSegmentPreDefinedDistance"));
   SegmentPreDefinedDistanceValuesType SegmentprevalueList = GetSegmentPreDefinedDistanceValues();
   UInt32 u32Cnt = static_cast<UInt32>(SegmentprevalueList.Count());
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : u32Cnt %d", u32Cnt));
   SegmentprevalueList.SetEventCallback(&_listEventHandler, &ListEventHandler::OnSegmentPreValUpdateListEvent);
   SegmentprevalueList.Request(0, u32Cnt);
}


/****************************************************************************
*    Function    : vOnSegmentPreValUpdateListEvent
*    Description : Handler for OPS pre defined distance value changes
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::ListEventHandler::OnSegmentPreValUpdateListEvent(const Courier::ListEvent& PreValListEvent)
{
   if (m_pWidget == 0)
   {
      return;
   }

   Courier::UInt16 l_u16PreData = 0;
   const Courier::UInt16* l_pPreData = PreValListEvent.GetItem<Courier::UInt16>();

   if (l_pPreData != 0)
   {
      // if the list event carries a list item
      l_u16PreData = *l_pPreData;
   }
   else
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: [%s]. l_u16PreData = NULL", m_pWidget->GetLegacyName()));
      l_u16PreData = 10;
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D:OnSegmentPreValUpdate [%19s] received %d: newIndex=%u, oldIndex=%u ",
                       m_pWidget->GetLegacyName(),
                       PreValListEvent.EventType(),
                       PreValListEvent.NewIndex(),
                       PreValListEvent.OldIndex()
                      ));
   bool l_bResult = true;

   switch (PreValListEvent.EventType())
   {
      // list item delivery bay - the order requested items come in is not specified
      case Courier::ListEventType::RequestedItem:
      {
         m_pWidget->vAddNewPreValData(m_pWidget->m_lstu16PreValData, (UInt16)PreValListEvent.NewIndex(), l_u16PreData);
         //m_lstu16PreValData.Insert(((UInt16)PreValListEvent.NewIndex()), l_u16PreData);
         //ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(),"OPSWidget2D : m_lstu16SegValData.Insert & size %d", m_lstu16PreValData.Size()));
      }
      break;

      // add & change event handling
      case Courier::ListEventType::ModifiedItem:
      case Courier::ListEventType::AddedItem:
      {
         m_pWidget->vAddNewPreValData(m_pWidget->m_lstu16PreValData, (UInt16)PreValListEvent.NewIndex(), l_u16PreData);

         // insert item to local list
         if (!PreValListEvent.HasDataItem())
         {
            // if the event does not carry the modified data item, create a request for it
            l_bResult = m_pWidget->m_lstu16PreValProperty.Request(PreValListEvent.NewIndex(), 1);
         }
      }
      break;

      case Courier::ListEventType::RefreshList:
      case Courier::ListEventType::ItemCountChanged:
      {
         l_bResult = m_pWidget->m_lstu16PreValProperty.Request(0, m_pWidget->m_lstu16PreValProperty.Count());
      }
      break;

      default:
      {
      }
      break;
   }

   if (!l_bResult)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: [%s]. Onpredef value l_bResult = false ", m_pWidget->GetLegacyName()));
   }
}


/****************************************************************************
*    Function    : bAddNewPreValData
*    Description : Add fns for OPS pre defined distance value changes
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::vAddNewPreValData(FeatStd::Internal::Vector<Courier::UInt16>& PreSegdata, FeatStd::UInt16 u16Index, const Courier::UInt16& DataEntry)
{
   bool l_bIsDataStored = true;
   UInt32 l_NumberOfItemsInList = static_cast<UInt32>(PreSegdata.Size());
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: vAddNewPreValData[%19s] - u16Index = [%d], and DataEntry = [%d]",
                       GetLegacyName(), u16Index, DataEntry));

   if (l_NumberOfItemsInList <= u16Index)
   {
      // resize vector to store the index value at given index
      Candera::UInt32 l_Index = l_NumberOfItemsInList;

      for (; l_Index < u16Index; ++l_Index)
      {
         l_bIsDataStored = PreSegdata.Insert(static_cast<Int>(l_Index), DataEntry);
      }

      l_bIsDataStored = (l_bIsDataStored == true) ? PreSegdata.Insert(u16Index, DataEntry) : false;
   }
   else
   {
      PreSegdata[u16Index] = DataEntry;
   }

   if (!m_lstbSensorData.Empty() && u16Index < m_lstbSensorData.Size())
   {
      UInt32 l_NewSize = static_cast<UInt32>(PreSegdata.Size());
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: l_NewSize = [%d]", l_NewSize));

      if (l_NewSize)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: preSegdata[%d] = [%d]", u16Index, PreSegdata[u16Index]));
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: m_lstbSensorData[%d] = [%d]", u16Index, m_lstbSensorData[u16Index]));
      }

      m_u8CurrentSector = (UInt8)u16Index;

      if (m_lstbSensorData[u16Index])
      {
         vHandleSensorsSegmentValuesUpdate();
      }
      else
      {
         vRemoveSectorNodeImgs(m_u8CurrentSector);
      }
   }

   if (false == l_bIsDataStored)
   {
      ETG_TRACE_SYS_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: [%s]. Adding new data failed."
                         "Invalid index or memory reallocation failed\n",
                         GetLegacyName()
                        ));
   }
}


/****************************************************************************
*    Function    : vRequestStatusValuelist
*    Description : Requests sensors segment pre defined value from list,
*                  sent OnSensorsAvailabilityListEvent
*    Parameters  : void
*    Return      : void
****************************************************************************/
void OPSWidget2D::vRequestStatusValuelist()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vRequestStatusValuelist"));
   StatusValuesType StatusvalueList = GetStatusValues();
   UInt32 u32Cnt = static_cast<UInt32>(StatusvalueList.Count());
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : u32Cnt %d", u32Cnt));
   StatusvalueList.SetEventCallback(&_listEventHandler, &ListEventHandler::OnStatusUpdateListEvent);
   StatusvalueList.Request(0, u32Cnt);
}


/****************************************************************************
*    Function    : vOnStatusUpdateListEvent
*    Description : Handler for OPS pre defined distance value changes
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::ListEventHandler::OnStatusUpdateListEvent(const Courier::ListEvent& StatusValListEvent)
{
   if (m_pWidget == 0)
   {
      return;
   }

   Courier::UInt16 l_u16PreData = 0;
   const Courier::UInt16* l_pPreData = StatusValListEvent.GetItem<Courier::UInt16>();

   if (l_pPreData != 0)
   {
      // if the list event carries a list item
      l_u16PreData = *l_pPreData;
   }
   else
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: [%19s]. l_u16PreData = NULL", m_pWidget->GetLegacyName()));
      l_u16PreData = 10;
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D:OnSegmentPreValUpdate [%19s] received %d: newIndex=%u, oldIndex=%u ",
                       m_pWidget->GetLegacyName(),
                       StatusValListEvent.EventType(),
                       StatusValListEvent.NewIndex(),
                       StatusValListEvent.OldIndex()
                      ));
   bool l_bResult = true;

   switch (StatusValListEvent.EventType())
   {
      // list item delivery bay - the order requested items come in is not specified
      case Courier::ListEventType::RequestedItem:
      {
         m_pWidget->vAddNewStatusData(m_pWidget->m_lstu16StatusData, (UInt16)StatusValListEvent.NewIndex(), l_u16PreData);
      }
      break;

      // change event handling
      case Courier::ListEventType::ModifiedItem:
      {
         m_pWidget->vAddNewStatusData(m_pWidget->m_lstu16StatusData, (UInt16)StatusValListEvent.NewIndex(), l_u16PreData);

         // change data in local list
         if (!StatusValListEvent.HasDataItem())
         {
            // if the event does not carry the modified data item, create a request for it
            l_bResult = m_pWidget->m_lstu16StatusProperty.Request(StatusValListEvent.NewIndex(), 1);
         }
      }
      break;

      // add event handling
      case Courier::ListEventType::AddedItem:
      {
         m_pWidget->vAddNewStatusData(m_pWidget->m_lstu16StatusData, (UInt16)StatusValListEvent.NewIndex(), l_u16PreData);

         // insert item to local list
         if (!StatusValListEvent.HasDataItem())
         {
            // if the event does not carry the modified data item, create a request for it
            l_bResult = m_pWidget->m_lstu16StatusProperty.Request(StatusValListEvent.NewIndex(), 1);
         }
      }
      break;

      case Courier::ListEventType::RefreshList:
      case Courier::ListEventType::ItemCountChanged:
      {
         l_bResult = m_pWidget->m_lstu16StatusProperty.Request(0, m_pWidget->m_lstu16PreValProperty.Count());
      }
      break;

      default:
      {
      }
      break;
   }

   if (!l_bResult)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: [%s]. Onstatus value l_bResult = false ", m_pWidget->GetLegacyName()));
   }
}


/****************************************************************************
*    Function    : vAddNewStatusData
*    Description : Add fns for OPS pre defined distance value changes
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::vAddNewStatusData(FeatStd::Internal::Vector<Courier::UInt16>& Statusdata, FeatStd::UInt16 u16Index, const Courier::UInt16& DataEntry)
{
   bool l_bIsDataStored = true;
   UInt32 l_NumberOfItemsInList = static_cast<UInt32>(Statusdata.Size());
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: vAddNewStatusData[%19s] - u16Index = [%d], and DataEntry = [%d]",
                       GetLegacyName(), u16Index, DataEntry));

   if (l_NumberOfItemsInList <= u16Index)
   {
      // resize vector to store the index value at given index
      Candera::UInt32 l_Index = l_NumberOfItemsInList;

      for (; l_Index < u16Index; ++l_Index)
      {
         l_bIsDataStored = Statusdata.Insert(static_cast<Int>(l_Index), DataEntry);
      }

      l_bIsDataStored = (l_bIsDataStored == true) ? Statusdata.Insert(u16Index, DataEntry) : false;
   }
   else
   {
      Statusdata[u16Index] = DataEntry;
   }

   if (!m_lstbSensorData.Empty() && u16Index < m_lstbSensorData.Size())
   {
      UInt32 l_NewSize = static_cast<UInt32>(Statusdata.Size());
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: l_NewSize = [%d]", l_NewSize));

      if (l_NewSize)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: Statusdata[%d] = [%d]", u16Index, Statusdata[u16Index]));
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: m_lstbSensorData[%d] = [%d]", u16Index, m_lstbSensorData[u16Index]));
      }

      m_u8CurrentSector = (UInt8)u16Index;

      if (m_lstbSensorData[u16Index])
      {
         vHandleSensorsSegmentValuesUpdate();
      }
      else
      {
         vRemoveSectorNodeImgs(m_u8CurrentSector);
      }
   }

   if (false == l_bIsDataStored)
   {
      ETG_TRACE_SYS_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: [%s]. Adding new status data failed."
                         "Invalid index or memory reallocation failed\n",
                         GetLegacyName()
                        ));
   }
}


/****************************************************************************
*    Function    : GetBasicNode
*    Description : Retrieves the base node based on display type.
*    Parameters  : enDisplayType
*    Return      : Candera::Node2D*
****************************************************************************/
Candera::Node2D* OPSWidget2D::GetBasicNode(enOPSDisplayMode enDisplayType) const
{
   Candera::Node2D* node = GetNode();

   if (node != 0)
   {
      node = node->GetFirstChild();
   }

   UInt8 u8DisplayType = (UInt8)enDisplayType;

   if (true == (bool)GetOPSMini())
   {
      u8DisplayType--;
   }

   for (UInt8 u8Index = 1; u8Index < u8DisplayType; u8Index++)
   {
      if (node != 0)
      {
         node = node->GetNextSibling();
      }
   }

   return (node != 0) ? node : 0;
}


/****************************************************************************
*    Function    : GetCarNode
*    Description : Retrieves the render node which holds all the Car images.
*    Parameters  : none
*    Return      : Candera::Node2D*
****************************************************************************/
Candera::Node2D* OPSWidget2D::GetCarNode()const
{
   enOPSDisplayMode enDisMode = (enOPSDisplayMode)GetOPSDisplayMode();
   Candera::Node2D* node = GetBasicNode(enDisMode);

   if (enDisMode == enRearOnly)
   {
      if (node != 0)
      {
         node = node->GetFirstChild();
      }

      return (node != 0) ? (node) : 0;
   }
   else
   {
      if (node != 0)
      {
         node = node->GetFirstChild();
      }

      VehicleTypeType u16VehicleType = GetVehicleType();

      for (UInt8 u8Index = 1; u8Index < u16VehicleType; u8Index++)
      {
         if (node != 0)
         {
            node = node->GetNextSibling();
         }
      }

      /*if (node != 0)
      {
         node = node->GetFirstChild();
      }*/
   }

   return (node != 0) ? (node) : 0;
}


/****************************************************************************
*    Function    : GetNthCarImgNode
*    Description : Retrieves the render node which represents the Car image.
*    Parameters  : none
*    Return      : RenderNode*
****************************************************************************/
Candera::RenderNode* OPSWidget2D::GetNthCarImgNode() const
{
   enOPSDisplayMode enDisMode = (enOPSDisplayMode)GetOPSDisplayMode();
   Candera::Node2D* node = GetCarNode();

   if (enDisMode == enRearOnly)
   {
      if (node != 0)
      {
         node = node->GetFirstChild();
      }

      VehicleTypeType u16VehicleType = GetVehicleType();

      for (UInt8 u8Index = 1; u8Index < u16VehicleType; u8Index++)
      {
         if (node != 0)
         {
            node = node->GetNextSibling();
         }
      }
   }
   else
   {
      if (node != 0)
      {
         node = node->GetFirstChild();
      }
   }

   return (node != 0) ? Candera::Dynamic_Cast<RenderNode*>(node) : 0;
}


/****************************************************************************
*    Function    : GetMainSecNode
*    Description : Retrieves the render node which has all the segments.
*    Parameters  : none
*    Return      : Candera::Node2D*
****************************************************************************/
Candera::Node2D* OPSWidget2D::GetMainSecNode() const
{
   Candera::Node2D* node = GetCarNode();

   if (node != 0)
   {
      node = node->GetNextSibling();
   }

   return (node != 0) ? (node) : 0;
}


/****************************************************************************
*    Function    : GetSenErrorWarningNode
*    Description : Retrieves the sensor warning img.
*    Parameters  : none
*    Return      : RenderNode*
****************************************************************************/
Candera::Node2D* OPSWidget2D::GetSenErrorWarningNode() const
{
   Candera::Node2D* node = GetMainSecNode();

   if (node != 0)
   {
      node = node->GetFirstChild();
   }

   return (node != 0) ? (node) : 0;
}


/****************************************************************************
*    Function    : GetSenErrorWarningImg
*    Description : Retrieves the sensor warning img.
*    Parameters  : none
*    Return      : RenderNode*
****************************************************************************/
Candera::RenderNode* OPSWidget2D::GetSenErrorWarningImg(UInt8 u8Count) const
{
   Candera::Node2D* node = GetSenErrorWarningNode();

   if (node != 0)
   {
      node = node->GetFirstChild();
   }

   for (UInt8 u8Index = 1; u8Index < u8Count; u8Index++)
   {
      if (node != 0)
      {
         node = node->GetNextSibling();
      }
   }

   return (node != 0) ? Candera::Dynamic_Cast<RenderNode*>(node) : 0;
}


/****************************************************************************
*    Function    : GetSecNode
*    Description : Retrieves the sec node.
*    Parameters  : u8NodeNo
*    Return      : Candera::Node2D*
****************************************************************************/
Candera::Node2D* OPSWidget2D::GetSecNode(UInt8 u8SecNodeNo, bool bHatchedval) const
{
   Candera::Node2D* node = GetMainSecNode();

   if (node != 0)
   {
      //First child is warning Img.
      node = node->GetFirstChild();

      for (UInt8 u8Index = 0; u8Index <= u8SecNodeNo; u8Index++)
      {
         if (node != 0)
         {
            node = node->GetNextSibling();
         }
      }
   }
   if ((node != 0) && (bHatchedval == true) && enFrontAndRear == (enOPSDisplayMode)GetOPSDisplayMode())
   {
      node = node->GetFirstChild();
      for (UInt8 i = 1; i < (UInt8)enHatched; i++)
      {
         node = node->GetNextSibling();
      }
   }
   return (node != 0) ? (node) : 0;
}


/****************************************************************************
*    Function    : GetRTANode
*    Description : Retrieves the RTA node.
*    Parameters  :
*    Return      : Candera::Node2D*
****************************************************************************/
Candera::Node2D* OPSWidget2D::GetRTANode() const
{
   Candera::Node2D* node = GetMainSecNode();

   if (node != 0)
   {
      UInt8 u8ChildCnt = (UInt8) node->GetChildCount();
      //First child is warning Img.
      node = node->GetFirstChild();

      for (UInt8 u8Index = 1; u8Index < u8ChildCnt; u8Index++)
      {
         if (node != 0)
         {
            node = node->GetNextSibling();
         }
      }
   }

   return (node != 0) ? (node) : 0;
}


/****************************************************************************
*    Function    : GetRTAImgNode
*    Description : Retrieves the RTA img. node.
*    Parameters  :
*    Return      : RenderNode*
****************************************************************************/
Candera::RenderNode* OPSWidget2D::GetRTAImgNode(bool bIsRight, enRTAState enStateIn) const
{
   Candera::Node2D* node = GetRTANode();

   if (node != 0)
   {
      //First child is the Left RTA
      node = node->GetFirstChild();

      if (bIsRight)
      {
         if (node != 0)
         {
            node = node->GetNextSibling();
         }
      }

      if (node != 0)
      {
         //Get the First child of Left or Right RTA
         node = node->GetFirstChild();

         for (UInt8 u8Index = 1; u8Index < (UInt8)enStateIn; u8Index++)
         {
            if (node != 0)
            {
               node = node->GetNextSibling();
            }
         }
      }
   }

   return (node != 0) ? Candera::Dynamic_Cast<RenderNode*>(node) : 0;
}


/****************************************************************************
*    Function    : GetImageTypeNode
*    Description : Retrieves the img type node (off,active,inactive).
*    Parameters  : u8NodeNo, enMode
*    Return      : Candera::Node2D*
****************************************************************************/
Candera::Node2D* OPSWidget2D::GetImageTypeNode(UInt8 u8SecNodeNo, enImageMode enMode, bool bHatchedVal) const
{
   Candera::Node2D* node = GetSecNode(u8SecNodeNo, bHatchedVal);

   if (node != 0)
   {
      node = node->GetFirstChild();

      for (UInt8 u8Index = 1; u8Index < (UInt8)enMode; u8Index++)
      {
         if (node != 0)
         {
            node = node->GetNextSibling();
         }
      }
   }

   return (node != 0) ? (node) : 0;
}


/****************************************************************************
*    Function    : GetTrailerImgNode
*    Description : Retrieves the render node which represents the Trailer image.
*    Parameters  : none
*    Return      : RenderNode*
****************************************************************************/
Candera::RenderNode* OPSWidget2D::GetTrailerImgNode() const
{
   Candera::Node2D* node = GetCarNode();

   //Trailer node is the last node in Car Node.

   if (node != 0)
   {
      UInt8 u8NofBitmap = (UInt8)node->GetChildCount();
      node = node->GetFirstChild();

      for (UInt8 u8Index = 1; u8Index < u8NofBitmap; u8Index++)
      {
         if (node != 0)
         {
            node = node->GetNextSibling();
         }
      }
   }

   return (node != 0) ? Candera::Dynamic_Cast<RenderNode*>(node) : 0;
}


/****************************************************************************
*    Function    : GetBackgroundImgNode
*    Description : Retrieves the render node which represents the Trailer image.
*    Parameters  : u8SectorNo,enMode,bErrorImg
*    Return      : RenderNode*
****************************************************************************/
Candera::RenderNode* OPSWidget2D::GetBackgroundImgNode(UInt8 u8SectorNo,
      enImageMode enMode, bool bHatchedVal, bool bErrorImg, bool bScannedVal) const
{
   Candera::Node2D* Backgroundnode = GetImageTypeNode(u8SectorNo, enMode, bHatchedVal);

   if (Backgroundnode != 0)
   {
      Backgroundnode = Backgroundnode->GetFirstChild();
   }

   if (!bErrorImg)
   {
      if (Backgroundnode != 0)
      {
         Backgroundnode = Backgroundnode->GetNextSibling();
         if (u8SectorNo >= 8 && bScannedVal)
         {
            Backgroundnode = Backgroundnode->GetNextSibling();			//Different BG to be shown in Scanned Enabled and Disabled cases.
         }
      }
   }

   return (Backgroundnode != 0) ? Candera::Dynamic_Cast<RenderNode*>(Backgroundnode) : 0;
}


/****************************************************************************
*    Function    : GetNthSegmentNode
*    Description : Retrieves the render node of the nth segment of the given
*                  Sector.
*    Parameters  : u8SectorNo,u8SegmentNo,enImageMode
*    Return      : RenderNode*
****************************************************************************/
Candera::RenderNode* OPSWidget2D::GetNthSegmentNode(UInt8 u8SectorNo,
      UInt8 u8SegmentNo, enImageMode enMode, bool bHatchedVal) const
{
   Candera::Node2D* Secnode;
   Candera::Node2D* Segnode = NULL;
   Secnode = GetImageTypeNode(u8SectorNo, enMode, bHatchedVal);
   if (Secnode != 0)
   {
      Segnode = Secnode->GetFirstChild();
   }
   for (UInt8 u8Index = 0; u8Index < u8SegmentNo; u8Index++)
   {
      if (Segnode != 0)
      {
         Segnode = Segnode->GetNextSibling();
      }
   }
   return (Segnode != 0) ? Candera::Dynamic_Cast<RenderNode*>(Segnode) : 0;
}


//////////////////////////////END OF CONTROL FNS///////////////////////////////////

//////////////////////////////VIEW FNS/////////////////////////////////
/****************************************************************************
*    Function    : vDisplayAllImgs
*    Description : Display fun for Normal mode
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::vDisplayAllImgs()
{
   vDisplayCarImg();
   bool bTrailerConnected = GetTrailerConnected();

   if (bTrailerConnected)
   {
      vDisplayTrailerImg(false);
   }

   vDisplayFrontSensorSectorImgs();

   if (!bTrailerConnected)
   {
      vDisplayRearSensorSectorImgs();
   }

   vDisplayLeftSensorSectorImgs();
   vDisplayRightSensorSectorImgs();
}


/****************************************************************************
*    Function    : vDisplayCarImg
*    Description : Display car fun
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::vDisplayCarImg()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vDisplayCarImg"));
   Candera::RenderNode* RNode = GetNthCarImgNode();
   vSetImage(RNode, true);
}


/****************************************************************************
*    Function    : vDisplayTrailerImg
*    Description : Display Trailer fun
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::vDisplayTrailerImg(bool bDisplayFlag)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vDisplayTrailerImg bDisplayFlag = [%d]", bDisplayFlag));
   Candera::RenderNode* RNode = GetTrailerImgNode();
   vSetImage(RNode, bDisplayFlag);
}


/****************************************************************************
*    Function    : vDisplayFrontSensorSectorImgs
*    Description : Display fun for front sensor sectors
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::vDisplayFrontSensorSectorImgs()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vDisplayFrontSensorSectorImgs"));
   vDisplaySensorSectorSegImgs(enFront);
}


/****************************************************************************
*    Function    : vDisplayFnsForSectorImgs
*    Description : Display fun for sector background img
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::vDisplayFnsForSectorBgImg(Candera::UInt8 u8SecNo, bool bFlag)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vDisplayFnsForSectorBgImg"));
   Candera::Node2D* RNode = GetImageTypeNode(u8SecNo, enBackground, false);

   if (RNode != 0)
   {
      RNode = RNode->GetFirstChild();
   }

   if (RNode != 0)
   {
      RNode = RNode->GetNextSibling();

      if (RNode != 0)
      {
         vSetImage(Candera::Dynamic_Cast<RenderNode*>(RNode), bFlag);
      }
   }
}


/****************************************************************************
*    Function    : vSetSensorIndex
*    Description : fun to set the index based on display mode
*    Parameters  : enOPSDisplayMode
*    Return      : void
****************************************************************************/
void OPSWidget2D::vSetSensorIndex(enOPSDisplayMode enDisMode)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vSetSensorIndex"));

   switch (enDisMode)
   {
      case enFrontAndRear:
      {
         m_u8StartIxFrontSen = 0;
         m_u8EndIxFrontSen    = 3;
         m_u8StartIxRearSen    = 4;
         m_u8EndIxRearSen    = 7;
         m_u8StartIxLeftSen = 8;
         m_u8EndIxLeftSen = 11;
         m_u8StartIxRightSen = 12;
         m_u8EndIxRightSen = 15;
      }
      break;

      case enAround360:
      {
         m_u8StartIxFrontSen    = 0;
         m_u8EndIxFrontSen    = 3;
         m_u8StartIxRearSen    = 4;
         m_u8EndIxRearSen    = 7;
         m_u8StartIxLeftSen    = 8;
         m_u8EndIxLeftSen    = 11;
         m_u8StartIxRightSen    = 12;
         m_u8EndIxRightSen    = 15;
      }
      break;

      case enRearOnly:
      {
         m_u8StartIxRearSen = 0;
         m_u8EndIxRearSen = 7;
      }
      break;

      default:
         break;
   }
}


/****************************************************************************
*    Function    : vGetSensorType
*    Description :fun to get the sensor type
*    Parameters  : u8SecNo
*    Return      : void
****************************************************************************/
enSensor OPSWidget2D::enGetSensorType(Candera::UInt8 u8SecNo)
{
   enSensor enType = enUnknown;
   enOPSDisplayMode enDisMode = (enOPSDisplayMode)GetOPSDisplayMode();

   if (u8SecNo <= 3)
   {
      if (enDisMode == enRearOnly)
      {
         enType = enRear;
      }
      else
      {
         enType = enFront;
      }
   }
   else if (u8SecNo >= 4 && u8SecNo <= 7)
   {
      enType = enRear;
   }
   else if (u8SecNo >= 8 && u8SecNo <= 11)
   {
      enType = enLeft;
   }
   else if (u8SecNo >= 12 && u8SecNo <= 15)
   {
      enType = enRight;
   }

   return enType;
}


/****************************************************************************
*    Function    : vGetStartAndEndIndex
*    Description :fun to get the start and end index
*    Parameters  : enSensor,&u8StartIndex,&u8EndIndex
*    Return      : void
****************************************************************************/
void OPSWidget2D::vGetStartAndEndIndex(enSensor enSensorType,
                                       Candera::UInt8* u8StartIndex, Candera::UInt8* u8EndIndex,
                                       Candera::UInt8* u8WarningIndex)const
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vGetStartAndEndIndex"));
   enOPSDisplayMode enDisMode = (enOPSDisplayMode)GetOPSDisplayMode();

   switch (enSensorType)
   {
      case enFront:
      {
         *u8StartIndex = m_u8StartIxFrontSen;
         *u8EndIndex = m_u8EndIxFrontSen;
         *u8WarningIndex = 1;
      }
      break;

      case enRear:
      {
         *u8StartIndex = m_u8StartIxRearSen;
         *u8EndIndex = m_u8EndIxRearSen;

         if (enDisMode == enRearOnly)
         {
            *u8WarningIndex = 1;
         }
         else
         {
            *u8WarningIndex = 2;
         }
      }
      break;

      case enLeft:
      {
         *u8StartIndex = m_u8StartIxLeftSen;
         *u8EndIndex = m_u8EndIxLeftSen;
         *u8WarningIndex = 3;
      }
      break;

      case enRight:
      {
         *u8StartIndex = m_u8StartIxRightSen;
         *u8EndIndex = m_u8EndIxRightSen;
         *u8WarningIndex = 4;
      }
      break;

      default:
         break;
   }
}


/****************************************************************************
*    Function    : vRemoveSectorNodeImgs
*    Description : fun to remove images on a Node
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::vRemoveSectorNodeImgs(UInt8 u8Node)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vRemoveSectorNodeImgs"));
   Candera::Node2D* node = GetSecNode(u8Node, m_u8MapLastDisplayedSegInfo[u8Node].prevHatchedVal);

   if (node != 0)
   {
      UInt8 u8Cnt = (UInt8)node->GetChildCount();

      if (u8Cnt)
      {
         node = node->GetFirstChild();

         for (UInt8 u8Index = 0; u8Index < u8Cnt; u8Index++)
         {
            if (node != 0)
            {
               vRemoveAllNodeAssociatedImgs(node);
               if ((u8Index + 1) == (UInt8)enHatched)
               {
                  vRemoveHatchedNodeAssociatedImgs(node);
               }
               node = node->GetNextSibling();
            }
         }
      }
   }
}


/****************************************************************************
*    Function    : vRemoveAllRearSensorImgs
*    Description : fun to remove images on rear sensor
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::vRemoveAllRearSensorImgs()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vRemoveAllRearSensorImgs"));
   UInt8 u8SxI = 0, u8ExI = 0, u8WIx = 0;
   vGetStartAndEndIndex(enRear, &u8SxI, &u8ExI, &u8WIx);

   for (UInt8 u8Index = u8SxI; u8Index <= u8ExI; u8Index++)
   {
      vRemoveSectorNodeImgs(u8Index);
   }

   RenderNode* node = GetSenErrorWarningImg(u8WIx);

   if (node != 0)
   {
      vSetImage(node, false);
   }
}


/****************************************************************************
*    Function    : vSubRemoveAllImgs
*    Description : Sub fun to remove images on a sector
*    Parameters  : Candera::Node2D*
*    Return      : void
****************************************************************************/
void OPSWidget2D::vSubRemoveAllImgs(Candera::Node2D* Lnode)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vSubRemoveAllImgs"));
   if (Lnode != 0)
   {
      UInt8 u8LCnt = (UInt8) Lnode->GetChildCount();
      //First child of the Car node should be car images.
      Lnode = Lnode->GetFirstChild();

      for (UInt8 u8Index = 0; u8Index < u8LCnt; u8Index++)
      {
         if (Lnode != 0)
         {
            vSetImage(Candera::Dynamic_Cast<RenderNode*>(Lnode), false);
            UInt8 u8LLCnt = (UInt8) Lnode->GetChildCount();
            Candera::Node2D* LLnode = Lnode->GetFirstChild();

            for (UInt8 u8I = 1; u8I <= u8LLCnt; u8I++)
            {
               if (LLnode != 0)
               {
                  vSetImage(Candera::Dynamic_Cast<RenderNode*>(LLnode), false);
                  vRemoveAllNodeAssociatedImgs(LLnode);
                  if (u8I == (UInt8)enHatched)
                  {
                     vRemoveHatchedNodeAssociatedImgs(LLnode);
                  }
                  LLnode = LLnode->GetNextSibling();
               }
            }

            vRemoveSectorNodeImgs(u8Index);
            Lnode = Lnode->GetNextSibling();
         }
      }
   }
}


/****************************************************************************
*    Function    : vRemoveAllNodeAssociatedImgs
*    Description : fun to remove images on a sector
*    Parameters  : Candera::Node2D*
*    Return      : void
****************************************************************************/
void OPSWidget2D::vRemoveAllImgs()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vRemoveAllImgs"));
   //Get the Main node
   Candera::Node2D* node = GetNode();

   if (node != 0)
   {
      //First child should be 4ch or 8ch or 360 mode node
      node = node->GetFirstChild();

      for (UInt8 u8Index = 1; u8Index < (UInt8)m_enPreviousDismode; u8Index++)
      {
         if (node != 0)
         {
            //Based on the mode migrate
            node = node->GetNextSibling();
         }
      }

      //if (m_enPreviousDismode != enRearOnly)
      //{
      //   if (node != 0)
      //   {
      //select the Parent node 8ch or 16ch
      //      node = node->GetFirstChild();
      //   }
      //}

      if (node != 0)
      {
         UInt8 u8Cnt = (UInt8) node->GetChildCount();
         //First child of the mode node should be car images node
         node = node->GetFirstChild();

         for (UInt8 u8Index = 0; u8Index < u8Cnt; u8Index++)
         {
            Candera::Node2D* Lnode = node;

            if (Lnode != 0)
            {
               //If any image is attached to the car node remove it
               vSetImage(Candera::Dynamic_Cast<RenderNode*>(Lnode), false);
               vSubRemoveAllImgs(Lnode);
            }

            node = node->GetNextSibling();
         }
      }
   }
}


/****************************************************************************
*    Function    : vRemoveHatchedNodeAssociatedImgs
*    Description : fun to remove images on a sector
*    Parameters  : Candera::Node2D*
*    Return      : void
****************************************************************************/

void OPSWidget2D::vRemoveHatchedNodeAssociatedImgs(Candera::Node2D* pNode)
{
   if (pNode != 0)
   {
      UInt8 u8LLCnt = (UInt8)pNode->GetChildCount();
      Candera::Node2D* LLnode = pNode->GetFirstChild();
      for (UInt8 u8I = 1; u8I <= u8LLCnt; u8I++)
      {
         if (LLnode != 0)
         {
            vSetImage(Candera::Dynamic_Cast<RenderNode*>(LLnode), false);
            vRemoveAllNodeAssociatedImgs(LLnode);
            LLnode = LLnode->GetNextSibling();
         }
      }
   }
}


/****************************************************************************
*    Function    : vRemoveAllNodeAssociatedImgs
*    Description : fun to remove images on a sector
*    Parameters  : Candera::Node2D*
*    Return      : void
****************************************************************************/
void OPSWidget2D::vRemoveAllNodeAssociatedImgs(Candera::Node2D* pNode)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vRemoveAllNodeAssociatedImgs"));
   Candera::Node2D* node = NULL;

   if (pNode != 0)
   {
      UInt8 u8NodeCnt = (UInt8)pNode->GetChildCount();
      node = pNode->GetFirstChild();

      for (UInt8 u8Index = 0 ; u8Index < u8NodeCnt ; u8Index++)
      {
         if (node != 0)
         {
            if (_animationHelperInterface != 0)
            {
               if (_animationHelperInterface->isAnimationEnabledOnNode(Candera::Dynamic_Cast<RenderNode*>(node)) == true)
               {
                  _animationHelperInterface->stopAnimation(Candera::Dynamic_Cast<RenderNode*>(node));
               }
            }
            vSetImage(Candera::Dynamic_Cast<RenderNode*>(node), false);
            node = node->GetNextSibling();
         }
      }
   }
}


/****************************************************************************
*    Function    : vDisplayRearSensorSectorImgs
*    Description : Display fun for rear sensor sectors
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::vDisplayRearSensorSectorImgs()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vDisplayRearSensorSectorImgs"));

   if (!GetTrailerConnected())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vDisplayRearSensorSectorImgs if "));
      vDisplayTrailerImg(false);
      vDisplaySensorSectorSegImgs(enRear);
   }
   else
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vDisplayRearSensorSectorImgs else"));
      vRemoveAllRearSensorImgs();
      vDisplayTrailerImg(true);
   }
}


/****************************************************************************
*    Function    : vDisplayLeftSensorSectorImgs
*    Description : Display fun for left sensor sectors
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::vDisplayLeftSensorSectorImgs()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vDisplayLeftSensorSectorImGetSpeedThresholdExceededgs"));
   vDisplaySensorSectorSegImgs(enLeft);
}


/****************************************************************************
*    Function    : vDisplayRightSensorSectorImgs
*    Description : Display fun for right sensor sectors
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::vDisplayRightSensorSectorImgs()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vDisplayRightSensorSectorImgs"));
   vDisplaySensorSectorSegImgs(enRight);
}


/****************************************************************************
*    Function    : vDisplayBackgroundImg
*    Description : Display fun for right sensor sectors
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::vDisplayBackgroundImg(Candera::UInt8 u8CurrentSec, bool bHatchedVal, bool bDisplayflag, bool bScannedVal)
{
   RenderNode* node = GetBackgroundImgNode(u8CurrentSec, enBackground, bHatchedVal, false, bScannedVal);

   if (node != 0)
   {
      vSetImage(node, bDisplayflag);
   }
}


/****************************************************************************
*    Function    : vDisplayBackgroundErrImg
*    Description : Display fun for right sensor sectors
*    Parameters  :
*    Return      : void
****************************************************************************/
void OPSWidget2D::vDisplayBackgroundErrImg(Candera::UInt8 u8CurrentSec, bool bDisplayflag)
{
   RenderNode* node = GetBackgroundImgNode(u8CurrentSec, enBackground, false, true, false);

   if (node != 0)
   {
      vSetImage(node, bDisplayflag);
   }
}


/****************************************************************************
*    Function    : vRemoveSensorAllSectorSegImgsAndDisplayWarning
*    Description : Display fun for rear sensor sector segments
*    Parameters  : enSensorType
*    Return      : void
****************************************************************************/
void OPSWidget2D::vRemoveSensorAllSectorSegImgsAndDisplayWarning(Candera::UInt8 u8StartIndex, Candera::UInt8 u8EndIndex,
      Candera::UInt8 u8WarningIndex)
{
   for (UInt8 u8SecIndex = u8StartIndex; u8SecIndex <= u8EndIndex; u8SecIndex++)
   {
      vRemoveSectorNodeImgs(u8SecIndex);
      vDisplayBackgroundErrImg(u8SecIndex, true);
   }

   RenderNode* node = GetSenErrorWarningImg(u8WarningIndex);

   if (node != 0)
   {
      vSetImage(node, true);
   }
}


/****************************************************************************
*    Function    : vDisplayRearSensorSectorSegImgs
*    Description : Display fun for rear sensor sector segments
*    Parameters  : enSensorType
*    Return      : void
****************************************************************************/
void OPSWidget2D::vDisplaySensorSectorSegImgs(enSensor enSensorType)
{
   UInt8 u8SecCnt = 0;
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vDisplaySensorSectorSegImgs"));
   UInt8 u8StartIx = 0, u8EndIx = 0, u8WarningIx = 0;
   vGetStartAndEndIndex(enSensorType, &u8StartIx, &u8EndIx, &u8WarningIx);

   for (UInt8 u8SecIndex = u8StartIx; u8SecIndex <= u8EndIx; u8SecIndex++)
   {
      m_u8CurrentSector = u8SecIndex;
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : m_u8CurrentSector %d, Size %d", m_u8CurrentSector, m_lstbSensorData.Size()));

      if (!m_lstbSensorData.Empty() && u8SecIndex < m_lstbSensorData.Size())
      {
         bool bIsSecAvailable = m_lstbSensorData[u8SecIndex];

         if (!bIsSecAvailable)
         {
            u8SecCnt++;
         }

         vDisplaySegmentImg();
      }
   }

   enOPSWidgetType enType = GetOPSWidgetType();

   if (enType == enBySegment)
   {
      if (u8SecCnt > 4)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vDisplaySensorSectorSegImgs u8SecCnt = 4"));
         vRemoveSensorAllSectorSegImgsAndDisplayWarning(u8StartIx, u8EndIx, u8WarningIx);
      }
      else
      {
         RenderNode* node = GetSenErrorWarningImg(u8WarningIx);

         if (node != 0)
         {
            vSetImage(node, false);
         }
      }
   }
}


/****************************************************************************
*    Function    : vGetActiveSegmentIndex
*    Description : fun for specific segment distance comparision
*    Parameters  : u8TotalAvailPreValue,Candera::UInt8*,Candera::UInt8*
*    Return      : void
****************************************************************************/
void OPSWidget2D::vGetActiveSegmentIndex(Candera::UInt8 u8ActVal, Candera::UInt8 u8TotalAvailPreValue,
      bool* bSegmentActive, Candera::UInt8* u8ActiveSegIndex)
{
   enOPSWidgetType enType = GetOPSWidgetType();
   UInt8 u8PreVal = 0;
   enImageMode enActiveColor = enUnknownMode;
   UInt8 u8SecNo = m_u8CurrentSector;
   bool bScannedVal = true;
   if (u8SecNo)
   {
      bScannedVal = m_lstbSensorScannedData[u8SecNo - 8];
   }
   if (!m_lstu16PreValData.Empty() && u8TotalAvailPreValue < m_lstu16PreValData.Size())
   {
      for (UInt8 u8Index = 0; u8Index < u8TotalAvailPreValue; u8Index++)
      {
         u8PreVal = (UInt8)m_lstu16PreValData[u8Index];
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : PredefValue @%d=%d", u8Index, u8PreVal));

         if (u8PreVal && m_lstbSensorData[u8SecNo])
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : make SegmentFinalImg active @Sec=%d,@Seg=%d", u8SecNo, u8Index));

            if (u8ActVal == 0)
            {
               vDisplaySegmentFinalImg(u8Index, enUnknownMode);
               vDisplayBackgroundImg(u8SecNo, m_lstbSensorHatchedData[u8SecNo], true, bScannedVal);
            }
            else if (u8ActVal <= u8PreVal /*&& u8ActVal != 0*/)
            {
               *u8ActiveSegIndex = u8Index;

               if (enType == enBySegment)
               {
                  vGetActiveColorPQ(u8Index, &enActiveColor);
               }
               else
               {
                  vGetActiveColor(u8SecNo, &enActiveColor);
               }

               vDisplaySegmentFinalImg(u8Index, enActiveColor);
               vDisplayBackgroundImg(u8SecNo, m_lstbSensorHatchedData[u8SecNo], true, bScannedVal);
               *bSegmentActive = true;
               break;
            }
         }
      }
   }
}


/****************************************************************************
*    Function    : vGetActiveColorPQ
*    Description : Based on distance the status value color is selected
*    Parameters  : enImageMode*
*    Return      : void
****************************************************************************/
void OPSWidget2D::vGetActiveColorPQ(Candera::UInt8 u8IxNo, enImageMode* enActiveColor)
{
   if (u8IxNo < 1)
   {
      *enActiveColor = enActiveRed;
   }
   else if (u8IxNo < 2)
   {
      *enActiveColor = enActiveYellow;
   }
   else
   {
      *enActiveColor = enGrey;
   }
}


/****************************************************************************
*    Function    : vGetActiveColor
*    Description : Based on distance the status value color is selected
*    Parameters  : enImageMode*
*    Return      : void
****************************************************************************/
void OPSWidget2D::vGetActiveColor(Candera::UInt8 u8SecNo, enImageMode* enActiveColor)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vGetActiveColor @Sec=%d", u8SecNo));

   if (!m_lstu16StatusData.Empty() && u8SecNo < m_lstu16StatusData.Size())
   {
      UInt8 u8StatusValue = (UInt8)m_lstu16StatusData[u8SecNo];

      switch (u8StatusValue)
      {
         case  enNoWarning:
            *enActiveColor = enBackground;
            break;

         case enLowWarning:
            *enActiveColor = enGrey;
            break;

         case enMediumWarning:
            *enActiveColor = enActiveYellow;
            break;

         case enHardWarning:
            *enActiveColor = enActiveRed;
            break;

         case enNotAvail:
            *enActiveColor = enNotAvailable;
            break;

         case enNotInit:
            *enActiveColor = enNotInitilized;
            break;

         case enError:
            *enActiveColor = enWarning;
            break;

         default:
            break;
      }
   }
}


/****************************************************************************
*    Function    : vDisplayActiveSegments
*    Description : Based on distance the segment is displayed on this fun
*    Parameters  : u8ActVal,u8TotalAvailPreValue,bSegmentActive,u8ActiveSegIndex
*    Return      : void
****************************************************************************/
void OPSWidget2D::vDisplayActiveSegments(Candera::UInt8 u8SecNo, Candera::UInt8 u8TotalAvailPreValue,
      bool bSegmentActive, Candera::UInt8 u8ActiveSegIndex)
{
   enOPSWidgetType enType = GetOPSWidgetType();
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vDisplayActiveSegments->u8SecNo=%d,bSegmentActive=%d,u8ActiveSegIndex=%d,enWidgetType=%d",
                       u8SecNo, bSegmentActive, u8ActiveSegIndex, enType));

   if (bSegmentActive)
   {
      if (enType == enBySegment)
      {
         for (UInt8 u8Index = 0; u8Index < u8TotalAvailPreValue; u8Index++)
         {
            if (u8Index != u8ActiveSegIndex)
            {
               vDisplaySegmentFinalImg(u8Index, enGrey);
            }
         }
      }

      /*else
      {
         enImageMode enActiveColor = enGrey;
         if(u8ActiveSegIndex < 2)
         {
            enActiveColor = enActiveRed;
         }
         else
         {
            enActiveColor = enActiveYellow;
         }
         for(UInt8 u8Index=0;u8Index<u8TotalAvailPreValue;u8Index++)
         {
            vDisplaySegmentFinalImg(u8Index,enGrey);
            vDisplaySegmentFinalImg(u8Index,enActiveColor);
         }
      }*/
   }
   else
   {
      for (UInt8 u8Index = 0; u8Index < u8TotalAvailPreValue; u8Index++)
      {
         vDisplaySegmentFinalImg(u8Index, enGrey);
      }
   }
}


/****************************************************************************
*    Function    : vSegmentValComparefns
*    Description : fun for specific segment distance comparision
*    Parameters  : u8ActVal,u8SecNo
*    Return      : void
****************************************************************************/
void OPSWidget2D::vSegmentValComparefns(Candera::UInt8 u8ActVal, Candera::UInt8 u8SecNo)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vSegmentValComparefns"));
   UInt8 u8TotalAvailPreValue = (UInt8)m_lstu16PreValData.Size();
   UInt8 u8ActiveSegIndex = 0;
   bool bSegmentActive = false;
   Candera::Node2D* node = GetImageTypeNode(u8SecNo, enActiveRed, (bool)m_lstbSensorHatchedData[u8SecNo]);

   if (node != 0)
   {
      UInt8 u8ActualSeg = (UInt8) node->GetChildCount();
      u8TotalAvailPreValue = u8ActualSeg;
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : u8TotalAvailPreValue %d", u8TotalAvailPreValue));
   vGetActiveSegmentIndex(u8ActVal, u8TotalAvailPreValue, &bSegmentActive, &u8ActiveSegIndex);
   //vDisplayActiveSegments(u8SecNo,u8TotalAvailPreValue,bSegmentActive,u8ActiveSegIndex);
}


/****************************************************************************
*    Function    : vDisplaySegmentImg
*    Description : Display fun for specific segment
*    Parameters  : none
*    Return      : void
****************************************************************************/
void OPSWidget2D::vDisplaySegmentImg()
{
   UInt8 u8CurrentSec = m_u8CurrentSector;
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vDisplaySegmentImg [%d], Segsize %d", u8CurrentSec, m_lstu16SegValData.Size()));

   if (!m_lstu16SegValData.Empty() && u8CurrentSec < m_lstu16SegValData.Size())
   {
      UInt8 u8ActVal = (UInt8)m_lstu16SegValData[u8CurrentSec];
      //if(u8ActVal>=0)
      {
         vSegmentValComparefns(u8ActVal, u8CurrentSec);
      }
   }
}


/****************************************************************************
*    Function    : vDisplayFnsforRTA
*    Description : Display fun for specific segment
*    Parameters  : none
*    Return      : void
****************************************************************************/
void OPSWidget2D::vDisplayFnsforRTA(Candera::UInt8 u8State, Candera::UInt8 u8Distance,
                                    bool bIsRightSide)
{
   enRTAState enActiveState = enRTAOff, enDeactivate1 = enRTAOff, enDeactivate2 = enRTAOff;

   if ((u8State == 0x0) || (u8Distance == 0xF))
   {
      enActiveState = enRTAOff;
      enDeactivate1 = enRTAOff;
      enDeactivate2 = enRTAOff;
   }
   else if ((u8State == 0x1) && (u8Distance == 0x1))
   {
      enActiveState = enRTAGrey;
      enDeactivate1 = enRTAYellow;
      enDeactivate2 = enRTARed;
   }
   else if ((u8State == 0x2) && (u8Distance == 0x2))
   {
      enActiveState = enRTAYellow;
      enDeactivate1 = enRTAGrey;
      enDeactivate2 = enRTARed;
   }
   else if ((u8State == 0x3) && (u8Distance == 0x3))
   {
      enActiveState = enRTARed;
      enDeactivate1 = enRTAGrey;
      enDeactivate2 = enRTAYellow;
   }

   vDisplayRTAImg(enActiveState, enDeactivate1, enDeactivate2, bIsRightSide);
}


/****************************************************************************
*    Function    : vDisplayRTAImg
*    Description : Display fun for specific segment
*    Parameters  : none
*    Return      : void
****************************************************************************/
void OPSWidget2D::vDisplayRTAImg(enRTAState enActiveState,
                                 enRTAState enDeactivate1, enRTAState enDeactivate2, bool bIsRightRTA)
{
   if (enDeactivate1 == enRTAOff)
   {
      Candera::RenderNode* Rnode = GetRTAImgNode(bIsRightRTA, enRTAGrey);
      vSetImage(Rnode, false);
      Rnode = GetRTAImgNode(bIsRightRTA, enRTAYellow);
      vSetImage(Rnode, false);
      Rnode = GetRTAImgNode(bIsRightRTA, enRTARed);
      vSetImage(Rnode, false);
   }
   else
   {
      Candera::RenderNode* Rnode = GetRTAImgNode(bIsRightRTA, enDeactivate1);
      vSetImage(Rnode, false);
      Rnode = GetRTAImgNode(bIsRightRTA, enDeactivate2);
      vSetImage(Rnode, false);
      Rnode = GetRTAImgNode(bIsRightRTA, enActiveState);
      vSetImage(Rnode, true);
   }
}


/****************************************************************************
*    Function    : vRemoveAlreadyDisplayedImg
*    Description : fun to remove displayed imgs on a sec
*    Parameters  : u8SenNo,u8SecNo,u8SegNo,bool:true->Active Img,false->Inact Img
*    Return      : void
****************************************************************************/
void OPSWidget2D::vRemoveAlreadyDisplayedImg(Candera::UInt8 u8SecNo,
      Candera::UInt8 u8SegNo, enImageMode enMode, bool bPrevHatchedVal)
{
   Candera::RenderNode* RSegInactiveNode = GetNthSegmentNode(u8SecNo, u8SegNo, enMode, bPrevHatchedVal);
   if (RSegInactiveNode != 0)
   {
      UInt8 u8ChildCount = static_cast<UInt8>(RSegInactiveNode->GetParent()->GetChildCount());
      for (; u8SegNo < u8ChildCount; u8SegNo++)
      {
         vGetActiveColorPQ(u8SegNo, &enMode);
         RSegInactiveNode = GetNthSegmentNode(u8SecNo, u8SegNo, enMode, bPrevHatchedVal);
         vSetImage(RSegInactiveNode, false);
         //Here we have to stop the animation
         if (_animationHelperInterface != 0)
         {
            if (_animationHelperInterface->isAnimationEnabledOnNode(RSegInactiveNode) == true)
            {
               _animationHelperInterface->stopAnimation(RSegInactiveNode);
            }
         }
      }
   }
}


/****************************************************************************
*    Function    : vGetPreviousSecDetails
*    Description : fun to get the previous active sector details
*    Parameters  : u8SenNo,u8SecNo,u8SegNo,bool:true->Active Img,false->Inact Img
*    Return      : void
****************************************************************************/
void OPSWidget2D::vGetPreviousSecDetails(Candera::UInt8* u8PreviousActSeg, enImageMode* enPreviousActMode, bool* bSegHatchedInfo, bool* bSegScannedInfo)
{
   UInt8 u8SecNo = m_u8CurrentSector;

   if (m_u8MapLastDisplayedSegInfo.find(u8SecNo) != m_u8MapLastDisplayedSegInfo.end())
   {
      if (m_u8MapLastDisplayedSegInfo[u8SecNo].prevSegVal.find(0) != m_u8MapLastDisplayedSegInfo[u8SecNo].prevSegVal.end())
      {
         *u8PreviousActSeg = m_u8MapLastDisplayedSegInfo[u8SecNo].prevSegVal[(UInt8)0];
      }

      if (m_u8MapLastDisplayedSegInfo[u8SecNo].prevSegVal.find(1) != m_u8MapLastDisplayedSegInfo[u8SecNo].prevSegVal.end())
      {
         *enPreviousActMode = (enImageMode)m_u8MapLastDisplayedSegInfo[u8SecNo].prevSegVal[(UInt8)1];
      }
      if (m_u8MapLastDisplayedSegInfo[u8SecNo].prevHatchedVal != *bSegHatchedInfo)
      {
         *bSegHatchedInfo = m_u8MapLastDisplayedSegInfo[u8SecNo].prevHatchedVal;
      }
      if (m_u8MapLastDisplayedSegInfo[u8SecNo].prevScannedVal != *bSegScannedInfo)
      {
         *bSegScannedInfo = m_u8MapLastDisplayedSegInfo[u8SecNo].prevScannedVal;
      }
   }
}


/****************************************************************************
*    Function    : vRemovePreviousActImgs
*    Description : Removal fun for Segment's previous active img
*    Parameters  : u8WarningIx
*    Return      : void
****************************************************************************/
void OPSWidget2D::vRemovePreviousActImgs(Candera::UInt8 u8WarningIx)
{
   UInt8 u8SecNo = m_u8CurrentSector;
   UInt8 u8PreviousActSeg = 0;
   enImageMode enPreviousActMode = enUnknownMode;
   bool bPrevHatchedVal = false;
   bool bPrevScannedVal = false;
   vGetPreviousSecDetails(&u8PreviousActSeg, &enPreviousActMode, &bPrevHatchedVal, &bPrevScannedVal);

   switch (enPreviousActMode)
   {
      case enUnknownMode:
         break;

      case enActiveRed:
      case enActiveYellow:
      case enGrey:
         vDisplayBackgroundImg(u8SecNo, bPrevHatchedVal, false, bPrevScannedVal);
         vRemoveAlreadyDisplayedImg(u8SecNo, u8PreviousActSeg, enPreviousActMode, bPrevHatchedVal);
         break;

      case enBackground:
         vDisplayBackgroundImg(u8SecNo, bPrevHatchedVal, false, bPrevScannedVal);
         break;

      case enNotAvailable:
         break;

      case enNotInitilized:
         break;

      case enWarning:
      {
         RenderNode* node = GetSenErrorWarningImg(u8WarningIx);

         if (node != 0)
         {
            vSetImage(node, false);
         }
      }
      break;

      default:
         break;
   }
}


/****************************************************************************
*    Function    : vDisplayCurrentActImgs
*    Description : Display fun for Segment's current active imgs
*    Parameters  : u8SegNo,enImageMode
*    Return      : void
****************************************************************************/
void OPSWidget2D::vDisplayCurrentActImgs(Candera::UInt8 u8SegNo, enImageMode enMode,
      Candera::UInt8 u8StartIndex, Candera::UInt8 u8EndIndex, Candera::UInt8 u8WarningIndex)
{
   UInt8 u8SecNo = m_u8CurrentSector;
   bool bScannedVal = true;
   if (u8SecNo)
   {
      bScannedVal = (bool)m_lstbSensorScannedData[u8SecNo - 8];
   }
   enOPSWidgetType enType = GetOPSWidgetType();
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : vDisplaySegmentFinalImg @sec=%d,Seg=%d,enMode=%d,u8StartIx=%d,u8EndIx=%d",
                       u8SecNo, u8SegNo, enMode, u8StartIndex, u8EndIndex));

   if (enType == enBySector)
   {
      for (UInt8 u8SecIndex = u8StartIndex; u8SecIndex <= u8EndIndex; u8SecIndex++)
      {
         if (enError == (enStatusValue)m_lstu16StatusData[u8SecIndex])
         {
            enMode = enWarning;
            break;
         }
      }
   }

   switch (enMode)
   {
      case enUnknownMode:
         break;

      case enActiveRed:
      case enActiveYellow:
      case enGrey:
      {
         vDisplayBackgroundImg(u8SecNo, (bool)m_lstbSensorHatchedData[u8SecNo], true, bScannedVal);
         Candera::RenderNode* RSegNode = GetNthSegmentNode(u8SecNo, u8SegNo, enMode, (bool)m_lstbSensorHatchedData[u8SecNo]);
         if (RSegNode != 0)
         {
            UInt8 u8ChildCount = static_cast<UInt8>(RSegNode->GetParent()->GetChildCount());
            vDisplayAllFromActiveSegments(u8SecNo, u8SegNo, u8ChildCount, enMode);
         }
      }
      break;

      case enBackground:
         vDisplayBackgroundImg(u8SecNo, (bool)m_lstbSensorHatchedData[u8SecNo], true, bScannedVal);
         break;

      case enNotAvailable:
         break;

      case enNotInitilized:
         vDisplayBackgroundImg(u8SecNo, (bool)m_lstbSensorHatchedData[u8SecNo], false, bScannedVal);
         break;

      case enWarning:
         vRemoveSensorAllSectorSegImgsAndDisplayWarning(u8StartIndex, u8EndIndex, u8WarningIndex);
         break;

      default:
         break;
   }
}


/****************************************************************************
*    Function    : vDisplayAllActiveSegments
*    Description : Display all Active Segment Images
*    Parameters  : u8SecNo,u8SegNo,u8ChildCount,enImageMode
*    Return      : void
****************************************************************************/
void OPSWidget2D::vDisplayAllFromActiveSegments(Candera::UInt8 u8SecNo, Candera::UInt8 u8SegNo, Candera::UInt8 u8ChildCount, enImageMode enMode)
{
   Candera::RenderNode* RSegNode = NULL;
   for (; u8SegNo < u8ChildCount; u8SegNo++)
   {
      vGetActiveColorPQ(u8SegNo, &enMode);
      RSegNode = GetNthSegmentNode(u8SecNo, u8SegNo, enMode, (bool)m_lstbSensorHatchedData[u8SecNo]);
      if (RSegNode != 0)
      {
         if (m_lstbSensorData[u8SecNo])
         {
            if (u8SecNo >= 8)	//For Flank
            {
               if (m_lstbSensorScannedData[u8SecNo - 8])
               {
                  vSetImage(RSegNode, true);
               }
               else
               {
                  vSetImage(RSegNode, false);
               }
            }
            else
            {
               vSetImage(RSegNode, true);
            }
            if (m_lstbSensorBlinkData[u8SecNo])
            {
               vHandleBlinkAnimationOfNode(RSegNode, u8SecNo, enMode);
            }
         }
         else
         {
            vDisplayBackgroundImg(u8SecNo, m_u8MapLastDisplayedSegInfo[u8SecNo].prevHatchedVal, false, m_u8MapLastDisplayedSegInfo[u8SecNo].prevScannedVal);
            vSetImage(RSegNode, false);
         }
      }
   }
}


/****************************************************************************
*    Function    : vDisplaySegmentFinalImg
*    Description : Display fun for Segment's final img
*    Parameters  : u8SegNo,enImageMode
*    Return      : void
****************************************************************************/
void OPSWidget2D::vDisplaySegmentFinalImg(Candera::UInt8 u8SegNo, enImageMode enMode)
{
   UInt8 u8SecNo = m_u8CurrentSector;
   UInt8 u8StartIx = 0, u8EndIx = 0, u8WarningIx = 0;
   enSensor enSensorType = enGetSensorType(u8SecNo);
   vGetStartAndEndIndex(enSensorType, &u8StartIx, &u8EndIx, &u8WarningIx);
   ETG_TRACE_ERR(("OPSWidget2D : vDisplaySegmentFinalImg @sec=%d,Seg=%d,enMode=%d,u8StartIx=%d,u8EndIx=%d\n",
                  u8SecNo, u8SegNo, enMode, u8StartIx, u8EndIx));
   vRemovePreviousActImgs(u8WarningIx);
   vDisplayCurrentActImgs(u8SegNo, enMode, u8StartIx, u8EndIx, u8WarningIx);
   m_u8MapLastDisplayedSegInfo[u8SecNo].prevSegVal[(UInt8)0] = u8SegNo;
   m_u8MapLastDisplayedSegInfo[u8SecNo].prevSegVal[(UInt8)1] = (UInt8) enMode;
   m_u8MapLastDisplayedSegInfo[u8SecNo].prevHatchedVal = (bool)m_lstbSensorHatchedData[u8SecNo];
   if (u8SecNo >= 8)
   {
      m_u8MapLastDisplayedSegInfo[u8SecNo].prevScannedVal = (bool)m_lstbSensorScannedData[u8SecNo - 8];
   }
}


void OPSWidget2D::vHandleSensorFieldBlink()
{
   SensorFieldBlinkingType SensorFieldBlinkingList = GetSensorFieldBlinking();
   UInt32 u32Cnt = static_cast<UInt32>(SensorFieldBlinkingList.Count());
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : u32Cnt = %d", u32Cnt));
   SensorFieldBlinkingList.SetEventCallback(&_listEventHandler, &ListEventHandler::OnSensorFieldBlinkListEvent);
   SensorFieldBlinkingList.Request(0, u32Cnt);
}


/****************************************************************************
*    Function    : vOnSensorFieldBlinkListEvent
*    Description : Call back for sensor field blink
*    Parameters  : const Courier::ListEvent &
*    Return      : void
****************************************************************************/
void OPSWidget2D::ListEventHandler::OnSensorFieldBlinkListEvent(const Courier::ListEvent& SensorListEvent)
{
   if (m_pWidget == 0)
   {
      return;
   }

   bool l_bSensorData = false;
   const bool* l_pSensorData = SensorListEvent.GetItem<bool>();

   if (l_pSensorData != 0)
   {
      l_bSensorData = *l_pSensorData;
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D:vOnSensorFieldBlinkListEvent [%19s] received %d: newIndex=%u, oldIndex=%u, l_bSensorData=%d ",
                       m_pWidget->GetLegacyName(),
                       SensorListEvent.EventType(),
                       SensorListEvent.NewIndex(),
                       SensorListEvent.OldIndex(),
                       l_bSensorData
                      ));
   bool l_bResult = true;

   switch (SensorListEvent.EventType())
   {
      // list item delivery bay - the order requested items come in is not specified
      case Courier::ListEventType::RequestedItem:
      {
         //m_lstbSensorData.Insert(((UInt16)SensorListEvent.NewIndex()), (bool)l_bSensorData);
         m_pWidget->vAddNewSensorFieldBlinkData(m_pWidget->m_lstbSensorBlinkData, (UInt16)SensorListEvent.NewIndex(), l_bSensorData);
         //ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(),"OPSWidget2D : m_lstbSensorData.Insert & size %d", m_lstbSensorData.Size()));
      }
      break;

      // change & add event handling
      case Courier::ListEventType::ModifiedItem:
      case Courier::ListEventType::AddedItem:
      {
         m_pWidget->vAddNewSensorFieldBlinkData(m_pWidget->m_lstbSensorBlinkData, (UInt16)SensorListEvent.NewIndex(), l_bSensorData);

         // insert item to local list
         if (!SensorListEvent.HasDataItem())
         {
            // if the event does not carry the modified data item, create a request for it
            l_bResult = m_pWidget->m_lstbSensorBlinkProperty.Request(SensorListEvent.NewIndex(), 1);
         }
      }
      break;

      case Courier::ListEventType::RefreshList:
      case Courier::ListEventType::ItemCountChanged:
      {
         l_bResult = m_pWidget->m_lstbSensorBlinkProperty.Request(0, m_pWidget->m_lstbSensorsAvailProperty.Count());
      }
      break;

      default:
      {
      }
      break;
   }

   if (!l_bResult)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: [%s]. OnSensorsAvaila l_bResult = false", m_pWidget->GetLegacyName()));
   }
}


/****************************************************************************
*    Function    : vAddNewSensorFieldBlinkData
*    Description : Add function for the new sensor field blink data
*    Parameters  : u16Index, bool DataEntry
*    Return      : bool
****************************************************************************/
void OPSWidget2D::vAddNewSensorFieldBlinkData(FeatStd::Internal::Vector<bool>& Segdata, FeatStd::UInt16 u16Index, const bool& DataEntry)
{
   bool l_bIsDataStored = true;
   Candera::UInt32 l_NumberOfItemsInList = static_cast<UInt32>(Segdata.Size());
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: vAddNewSensorFieldBlinkIntervalData[%19s] - l_NumberOfItemsInList = [%d]u16Index = [%d], and DataEntry = [%d]",
                       GetLegacyName(), l_NumberOfItemsInList, u16Index, DataEntry));
   if (l_NumberOfItemsInList <= u16Index)
   {
      Candera::UInt32 l_Index = l_NumberOfItemsInList;
      for (; l_Index < u16Index; ++l_Index)
      {
         l_bIsDataStored = Segdata.Insert(static_cast<Int>(l_Index), DataEntry);
      }
      l_bIsDataStored = (l_bIsDataStored == true) ? Segdata.Insert(u16Index, DataEntry) : false;
   }
   else
   {
      Segdata[u16Index] = DataEntry;
   }
   if (!m_lstbSensorData.Empty() && u16Index < m_lstbSensorData.Size())
   {
      UInt32 l_NewSize = static_cast<UInt32>(Segdata.Size());
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: l_NewSize = [%d]", l_NewSize));
      if (l_NewSize)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: Statusdata[%d] = [%d]", u16Index, Segdata[u16Index]));
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: m_lstbSensorData[%d] = [%d]", u16Index, m_lstbSensorData[u16Index]));
      }
      m_u8CurrentSector = (UInt8)u16Index;
      if (m_lstbSensorData[u16Index])
      {
         vHandleSensorsSegmentValuesUpdate();
      }
      else
      {
         vRemoveSectorNodeImgs(m_u8CurrentSector);
      }
   }
   if (false == l_bIsDataStored)
   {
      ETG_TRACE_SYS_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: [%s]. Adding new status data failed."
                         "Invalid index or memory reallocation failed\n",
                         GetLegacyName()
                        ));
   }
}


void OPSWidget2D::vHandleBlinkInterval()
{
   BlinkIntervalType BlinkIntervalList = GetBlinkInterval();
   UInt32 u32Cnt = static_cast<UInt32>(BlinkIntervalList.Count());
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D : u32Cnt = %d", u32Cnt));
   BlinkIntervalList.SetEventCallback(&_listEventHandler, &ListEventHandler::OnBlinkIntervalListEvent);
   BlinkIntervalList.Request(0, u32Cnt);
}


/****************************************************************************
*    Function    : vOnBlinkIntervalListEvent
*    Description : Call back for blink interval
*    Parameters  : const Courier::ListEvent &
*    Return      : void
****************************************************************************/
void OPSWidget2D::ListEventHandler::OnBlinkIntervalListEvent(const Courier::ListEvent& SensorListEvent)
{
   if (m_pWidget == 0)
   {
      return;
   }

   Courier::UInt16 l_u16BlinkInterval = 0;
   const Courier::UInt16*  l_u16PBlinkInterval = SensorListEvent.GetItem<Courier::UInt16>();

   if (l_u16PBlinkInterval != 0)
   {
      l_u16BlinkInterval = *l_u16PBlinkInterval;
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D:vOnSensorFieldBlinkListEvent [%19s] received %d: newIndex=%u, oldIndex=%u, l_bSensorData=%d ",
                       m_pWidget->GetLegacyName(),
                       SensorListEvent.EventType(),
                       SensorListEvent.NewIndex(),
                       SensorListEvent.OldIndex(),
                       l_u16BlinkInterval
                      ));
   bool l_bResult = true;

   switch (SensorListEvent.EventType())
   {
      // list item delivery bay - the order requested items come in is not specified
      case Courier::ListEventType::RequestedItem:
      {
         //m_lstbSensorData.Insert(((UInt16)SensorListEvent.NewIndex()), (bool)l_bSensorData);
         m_pWidget->vAddNewSensorFieldBlinkIntervalData(m_pWidget->m_lstu16BlinkIntervalData, (UInt16)SensorListEvent.NewIndex(), l_u16BlinkInterval);
         //ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(),"OPSWidget2D : m_lstbSensorData.Insert & size %d", m_lstbSensorData.Size()));
      }
      break;

      // change & add event handling
      case Courier::ListEventType::ModifiedItem:
      case Courier::ListEventType::AddedItem:
      {
         m_pWidget->vAddNewSensorFieldBlinkIntervalData(m_pWidget->m_lstu16BlinkIntervalData, (UInt16)SensorListEvent.NewIndex(), l_u16BlinkInterval);

         // insert item to local list
         if (!SensorListEvent.HasDataItem())
         {
            // if the event does not carry the modified data item, create a request for it
            l_bResult = m_pWidget->m_lstu16BlinkIntervalProperty.Request(SensorListEvent.NewIndex(), 1);
         }
      }
      break;

      case Courier::ListEventType::RefreshList:
      case Courier::ListEventType::ItemCountChanged:
      {
         l_bResult = m_pWidget->m_lstu16BlinkIntervalProperty.Request(0, m_pWidget->m_lstbSensorsAvailProperty.Count());
      }
      break;

      default:
      {
      }
      break;
   }

   if (!l_bResult)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: [%s]. OnSensorsAvaila l_bResult = false", m_pWidget->GetLegacyName()));
   }
}


/****************************************************************************
*    Function    : vAddNewSensorFieldBlinkIntervalData
*    Description : Add function for the new blink interval data
*    Parameters  : u16Index, bool DataEntry
*    Return      : bool
****************************************************************************/
void OPSWidget2D::vAddNewSensorFieldBlinkIntervalData(FeatStd::Internal::Vector<Courier::UInt16>& Segdata, FeatStd::UInt16 u16Index, const Courier::UInt16& DataEntry)
{
   bool l_bIsDataStored = true;
   UInt32 l_NumberOfItemsInList = static_cast<UInt32>(Segdata.Size());
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: vAddNewSensorFieldBlinkIntervalData[%19s] - l_NumberOfItemsInList = [%d]u16Index = [%d], and DataEntry = [%d]",
                       GetLegacyName(), l_NumberOfItemsInList, u16Index, DataEntry));
   if (l_NumberOfItemsInList <= u16Index)
   {
      Candera::UInt32 l_Index = l_NumberOfItemsInList;
      for (; l_Index < u16Index; ++l_Index)
      {
         l_bIsDataStored = Segdata.Insert(static_cast<Int>(l_Index), DataEntry);
      }
      l_bIsDataStored = (l_bIsDataStored == true) ? Segdata.Insert(u16Index, DataEntry) : false;
   }
   else
   {
      Segdata[u16Index] = DataEntry;
   }
   if (!m_lstbSensorData.Empty() && u16Index < m_lstbSensorData.Size())
   {
      UInt32 l_NewSize = static_cast<UInt32>(Segdata.Size());
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: l_NewSize = [%d]", l_NewSize));
      if (l_NewSize)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: Statusdata[%d] = [%d]", u16Index, Segdata[u16Index]));
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: m_lstbSensorData[%d] = [%d]", u16Index, m_lstbSensorData[u16Index]));
      }
      m_u8CurrentSector = (UInt8)u16Index;
      if (m_lstbSensorData[u16Index])
      {
         vHandleSensorsSegmentValuesUpdate();
      }
      else
      {
         vRemoveSectorNodeImgs(m_u8CurrentSector);
      }
   }
   if (false == l_bIsDataStored)
   {
      ETG_TRACE_SYS_DCL((APP_TRACECLASS_ID(), "OPSWidget2D: [%s]. Adding new status data failed."
                         "Invalid index or memory reallocation failed\n",
                         GetLegacyName()
                        ));
   }
}


/****************************************************************************
*    Function    : vStartBlinkAnimationOfNode
*    Description : Starts the blink animation on the given node if mode of the node is active
*    Parameters  : RSegNode - Node to be blink animated
*                  enMode - Mode of the node
*    Return      : void
****************************************************************************/
void OPSWidget2D::vHandleBlinkAnimationOfNode(Candera::RenderNode* RSegNode, Candera::UInt8 u8SecNo, enImageMode enMode)
{
   if ((enActiveRed == enMode) && (RSegNode->IsRenderingEnabled() == true) && (_animationHelperInterface != 0))
   {
      float values[2] = { 1.0f, 0.0f };
      _animationHelperInterface->startAnimation(RSegNode, m_lstu16BlinkIntervalData[u8SecNo], values, &values[1], enAnimateRendering, BLINK_ANIMATION_REPEAT_COUNT);
   }
}


/****************************************************************************
*   Function    :
*   Description :
*   Parameters  :
*   Return      :
****************************************************************************/
void OPSWidget2D::OnAnimationTimeDispatcherChanged()
{
   if (_animationHelperInterface != 0)
   {
      _animationHelperInterface->setTimeDispatcher(GetAnimationTimeDispatcher());
   }
}


// ------------------------------------------------------------------------
void OPSWidget2D::OnParentViewActivate(bool enable)
{
   if (! enable)
   {
      stopExistingAnimations();
   }
   Base::OnParentViewActivate(enable);
}


void OPSWidget2D::stopExistingAnimations()
{
   if (_animationHelperInterface != 0)
   {
      _animationHelperInterface->stopAllExistingAnimations();
   }
}
