/* ***************************************************************************************
* FILE:          NodeControlWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  NodeControlWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "NodeControlWidget2D.h"

#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_NODECONTROL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/NodeControlWidget2D.cpp.trc.h"
#endif
#define IS_TRACE_ACTIVE etg_bIsTraceActive(APP_TRACECLASS_ID(), 5)


CGI_WIDGET_RTTI_DEFINITION(NodeControlWidget2D);


/*****************************************************************************/
static void TraceSetter(const NodeControlWidget2D& widget, const char* propName, const char* propValue)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Set %25s=%75s %s", propName, propValue, HMIBASE_TO_STRING_VW(&widget)));
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Set node=%s", HMIBASE_TO_STRING_SPN2D(widget.GetNode())));
}


/*****************************************************************************/
NodeControlWidget2D::NodeControlWidget2D() : Base(), _invalid(true)
{
}


/*****************************************************************************/
NodeControlWidget2D::~NodeControlWidget2D()
{
}


/*****************************************************************************/
void NodeControlWidget2D::Update()
{
   Base::Update();

   if (_invalid)
   {
      _invalid = false;
      Invalidate();
   }
}


/*****************************************************************************/
const Candera::Node2D* NodeControlWidget2D::getNodeOrDefaultNode() const
{
   Candera::Node2D* node = GetNode();

   if (node == NULL)
   {
      static Candera::Group2D* _defaultNode = Candera::Group2D::Create();
      node = _defaultNode;
   }

   return node;
}


/*****************************************************************************/
#define DefNodeControlCopyProp(propName) set##propName(original->get##propName())

/*****************************************************************************/
#define DefNodeControlSetPropertyImpl(propName)\
   void NodeControlWidget2D::set##propName(const propName##Type& value)\
   {\
      if ((GetNode() != NULL) && is##propName##Enabled())\
      {\
         if (IS_TRACE_ACTIVE)\
         {\
            TraceSetter(*this, #propName, HMIBASE_TO_STRING(value));\
         }\
         GetNode()->Set##propName(value);\
         _invalid = true;\
         triggerUpdate();\
      }\
   }

/*****************************************************************************/
#define DefNodeControlGetPropertyImpl(propName)\
   NodeControlWidget2D::propName##Type NodeControlWidget2D::get##propName() const\
   {\
      const Candera::Node2D* node = getNodeOrDefaultNode();\
      return (node != NULL) ? node->Get##propName() : propName##Type();\
   }

/*****************************************************************************/
#define DefNodeControlPropertyImpl(propName)\
   DefNodeControlSetPropertyImpl(propName)\
   DefNodeControlGetPropertyImpl(propName)

/*****************************************************************************/
#define DefNodeControlLayoutPropertyImpl(propName, getter)\
   void NodeControlWidget2D::set##propName(const propName##Type& value)\
   {\
      if ((GetNode() != NULL) && is##propName##Enabled())\
      {\
         if (IS_TRACE_ACTIVE)\
         {\
            TraceSetter(*this, #propName, HMIBASE_TO_STRING(value));\
         }\
         Candera::Layouter::Set##propName(*GetNode(), value);\
         _invalid = true;\
         triggerUpdate();\
      }\
   }\
   NodeControlWidget2D::propName##Type NodeControlWidget2D::get##propName() const\
   {\
      const Candera::Node2D* node = getNodeOrDefaultNode();\
      return (node != NULL) ? Candera::Layouter::getter##propName(const_cast<Candera::Node2D&>(*node)) : propName##Type();\
   }

/*****************************************************************************/
#define DefNodeControlGridLayoutPropertyImpl(propName, getter)\
   void NodeControlWidget2D::set##propName(const propName##Type& value)\
   {\
      if ((GetNode() != NULL) && is##propName##Enabled())\
      {\
         if (IS_TRACE_ACTIVE)\
         {\
            TraceSetter(*this, #propName, HMIBASE_TO_STRING(value));\
         }\
         Candera::GridLayouter::Set##propName(*GetNode(), value);\
         _invalid = true;\
         triggerUpdate();\
      }\
   }\
   NodeControlWidget2D::propName##Type NodeControlWidget2D::get##propName() const\
   {\
      const Candera::Node2D* node = getNodeOrDefaultNode();\
      return (node != NULL) ? Candera::GridLayouter::getter##propName(const_cast<Candera::Node2D&>(*node)) : propName##Type();\
   }

/*****************************************************************************/
DefNodeControlPropertyImpl(Position)
DefNodeControlPropertyImpl(PivotPoint)
DefNodeControlPropertyImpl(PivotOffset)
DefNodeControlPropertyImpl(BoundingRectangle)
DefNodeControlPropertyImpl(Rotation)
DefNodeControlPropertyImpl(Scale)

DefNodeControlPropertyImpl(AlphaValue)
DefNodeControlSetPropertyImpl(RenderingEnabled)
DefNodeControlPropertyImpl(RenderOrderRank)

DefNodeControlLayoutPropertyImpl(VerticalAlignment, Get)
DefNodeControlLayoutPropertyImpl(HorizontalAlignment, Get)
DefNodeControlLayoutPropertyImpl(Size, Get)
DefNodeControlLayoutPropertyImpl(MinimumSize, Get)
DefNodeControlLayoutPropertyImpl(MaximumSize, Get)
DefNodeControlLayoutPropertyImpl(Margin, Get)
DefNodeControlLayoutPropertyImpl(StretchBehavior, Get)
DefNodeControlLayoutPropertyImpl(LayoutDirection, Get)
DefNodeControlLayoutPropertyImpl(Collapsible, Is)

DefNodeControlGridLayoutPropertyImpl(Row, Get)
DefNodeControlGridLayoutPropertyImpl(RowSpan, Get)
DefNodeControlGridLayoutPropertyImpl(Column, Get)
DefNodeControlGridLayoutPropertyImpl(ColumnSpan, Get)


/*****************************************************************************/
void NodeControlWidget2D::setClippingRectangle(const ClippingRectangleType& value)
{
   if ((GetNode() != NULL) && isClippingRectangleEnabled())
   {
      Candera::RenderNode* renderNode = Candera::Dynamic_Cast<Candera::RenderNode*>(GetNode());
      if (renderNode != NULL)
      {
         if (IS_TRACE_ACTIVE)
         {
            TraceSetter(*this, "ClippingRectangle", HMIBASE_TO_STRING(value));
         }
         renderNode->SetClippingRect(value);
         _invalid = true;
         triggerUpdate();
      }
   }
}


/*****************************************************************************/
NodeControlWidget2D::ClippingRectangleType NodeControlWidget2D::getClippingRectangle() const
{
   Candera::RenderNode* renderNode = Candera::Dynamic_Cast<Candera::RenderNode*>(GetNode());
   return (renderNode != NULL) ? renderNode->GetClippingRect() : Candera::Rectangle();
}


/*****************************************************************************/
void NodeControlWidget2D::setScopeMask(const ScopeMaskType& value)
{
   if ((GetNode() != NULL) && isScopeMaskEnabled())
   {
      if (IS_TRACE_ACTIVE)
      {
         TraceSetter(*this, "ScopeMask", HMIBASE_TO_STRING(value));
      }
      ScopeMaskType cond = ScopeMaskType(0);
      for (Candera::UInt index = 0; cond != ~ScopeMaskType(0); ++index)     //lint !e441 cond is changed by index
      {
         bool enabled = ((value & (ScopeMaskType(1) << index)) != 0);
         GetNode()->SetScopeEnabled(index, enabled);
         cond = cond | (ScopeMaskType(1) << index);
      }
      _invalid = true;
      triggerUpdate();
   }
}


/*****************************************************************************/
NodeControlWidget2D::ScopeMaskType NodeControlWidget2D::getScopeMask() const
{
   ScopeMaskType value = ~ScopeMaskType();

   const Candera::Node2D* node = getNodeOrDefaultNode();
   if (node != NULL)
   {
      ScopeMaskType cond = ScopeMaskType(0);
      for (Candera::UInt index = 0; cond != ~ScopeMaskType(0); ++index)    //lint !e441 cond is changed by index
      {
         if (!node->IsScopeEnabled(index))
         {
            value = value & ~(ScopeMaskType(1) << index);
         }
         cond = cond | (ScopeMaskType(1) << index);
      }
   }

   return value;
}


/*****************************************************************************/
NodeControlWidget2D::RenderingEnabledType NodeControlWidget2D::getRenderingEnabled() const
{
   const Candera::Node2D* node = getNodeOrDefaultNode();
   return (node != NULL) ? node->IsRenderingEnabled() : true;
}


/*****************************************************************************/
bool NodeControlWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   if (!Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      return false;
   }

   const NodeControlWidget2D* original = CLONEABLE_WIDGET_CAST<const NodeControlWidget2D*>(originalWidget);
   if (original == NULL)
   {
      return false;
   }

   _config = original->_config;

   DefNodeControlCopyProp(Position);
   DefNodeControlCopyProp(PivotPoint);
   DefNodeControlCopyProp(PivotOffset);
   DefNodeControlCopyProp(BoundingRectangle);
   DefNodeControlCopyProp(Rotation);
   DefNodeControlCopyProp(Scale);
   DefNodeControlCopyProp(ScopeMask);
   DefNodeControlCopyProp(AlphaValue);
   DefNodeControlCopyProp(RenderingEnabled);
   DefNodeControlCopyProp(RenderOrderRank);
   DefNodeControlCopyProp(ClippingRectangle);

   DefNodeControlCopyProp(VerticalAlignment);
   DefNodeControlCopyProp(HorizontalAlignment);
   DefNodeControlCopyProp(Size);
   DefNodeControlCopyProp(MinimumSize);
   DefNodeControlCopyProp(MaximumSize);
   DefNodeControlCopyProp(Margin);
   DefNodeControlCopyProp(StretchBehavior);
   DefNodeControlCopyProp(LayoutDirection);
   DefNodeControlCopyProp(Collapsible);

   DefNodeControlCopyProp(Row);
   DefNodeControlCopyProp(RowSpan);
   DefNodeControlCopyProp(Column);
   DefNodeControlCopyProp(ColumnSpan);

   return true;
}
