/* ***************************************************************************************
* FILE:          MultiSliderWidget2D.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  MultiSliderWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "MultiSliderWidget2D.h"
#include "Widgets/3D/MultiSliderHelper/MultiSliderHelperWidget3D.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_SLIDER
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/MultiSliderWidget2D.cpp.trc.h"
#endif

using Candera::Node2D;
using Candera::Vector2;
using FeatStd::Float;
using FeatStd::Int;
using FeatStd::UInt;
using FeatStd::SizeType;

static const SizeType INVALID_TOUCHED_VALUE_INDEX = ~SizeType();

CGI_WIDGET_RTTI_DEFINITION(MultiSliderWidget2D)

MultiSliderWidget2D::MultiSliderWidget2D() :
   m_centerTouched(false),
   m_viewChanged(false),
   m_handleNodeChanged(false),
   m_touchedValueIndex(INVALID_TOUCHED_VALUE_INDEX)
{
}


MultiSliderWidget2D::~MultiSliderWidget2D()
{
}


void MultiSliderWidget2D::Update()
{
   Base::Update();
   if (m_viewChanged)
   {
      UpdateView();
      m_viewChanged = false;
   }
   if (m_handleNodeChanged)
   {
      UpdateHandleNodes();
      m_handleNodeChanged = false;
   }

   const SizeType count = m_valueChanged.GetCount();
   bool updateView = false;
   for (SizeType i = 0; i < count; ++i)
   {
      bool& valueChanged = m_valueChanged.Get(i);
      if (valueChanged)
      {
         UpdateHandle(i);
         valueChanged = false;
         updateView = true;
      }
   }

   if (updateView)
   {
      UpdateView();
   }
}


void MultiSliderWidget2D::OnChanged(::FeatStd::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   switch (propertyId)
   {
      case MultiSliderViewPropertyId:
         m_viewChanged = true;
         break;

      case HandleNodePropertyId:
         m_handleNodeChanged = true;
         break;

      case ValuesPropertyId:
      {
         const SizeType count = GetValues().GetCount();
         const bool countChanged = (count != m_valueChanged.GetCount());
         if (countChanged)
         {
            m_valueChanged.Clear();
            m_valueChanged.Reserve(count);
         }
         for (SizeType i = 0; i < count; ++i)
         {
            if (countChanged)
            {
               m_valueChanged.Add(true);
            }
            else
            {
               m_valueChanged.Get(i) = true;
            }
         }

         PropertyModified("Values");
         Invalidate();
         break;
      }

      default:
         break;
   }
}


bool MultiSliderWidget2D::OnTouchMessage(const Courier::TouchMsg& msg)
{
   bool consumed = false;

   bool isCenter = false;

   const SizeType valueCount = GetValues().GetCount();
   Candera::Node2D* node = GetNode();

   if (node != 0 && valueCount > 0)
   {
      Vector2 touchPos = GetRelativeTouchPos(Vector2(Float(msg.GetXPos()), Float(msg.GetYPos())));
      if (touchPos.GetLength() <= GetRadiusCenter())
      {
         isCenter = true;
      }

      switch (msg.GetState())
      {
         case Courier::TouchMsgState::Down:
            m_centerTouched = isCenter;
            break;

         case Courier::TouchMsgState::Move:
         {
            const Float segmentAngle = 360.0F / static_cast<Float>(valueCount);
            const Float angle = GetStartAngle() + (segmentAngle * static_cast<Float>(m_touchedValueIndex));
            // rotate touched value axis to 12 o'clock position and take vertical offset to center point (-y) to calculate new value
            Candera::Matrix3x2 m;
            m.SetRotation(-angle);
            SetValue(m_touchedValueIndex, CalculateValue(-m.Multiply(touchPos).GetY()));
            break;
         }

         case Courier::TouchMsgState::Up:
            if (isCenter && m_centerTouched)
            {
               for (SizeType valueIndex = 0; valueIndex < valueCount; ++valueIndex)
               {
                  SetValue(valueIndex, 0.0f);
               }
            }
            m_touchedValueIndex = INVALID_TOUCHED_VALUE_INDEX;
            break;

         default:
            break;
      }
   }

   return consumed;
}


bool MultiSliderWidget2D::OnTouch(const Candera::Camera2D& camera, const Candera::Vector2& touchedPoint)
{
   // check center position
   bool touched = (GetRelativeTouchPos(touchedPoint).GetLength() <= GetRadiusCenter());
   m_touchedValueIndex = INVALID_TOUCHED_VALUE_INDEX;

   if (!touched)
   {
      // check slider handles
      Node2D* node = GetNode();
      if (node != 0)
      {
         SizeType index = 0;
         for (Node2D* child = node->GetFirstChild(); (child != 0) && !touched; child = child->GetNextSibling())
         {
            touched = IsPickIntersectingNode(camera, child, touchedPoint);
            if (touched)
            {
               m_touchedValueIndex = index;
            }
            ++index;
         }
      }
   }

   return touched;
}


Candera::Vector2 MultiSliderWidget2D::GetRelativeTouchPos(const Candera::Vector2& touchPos) const
{
   Vector2 relativeTouchPos;

   Node2D* node = GetNode();
   if (node != 0)
   {
      Candera::Vector2 worldPos = node->GetWorldPosition();
      const Candera::Vector2& pivotOffset = node->GetPivotOffset();

      relativeTouchPos.SetX(Float(touchPos.GetX() - worldPos.GetX() + pivotOffset.GetX()));
      relativeTouchPos.SetY(Float(touchPos.GetY() - worldPos.GetY() + pivotOffset.GetY()));
   }

   return relativeTouchPos;
}


FeatStd::Float MultiSliderWidget2D::CalculateValue(FeatStd::Float radius) const
{
   const Float min = GetRadiusMin();
   const Float max = GetRadiusMax();
   Float value = (radius - min) / (max - min);

   if (value < 0.0F)
   {
      value = 0.0F;
   }
   else if (value > 1.0F)
   {
      value = 1.0F;
   }

   return value;
}


void MultiSliderWidget2D::SetValue(FeatStd::SizeType valueIndex, FeatStd::Float value)
{
   if (valueIndex < GetValues().GetCount())
   {
      GetValues().Get(valueIndex) = value;
   }
   if (valueIndex < m_valueChanged.GetCount())
   {
      m_valueChanged.Get(valueIndex) = true;
   }
   PropertyModified("Values");
   Invalidate();
}


void MultiSliderWidget2D::UpdateView()
{
   MultiSliderHelperWidget3D* multiSliderView = Candera::Dynamic_Cast<MultiSliderHelperWidget3D*>(GetMultiSliderView());
   if (multiSliderView != 0)
   {
      multiSliderView->UpdateValues(GetValues());
      Invalidate();
   }
}


void MultiSliderWidget2D::UpdateHandle(FeatStd::SizeType valueIndex)
{
   Node2D* node = GetNode();
   if ((node != 0) && (valueIndex != INVALID_TOUCHED_VALUE_INDEX))
   {
      Node2D* handle = node->GetFirstChild();
      for (SizeType i = 0; (i < valueIndex) && (handle != 0); ++i)
      {
         handle = handle->GetNextSibling();
      }
      if (handle != 0)
      {
         const SizeType valueCount = GetValues().GetCount();
         const Float segmentAngle = 360.0F / static_cast<Float>(valueCount);
         const Float angle = GetStartAngle() + (segmentAngle * static_cast<Float>(valueIndex));
         const Float angleRad = Candera::Math::DegreeToRadian(angle);
         const Float value = GetValues().Get(valueIndex) * (GetRadiusMax() - GetRadiusMin()) + GetRadiusMin();
         const Float x = Candera::Math::Sine(angleRad) * value - node->GetPivotOffset().GetX();
         const Float y = -Candera::Math::Cosine(angleRad) * value - node->GetPivotOffset().GetY();
         handle->SetPosition(x, y);
      }
   }
}


void MultiSliderWidget2D::UpdateHandleNodes()
{
   Node2D* node = GetNode();

   if (node != 0)
   {
      // remove old handle nodes
      Node2D* child = node->GetFirstChild();
      while (child != 0)
      {
         node->RemoveChild(child);
         Node2D* nextSibling = child->GetNextSibling();
         child->Dispose();
         child = nextSibling;
      }

      Node2D* handleNode = GetHandleNode();
      if (handleNode != 0)
      {
         const SizeType valueCount = GetValues().GetCount();
         const Float segmentAngle = 360.0F / static_cast<Float>(valueCount);

         // clone handle node for each value and set its position and rotation
         for (SizeType i = 0; i < valueCount; i++)
         {
            Node2D* handle = handleNode->Clone();
            if (handle != 0)
            {
               const Float angle = GetStartAngle() + (segmentAngle * static_cast<Float>(i));
               handle->SetRotation(angle);
               handle->SetRenderingEnabled(true);
               node->AddChild(handle);

               UpdateHandle(i);
            }
         }
      }
   }
}
