/* ***************************************************************************************
* FILE:          MeterWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  MeterWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "Widgets/2D/BaseWidget2D.h"
#include "MeterWidget2D.h"
#include "Candera/Engine2D/Effects/Effect2D.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_METER
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/MeterWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(MeterWidget2D);


/****************************************************************************
*    Description : Constructor
*    Parameters  : void
****************************************************************************/
MeterWidget2D::MeterWidget2D() :
   _requestReconfiguration(false),
   _currentIndicatorRotation(0.0f)
{
}


/****************************************************************************
*    Description : Destructor
****************************************************************************/
MeterWidget2D::~MeterWidget2D()
{
}


/****************************************************************************
*    Description : Initializes the widget so that all referred resource can be
*                                resolved
****************************************************************************/
void MeterWidget2D::InitWidget()
{
   Base::InitWidget();
   if (NULL != GetAssetProvider())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "MeterWidget2D [%s]: Initialized", GetLegacyName()));
   }
}


/****************************************************************************
*    Description : overrides WidgetBase::Update
****************************************************************************/
void MeterWidget2D::Update()
{
   Base::Update();
   if (_requestReconfiguration && IsEnabled())
   {
      updateGraph();
      _requestReconfiguration = false;
      Invalidate();
   }
}


/****************************************************************************
*    Description : Overrides MeterWidget2DBase::OnChanged
*    Return      : void
****************************************************************************/
void MeterWidget2D::OnChanged(::Candera::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   switch (propertyId)
   {
      case CurrentValuePropertyId:
      {
         // Adjust the value when it is not within limits.
         if (GetMinimumValue() < GetMaximumValue())
         {
            if (GetCurrentValue() < GetMinimumValue())
            {
               SetCurrentValue(GetMinimumValue());
            }
            else if (GetCurrentValue() > GetMaximumValue())
            {
               SetCurrentValue(GetMaximumValue());
            }
         }

         // When the value is out of the valid range, we should not start progress the graph.
         bool tubeValueWithinRange = isGraphValueWithinValidRange();
         if (tubeValueWithinRange == true)
         {
            _requestReconfiguration = true;
         }
      }
      break;

      default:
         break;
   }
}


/****************************************************************************
*    Description : check boundary condition for min, max value and angle.
*    Return      : true, if value in tube range
****************************************************************************/
bool MeterWidget2D::isGraphValueWithinValidRange() const
{
   // The graph value will be within range only for the below condition. In all other cases it will be outside the valid range.
   if (((GetMinimumValue() < GetMaximumValue()) && (GetCurrentValue() >= GetMinimumValue()) && (GetCurrentValue() <= GetMaximumValue())) ||
         ((GetMinimumValue() > GetMaximumValue()) && (GetCurrentValue() <= GetMinimumValue()) && (GetCurrentValue() >= GetMaximumValue())))
   {
      return true;
   }
   return false;
}


/****************************************************************************
*    Description : Rotate the indicator node according to current value to
*                    progress.
****************************************************************************/
void MeterWidget2D::updateGraph()
{
   _currentIndicatorRotation = calculateCurrentRotationVal();
   if (NULL != GetIndicatorNode())
   {
      GetIndicatorNode()->SetRotation(_currentIndicatorRotation);
   }
}


/****************************************************************************
*    calculateCurrentRotationVal
****************************************************************************/
Candera::Float MeterWidget2D::calculateCurrentRotationVal()
{
   //logic to get current rotation value

   if (GetMaximumValue() != GetMinimumValue())  // Max and min value shall not be equal.
   {
      if (GetRotationType() == Candera::ClockWise)//clockwise rotation
      {
         if (GetMaximumAngle() >= GetMinimumAngle())
         {
            _currentIndicatorRotation = GetMinimumAngle() + ((GetMaximumAngle() - GetMinimumAngle()) / (GetMaximumValue() - GetMinimumValue())) * (GetCurrentValue() - GetMinimumValue());
         }
         else
         {
            _currentIndicatorRotation = ((GetMaximumAngle() - GetMinimumAngle()) / (GetMaximumValue() - GetMinimumValue())) * (GetCurrentValue() - GetMinimumValue());
         }
      }
      else if (GetRotationType() == Candera::AntiClockWise) //anti-clockwise rotation
      {
         if (GetMinimumAngle() >= GetMaximumAngle())
         {
            _currentIndicatorRotation = GetMinimumAngle() + ((GetMaximumAngle() - GetMinimumAngle()) / (GetMaximumValue() - GetMinimumValue())) * (GetCurrentValue() - GetMinimumValue());
         }
         else
         {
            _currentIndicatorRotation = ((GetMaximumAngle() - GetMinimumAngle()) / (GetMaximumValue() - GetMinimumValue())) * (GetCurrentValue() - GetMinimumValue());
         }
      }
   }
   else
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "MeterWidget2D: Both min and max val are equal"));
   }

   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "MeterWidget2D: CurrentValue:[%f] CurrentRotation:[%f]", GetCurrentValue(), _currentIndicatorRotation));
   return _currentIndicatorRotation;
}
