/* ***************************************************************************************
* FILE:          MeshWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  MeshWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"

#include "MeshWidget2D.h"

#include <Candera/Engine3D/Mathematics/Math3D.h>
#include <Candera/System/Mathematics/Math.h>

#include <Candera/Engine2D/Core/RenderNode.h>
#include <Candera/EngineBase/Layout/Layouter.h>
#include <Candera/EngineBase/Common/InterfaceAdapter.h>
#include <Candera/Engine2D/Core/VertexBuffer2D.h>
#include <Candera/Engine2D/Core/Mesh2D.h>
#include <Courier/Visualization/ViewScene2D.h>

#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_MESH
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/MeshWidget2D.cpp.trc.h"
#endif

namespace hmibase {
namespace widget {
namespace mesh {

CGI_WIDGET_RTTI_DEFINITION(MeshWidget2D)

MeshWidget2D::MeshWidget2D() :
   m_geometry(0),
   m_meshNode(0),
   m_updateMesh(true),
   m_setLayouter(false)
{
#if defined(CANDERA_LAYOUT_ENABLED)
   m_interfaceAdapter.Init(this);
#endif
}


MeshWidget2D::~MeshWidget2D()
{
   Finalize();
}


void MeshWidget2D::Finalize()
{
   if (0 != m_geometry)
   {
      FEATSTD_DELETE(m_geometry);
      m_geometry = 0;
   }

   Candera::Node2D* node = GetNode2D();
   if (0 != node)
   {
      node->SetLayouter(0);
   }
}


bool MeshWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned = false;
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const MeshWidget2D* original = CLONEABLE_WIDGET_CAST<const MeshWidget2D*>(originalWidget);
      if (original != NULL)
      {
         SetRotation(original->GetRotation());
         SetCameraDistance(original->GetCameraDistance());
         SetFitToMaximumSizeRotation(original->GetFitToMaximumSizeRotation());
         SetRotationAxis(original->GetRotationAxis());
         SetFieldOfView(original->GetFieldOfView());
         SetFitStrategy(original->GetFitStrategy());
         SetHorizontalOrigin(original->GetHorizontalOrigin());
         SetVerticalOrigin(original->GetVerticalOrigin());
         SetCustomHorizontalOrigin(original->GetCustomHorizontalOrigin());
         SetCustomVerticalOrigin(original->GetCustomVerticalOrigin());
         SetOriginOffset(original->GetOriginOffset());

         cloned = true;
      }
   }
   return cloned;
}


void MeshWidget2D::OnChanged(FeatStd::UInt32 propertyId)
{
   Base::OnChanged(propertyId);
   m_updateMesh = true;
}


static void MeshWidget2DSetupOrigin(const MeshWidget2D& widget, const Candera::Rectangle& rectangle, Candera::Vector2& origin)
{
   switch (widget.GetHorizontalOrigin())
   {
      case HorizontalOrigin::Left:
         break;
      case HorizontalOrigin::Center:
         origin.SetX(0.5F);
         break;
      case HorizontalOrigin::Right:
         origin.SetX(1.0F);
         break;
      case HorizontalOrigin::Custom:
         origin.SetX(widget.GetCustomHorizontalOrigin());
         break;
      default:
         break;
   }
   switch (widget.GetVerticalOrigin())
   {
      case VerticalOrigin::Top:
         break;
      case VerticalOrigin::Center:
         origin.SetY(0.5F);
         break;
      case VerticalOrigin::Bottom:
         origin.SetY(1.0F);
         break;
      case VerticalOrigin::Custom:
         origin.SetY(widget.GetCustomVerticalOrigin());
         break;
      default:
         break;
   }
   if ((0.0F != widget.GetOriginOffset().GetX()) && (0.0F != rectangle.GetWidth()))
   {
      origin.SetX(origin.GetX() + widget.GetOriginOffset().GetX() / rectangle.GetWidth());
   }
   if ((0.0F != widget.GetOriginOffset().GetY()) && (0.0F != rectangle.GetHeight()))
   {
      origin.SetY(origin.GetY() + widget.GetOriginOffset().GetY() / rectangle.GetHeight());
   }
}


static void MeshWidget2DSetupNormalizedMesh(const Candera::Vector2& origin, Candera::Vector3(&mesh)[4])
{
   mesh[0].SetX(-origin.GetX());
   mesh[0].SetY(-origin.GetY());
   mesh[1].SetX(1.0F - origin.GetX());
   mesh[1].SetY(-origin.GetY());
   mesh[2].SetX(-origin.GetX());
   mesh[2].SetY(1.0F - origin.GetY());
   mesh[3].SetX(1.0F - origin.GetX());
   mesh[3].SetY(1.0F - origin.GetY());
}


static void MeshWidget2DSetupProjectionMatrix(FeatStd::Float fieldOfView, Candera::Matrix4& projection)
{
   projection.SetIdentity();
   // by choosing a zNear of 1, a zFar of 10001 and an aspect ratio of 1 the Candera::Math3D::SetPerspectiveProjection is simplified to the following values:
   FeatStd::Float fov = MeshWidget2D::NormalizeDegree(fieldOfView);
   if ((fov > 0.0F) && (fov < 180.0F))
   {
      FeatStd::Float f = 1.0F / Candera::Math::Tangent(Candera::Math::DegreeToRadian(fov * 0.5F));
      projection(0, 0) = f;
      projection(1, 1) = f;
      projection(2, 2) = -1.0002F;
      projection(3, 2) = -2.0002F;
      projection(2, 3) = -1.0F;
      projection(3, 3) = 0.0F;
   }
   else
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "invalid field of view!"));
   }
}


static void MeshWidget2DSetupViewMatrix(FeatStd::Float cameraDistance, Candera::Matrix4& view)
{
   view.SetIdentity();
   view.Set(2, 2, -1.0F);
   Candera::Matrix4 translation;
   translation.SetTranslation(0.0F, 0.0F, cameraDistance);
   view = translation * view;
}


static void MeshWidget2DSetupWorldMatrix(FeatStd::Float rotation, const Candera::Vector3& rotationAxis, Candera::Matrix4& world)
{
   if (0.0F != rotation)
   {
      Candera::Vector3 normalizedRotationAxis = rotationAxis;
      if (normalizedRotationAxis.Normalize() != 0.0F)
      {
         world.SetRotationAxis(-rotation, normalizedRotationAxis.GetX(), normalizedRotationAxis.GetY(), normalizedRotationAxis.GetZ());
      }
      else
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "invalid rotation axis!"));
      }
   }
}


static void MeshWidget2DProjectMeshBackToObjectPlane(const Candera::Matrix4& projection, Candera::Vector3(&mesh)[4])
{
   Candera::Matrix4 view;
   MeshWidget2DSetupViewMatrix(1.0F, view);
   Candera::Matrix4 transformation = view * projection;
   transformation.Inverse();
   for (FeatStd::SizeType i = 0; i < 4; ++i)
   {
      mesh[i].SetZ(1.0F);
      mesh[i].TransformCoordinate(transformation);
      // simplified Plane::IsIntersectingLine due to fixed values of the Plane((Vector3(0.0F, 0.0F, 1.0F), 0.0F) and camera x,y being 0.0F and z 1.0F.
      const FeatStd::Float directionZ = mesh[i].GetZ() - 1.0F;
      const FeatStd::Float temp = 1.0F / directionZ;
      if (temp != 0.0F)
      {
         mesh[i].SetX(temp * mesh[i].GetX());
         mesh[i].SetY(temp * mesh[i].GetY());
         mesh[i].SetZ(1.0F - (temp * directionZ));
      }
   }
}


static void MeshWidget2DProjectMeshToProjectionPlane(FeatStd::Float rotation, const Candera::Vector3& rotationAxis, const Candera::Matrix4& view, const Candera::Matrix4& projection, Candera::Vector3(&mesh)[4])
{
   // project mesh to projection plane
   Candera::Matrix4 world;
   MeshWidget2DSetupWorldMatrix(rotation, rotationAxis, world);
   Candera::Matrix4 transformation = world * view * projection;
   for (FeatStd::SizeType i = 0; i < 4; ++i)
   {
      mesh[i].TransformCoordinate(transformation);
   }
}


static void MeshWidget2DMapMesh(FeatStd::Float centerX, FeatStd::Float centerY, FeatStd::Float width, FeatStd::Float height, const Candera::Vector3(&mesh)[4], Candera::Vector3(&mappedMesh)[4])
{
   for (FeatStd::SizeType i = 0; i < 4; ++i)
   {
      mappedMesh[i].SetX(centerX + mesh[i].GetX() * width);
      mappedMesh[i].SetY(centerY + mesh[i].GetY() * height);
   }
}


static void MeshWidget2DGetBoundaries(const Candera::Vector3(&mesh)[4], FeatStd::Float& xMin, FeatStd::Float& xMax, FeatStd::Float& yMin, FeatStd::Float& yMax)
{
   xMin = Candera::Math::MaxFloat();
   xMax = -Candera::Math::MaxFloat();
   yMin = Candera::Math::MaxFloat();
   yMax = -Candera::Math::MaxFloat();
   for (FeatStd::SizeType i = 0; i < 4; ++i)
   {
      if (xMin > mesh[i].GetX())
      {
         xMin = mesh[i].GetX();
      }
      if (xMax < mesh[i].GetX())
      {
         xMax = mesh[i].GetX();
      }
      if (yMin > mesh[i].GetY())
      {
         yMin = mesh[i].GetY();
      }
      if (yMax < mesh[i].GetY())
      {
         yMax = mesh[i].GetY();
      }
   }
}


static void MeshWidget2DCalculateFitScale(FeatStd::Float centerX, FeatStd::Float centerY, FeatStd::Float xMin, FeatStd::Float xMax, FeatStd::Float yMin, FeatStd::Float yMax, const Candera::Rectangle& rectangle, FeatStd::Float& scale)
{
   FeatStd::Float scaleX = 1.0F;
   FeatStd::Float scaleY = 1.0F;
   if ((xMin < rectangle.GetLeft()) && ((rectangle.GetLeft() - centerX) != 0.0F))
   {
      scaleX = (rectangle.GetLeft() - centerX) / (xMin - centerX);
   }
   if ((xMax > (rectangle.GetLeft() + rectangle.GetWidth())) && ((rectangle.GetLeft() + rectangle.GetWidth() - centerX) != 0.0F))
   {
      FeatStd::Float scaleX2 = (rectangle.GetLeft() + rectangle.GetWidth() - centerX) / (xMax - centerX);
      if (scaleX2 < scaleX)
      {
         scaleX = scaleX2;
      }
   }
   if ((yMin < rectangle.GetTop()) && ((rectangle.GetTop() - centerY) != 0.0F))
   {
      scaleY = (rectangle.GetTop() - centerY) / (yMin - centerY);
   }
   if ((yMax > (rectangle.GetTop() + rectangle.GetHeight())) && ((rectangle.GetTop() + rectangle.GetHeight() - centerY) != 0.0F))
   {
      FeatStd::Float scaleY2 = (rectangle.GetTop() + rectangle.GetHeight() - centerY) / (yMax - centerY);
      if (scaleY2 < scaleY)
      {
         scaleY = scaleY2;
      }
   }
   scale = (scaleX < scaleY) ? scaleX : scaleY;
}


static void MeshWidget2DUpdateMesh(const Candera::Vector2& size, Candera::Effect2D* effect, const MeshWidget2D& widget, Candera::Vector3(&mappedMesh)[4])
{
   if (0 != effect)
   {
      Candera::Rectangle rectangle;
      effect->GetBoundingRectangle(rectangle);
      if (size.GetX() >= 0.0F)
      {
         rectangle.SetWidth(size.GetX());
      }
      if (size.GetY() >= 0.0F)
      {
         rectangle.SetHeight(size.GetY());
      }
      Candera::Vector2 origin;
      MeshWidget2DSetupOrigin(widget, rectangle, origin);
      Candera::Vector3 mesh[4];
      MeshWidget2DSetupNormalizedMesh(origin, mesh);
      Candera::Matrix4 projection;
      MeshWidget2DSetupProjectionMatrix(widget.GetFieldOfView(), projection);
      Candera::Matrix4 view;
      MeshWidget2DSetupViewMatrix(widget.GetCameraDistance(), view);
      MeshWidget2DProjectMeshBackToObjectPlane(projection, mesh);
      Candera::Vector3 projectedNormalizedMesh[4];
      for (FeatStd::SizeType i = 0; i < 4; ++i)
      {
         projectedNormalizedMesh[i] = mesh[i];
      }
      MeshWidget2DProjectMeshToProjectionPlane(widget.GetRotation(), widget.GetRotationAxis(), view, projection, projectedNormalizedMesh);
      FeatStd::Float xMin = Candera::Math::MaxFloat();
      FeatStd::Float xMax = -Candera::Math::MaxFloat();
      FeatStd::Float yMin = Candera::Math::MaxFloat();
      FeatStd::Float yMax = -Candera::Math::MaxFloat();
      FeatStd::Float centerX = rectangle.GetLeft() + rectangle.GetWidth() * (origin.GetX());
      FeatStd::Float centerY = rectangle.GetTop() + rectangle.GetHeight() * (origin.GetY());
      FeatStd::Float scale = 1.0F;
      if (widget.GetFitStrategy() == Mesh2DFitStrategy::FitToMaximumSize)
      {
         MeshWidget2DProjectMeshToProjectionPlane(widget.GetFitToMaximumSizeRotation(), widget.GetRotationAxis(), view, projection, mesh);
         MeshWidget2DMapMesh(centerX, centerY, rectangle.GetWidth(), rectangle.GetHeight(), mesh, mappedMesh);
         MeshWidget2DGetBoundaries(mappedMesh, xMin, xMax, yMin, yMax);
         MeshWidget2DCalculateFitScale(centerX, centerY, xMin, xMax, yMin, yMax, rectangle, scale);
         MeshWidget2DMapMesh(centerX, centerY, rectangle.GetWidth() * scale, rectangle.GetHeight() * scale, projectedNormalizedMesh, mappedMesh);
      }
      else
      {
         MeshWidget2DMapMesh(centerX, centerY, rectangle.GetWidth(), rectangle.GetHeight(), projectedNormalizedMesh, mappedMesh);
         if (widget.GetFitStrategy() == Mesh2DFitStrategy::FitToActualSize)
         {
            MeshWidget2DGetBoundaries(mappedMesh, xMin, xMax, yMin, yMax);
            MeshWidget2DCalculateFitScale(centerX, centerY, xMin, xMax, yMin, yMax, rectangle, scale);
            if (scale != 1.0F)
            {
               MeshWidget2DMapMesh(centerX, centerY, rectangle.GetWidth() * scale, rectangle.GetHeight() * scale, projectedNormalizedMesh, mappedMesh);
            }
         }
      }
   }
}


void MeshWidget2D::Update()
{
   Base::Update();
#if defined(CANDERA_LAYOUT_ENABLED)
   if (m_setLayouter)
   {
      m_setLayouter = false;
      Candera::Node2D* node = GetNode2D();
      if (0 != node)
      {
         node->SetLayouter(&m_interfaceAdapter);
      }
   }
#endif
   if (m_updateMesh)
   {
      m_updateMesh = false;
      Candera::Node2D* node = GetNode2D();
      if (0 != node)
      {
         Candera::Layouter::InvalidateLayout(node);
         Invalidate();
      }
   }
}


void MeshWidget2D::OnNodeChanged()
{
   Base::OnNodeChanged();

#if defined(CANDERA_LAYOUT_ENABLED)
   Candera::Node2D* node = GetNode2D();
   if (0 != node)
   {
      node->SetLayouter(0);
   }
#endif
}


void MeshWidget2D::OnBeforeNodeChanged()
{
   Base::OnBeforeNodeChanged();

#if defined(CANDERA_LAYOUT_ENABLED)
   m_setLayouter = true;
#endif
}


Candera::Vector2 MeshWidget2D::MeshWidget2DInterfaceAdapter::OnMeasure(const Candera::AbstractNodePointer& node, const Candera::Vector2& clientArea)
{
   Candera::Vector2 preferredSize;
   if (0 != GetReceiver())
   {
      Candera::Node2D* node2D = GetReceiver()->GetNode2D();
      Candera::RenderNode* renderNode = (0 != GetReceiver()->m_meshNode) ? GetReceiver()->m_meshNode : Candera::Dynamic_Cast<Candera::RenderNode*>(node2D);
      if (0 != renderNode)
      {
         Candera::Effect2D* effect = renderNode->GetEffect(0);
         if (0 != effect)
         {
            Candera::Vector3 mappedMesh[4];
            MeshWidget2DUpdateMesh(Candera::Vector2(-1.0F, -1.0F), effect, *GetReceiver(), mappedMesh);
            FeatStd::Float xMin = Candera::Math::MaxFloat();
            FeatStd::Float xMax = -Candera::Math::MaxFloat();
            FeatStd::Float yMin = Candera::Math::MaxFloat();
            FeatStd::Float yMax = -Candera::Math::MaxFloat();
            MeshWidget2DGetBoundaries(mappedMesh, xMin, xMax, yMin, yMax);
            if (LayoutAlignment::StretchBehavior::None == Candera::Layouter::GetStretchBehavior(*node.ToCanderaObject()))
            {
               preferredSize.SetX(xMax);
               preferredSize.SetY(yMax);
            }
            else
            {
               preferredSize.SetX(xMax - xMin);
               preferredSize.SetY(yMax - yMin);
            }
         }
      }
   }

   Candera::Vector2 childrenPreferredSize = OverlayLayouter::OnMeasure(node, clientArea);
   if (childrenPreferredSize.GetX() > preferredSize.GetX())
   {
      preferredSize.SetX(childrenPreferredSize.GetX());
   }
   if (childrenPreferredSize.GetY() > preferredSize.GetY())
   {
      preferredSize.SetY(childrenPreferredSize.GetY());
   }
   return preferredSize;
}


void MeshWidget2D::MeshWidget2DInterfaceAdapter::OnArrange(const Candera::AbstractNodePointer& node, const Candera::Rectangle& clientArea)
{
   if (0 != GetReceiver() && (0 != node.ToCanderaObject()))
   {
      Candera::Node2D* node2D = GetReceiver()->GetNode2D();
      Candera::RenderNode* renderNode = (0 != GetReceiver()->m_meshNode) ? GetReceiver()->m_meshNode : Candera::Dynamic_Cast<Candera::RenderNode*>(node2D);
      if (0 != renderNode)
      {
         Candera::Effect2D* effect = renderNode->GetEffect(0);
         if (0 != effect)
         {
            Candera::Vector3 mappedMesh[4];
            LayoutAlignment::StretchBehavior::Enum stretchBehavior = Candera::Layouter::GetStretchBehavior(*node.ToCanderaObject());
            MeshWidget2DUpdateMesh(Candera::Vector2(-1.0F, -1.0F), effect, *GetReceiver(), mappedMesh);
            if (LayoutAlignment::StretchBehavior::None != stretchBehavior)
            {
               FeatStd::Float xMin = Candera::Math::MaxFloat();
               FeatStd::Float xMax = -Candera::Math::MaxFloat();
               FeatStd::Float yMin = Candera::Math::MaxFloat();
               FeatStd::Float yMax = -Candera::Math::MaxFloat();
               MeshWidget2DGetBoundaries(mappedMesh, xMin, xMax, yMin, yMax);
               FeatStd::Float scaleX = clientArea.GetWidth() / (xMax - xMin);
               FeatStd::Float scaleY = clientArea.GetHeight() / (yMax - yMin);
               switch (stretchBehavior)
               {
                  case LayoutAlignment::StretchBehavior::Uniform:
                     if (scaleX > scaleY)
                     {
                        scaleX = scaleY;
                     }
                     else
                     {
                        scaleY = scaleX;
                     }
                     break;
                  case LayoutAlignment::StretchBehavior::UniformToFill:
                     if (scaleX < scaleY)
                     {
                        scaleX = scaleY;
                     }
                     else
                     {
                        scaleY = scaleX;
                     }
                     break;
                  default:
                     break;
               }
               for (FeatStd::SizeType i = 0; i < 4; ++i)
               {
                  mappedMesh[i].SetX((mappedMesh[i].GetX() - xMin) * scaleX);
                  mappedMesh[i].SetY((mappedMesh[i].GetY() - yMin) * scaleY);
               }
            }
            for (FeatStd::SizeType i = 0; i < 4; ++i)
            {
               GetReceiver()->m_vertices[i].Position.SetX(mappedMesh[i].GetX());
               GetReceiver()->m_vertices[i].Position.SetY(mappedMesh[i].GetY());
            }
            if (0 == GetReceiver()->m_geometry)
            {
               GetReceiver()->m_indices[0] = 0;
               GetReceiver()->m_indices[1] = 1;
               GetReceiver()->m_indices[2] = 2;
               GetReceiver()->m_indices[3] = 2;
               GetReceiver()->m_indices[4] = 1;
               GetReceiver()->m_indices[5] = 3;
               GetReceiver()->m_vertices[0].TextureCoordinate.SetX(0.0F);
               GetReceiver()->m_vertices[0].TextureCoordinate.SetY(1.0F);
               GetReceiver()->m_vertices[1].TextureCoordinate.SetX(1.0F);
               GetReceiver()->m_vertices[1].TextureCoordinate.SetY(1.0F);
               GetReceiver()->m_vertices[2].TextureCoordinate.SetX(0.0F);
               GetReceiver()->m_vertices[2].TextureCoordinate.SetY(0.0F);
               GetReceiver()->m_vertices[3].TextureCoordinate.SetX(1.0F);
               GetReceiver()->m_vertices[3].TextureCoordinate.SetY(0.0F);
               GetReceiver()->m_geometry = FEATSTD_NEW(Candera::VertexGeometry2D)(&(GetReceiver()->m_vertices[0]), 0, &(GetReceiver()->m_indices[0]), 0, 4, 6, Candera::VertexGeometry2D::ClientMemory, Candera::VertexGeometry2D::IndexedArrayBuffer, Candera::VertexGeometry2D::DynamicWrite);
               Candera::VertexBuffer2D::SharedPointer vertexBuffer = Candera::VertexBuffer2D::Create();
               if ((0 != GetReceiver()->m_geometry) && (!vertexBuffer.PointsToNull()))
               {
                  vertexBuffer->SetVertexGeometry2D(GetReceiver()->m_geometry, 0);
                  GetReceiver()->m_meshNode = Candera::Mesh2D::Create();
                  if (0 != GetReceiver()->m_meshNode)
                  {
                     GetReceiver()->m_meshNode->SetVertexBuffer2D(vertexBuffer);

                     Courier::View* view = GetReceiver()->GetParentView();
                     if (0 != view)
                     {
                        Courier::ViewScene2D* viewScene2D = view->ToViewScene2D();
                        if (0 != viewScene2D)
                        {
                           Courier::ViewScene2D::CameraPtrVector& cameras = viewScene2D->GetCameraPtrVector();
                           for (FeatStd::SizeType i = 0; i < cameras.Size(); ++i)
                           {
                              Candera::RenderTarget* renderTarget = cameras[i]->GetRenderTarget();
                              if (0 != renderTarget)
                              {
                                 static_cast<void>(renderTarget->Activate());
                              }
                              GetReceiver()->m_meshNode->Upload();
                           }
                        }
                     }
                     else // required for SceneComposer
                     {
                        GetReceiver()->m_meshNode->Upload();
                     }

                     if (0 != node2D)
                     {
                        node2D->AddChild(GetReceiver()->m_meshNode);
                        GetReceiver()->m_meshNode->AddEffect(effect);
                        // RemoveEffect causes a reset of the layouter to the default layouter
                        renderNode->RemoveEffect(effect);
                        node2D->SetLayouter(&GetReceiver()->m_interfaceAdapter);
                     }
                  }
               }
            }
         }
      }
   }
}


FeatStd::Float MeshWidget2D::NormalizeDegree(FeatStd::Float value)
{
   if (value < 0.0F)
   {
      return 360.0F - NormalizeDegree(-value);
   }
   if (value < 360.0F)
   {
      return value;
   }
   FeatStd::Float normalizedValue = FeatStd::Float(FeatStd::UInt(value) % FeatStd::UInt(360)) + (value - FeatStd::Float(FeatStd::UInt(value)));
   if (normalizedValue > 360.0F)
   {
      return normalizedValue - 360.0F;
   }
   return normalizedValue;
}


} // namespace mesh
} // namespace widget
} // namespace hmibase
