/* ***************************************************************************************
* FILE:          SwipeCalculator.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SwipeCalculator is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "SwipeCalculator.h"

#include <Candera/System/Mathematics/Math.h>
#include <FeatStd/Platform/PerfCounter.h>
#include <TouchGestures/GestureBasetypes.h>
#include <hmibase/util/Ticker.h>
#include <Widgets/2D/Common/FlexScrollable.h>


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/SwipeCalculator.cpp.trc.h"
#endif


using namespace FeatStd;
using namespace Candera;

static Float Signum(const Float x)
{
   return (x < 0) ? -1.0F : 1.0F;
}


SwipeCalculator::SwipeCalculator(ItemSpeedAnimationManager& itemSpeedAnimationManager) :
   _scrollable(0),
   _timeDispatcher(0),
   _swipingAdjuster(0),
   _listener(0),
   _startPosition(0.0F),
   _lastScrollPosition(),
   _detectDirectionPosition(0.0F),
   _startTime(0),
   _velocity(0.0F),
   _lastScrollTime(0.0F),
   _isVelocityAdjustable(false),
   _swipingAcceleration(0.0F),
   _maxSwipingDistance(0.0F),
   _remainingSwipeDistance(0.0F),
   _state(SwipingState::Idle),
   _previousState(SwipingState::Idle),
   _startSwiping(false),
   _itemSpeedAnimationManager(itemSpeedAnimationManager)
{
}


SwipeCalculator::~SwipeCalculator()
{
   _scrollable = 0;
   _listener = 0;
   _timeDispatcher = 0;
   _swipingAdjuster = 0;
}


void SwipeCalculator::Init(FlexScrollable* scrollable, TimeDispatcher* timeDispatcher, SwipingAdjuster* swipingAdjuster)
{
   _scrollable = scrollable;
   _timeDispatcher = timeDispatcher;
   _swipingAdjuster = swipingAdjuster;
}


bool SwipeCalculator::Update()
{
   if (SwipingState::Swiping == _state || _startSwiping)
   {
      OnSwipe(GetTimeNow());

      if (!_isVelocityAdjustable)
      {
         OnSwipeEnd();
      }

      _startSwiping = false;
   }

   return (SwipingState::Swiping == _state);
}


void SwipeCalculator::SetSwipingAcceleration(const FeatStd::Float& swipingAcceleration)
{
   _swipingAcceleration = swipingAcceleration;
}


void SwipeCalculator::SetMaxSwipingDistance(FeatStd::Float val)
{
   _maxSwipingDistance = (Math::FloatAlmostEqual(val, 0.0f) ? Math::MaxFloat() : Math::Absolute(val));
}


bool SwipeCalculator::OnDragGesture(const hmibase::input::gesture::GestureEvent& gestureData)
{
   if (!(ValidPoint(gestureData) && ValidDirection(gestureData)))
   {
      return false;
   }

   Vector2 touchPosition(static_cast<Float>(gestureData._pt1.x), static_cast<Float>(gestureData._pt1.y));
   if (0 != _listener)
   {
      switch (gestureData._gestureState)
      {
         case hmibase::input::gesture::GestureEvent::ET_START:
            OnGestureBegin(touchPosition);
            OnScrollStart(touchPosition, gestureData._nTimestamp);
            break;
         case hmibase::input::gesture::GestureEvent::ET_MOVE:
            OnScroll(touchPosition);
            break;
         case hmibase::input::gesture::GestureEvent::ET_END:
            OnScrollEnd(touchPosition);
            break;
         default:
            break;
      }
   }

   return true;
}


bool SwipeCalculator::OnSwipeGesture(const hmibase::input::gesture::GestureEvent& gestureData)
{
   if (!(gestureData._gestureState == hmibase::input::gesture::GestureEvent::ET_END && ValidDirection(gestureData) && ValidVelocity(gestureData)))
   {
      return false;
   }

   _startTime = gestureData._nTimestamp;
   _velocity = GetVelocity(gestureData._velocity1) / 1000;
   _isVelocityAdjustable = true;

   bool nextState(true);
   Vector2 swipeStartPos(static_cast<Float>(gestureData._pt1.x), static_cast<Float>(gestureData._pt1.y));

   if ((SwipingState::Scrolling == _state) && (0 != _listener))
   {
      nextState = _listener->OnScrollEnd(_lastScrollPosition);
   }
   else
   {
      OnGestureBegin(swipeStartPos);
   }

   if (nextState)
   {
      OnSwipeStart(swipeStartPos);
   }
   else
   {
      Reset();
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SwipeCalculator::OnSwipeGesture velocity = %f", _velocity));

   return true;
}


bool SwipeCalculator::IsIdle() const
{
   return (SwipingState::Idle == _state);
}


bool SwipeCalculator::IsTouched() const
{
   return SwipingState::Touched == _state;
}


bool SwipeCalculator::IsFocused() const
{
   return SwipingState::Focused == _state;
}


bool SwipeCalculator::IsScrolling() const
{
   return SwipingState::Scrolling == _state;
}


bool SwipeCalculator::WasScrolling() const
{
   return (SwipingState::Scrolling == _previousState);
}


void SwipeCalculator::RegisterListener(SwipingListener* listener)
{
   _listener = listener;
}


void SwipeCalculator::OnGestureBegin(const Candera::Vector2& position)
{
   if (0 != _listener)
   {
      _listener->OnGestureBegin(position);
   }
}


void SwipeCalculator::OnScrollStart(const Candera::Vector2& touchPosition, int /*timestamp*/)
{
   _startPosition = GetPosition(touchPosition);
   _detectDirectionPosition = _startPosition;

   bool nextState(true);

   if (0 != _listener)
   {
      nextState = _listener->OnScrollStart(touchPosition);
   }

   if (!nextState)
   {
      Reset();
   }
}


void SwipeCalculator::OnScroll(const Candera::Vector2& touchPosition)
{
   if (_state != SwipingState::Scrolling)
   {
      SetState(SwipingState::Scrolling);
   }

   FeatStd::Float pos = GetPosition(touchPosition);
   FeatStd::Float currentTime = GetTimeNow();
   FeatStd::Float detectDirectionDistance = pos - _detectDirectionPosition;
   FeatStd::Float distance = GetDistance(pos);

   bool forward = true;
   if (detectDirectionDistance > 0.0f)
   {
      forward = false;
   }

   _detectDirectionPosition = pos;

   bool nextState(true);

   if (0 != _listener)
   {
      nextState = _listener->OnScroll(distance, forward);
   }
   _itemSpeedAnimationManager.RegisterSize(distance / (currentTime - _lastScrollTime), _state);

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SwipeCalculator::OnScroll distance = %f", distance));

   _startPosition = pos;
   _lastScrollPosition = touchPosition;
   _lastScrollTime = currentTime;

   if (!nextState)
   {
      Reset();
   }
}


void SwipeCalculator::OnScrollEnd(const Candera::Vector2& touchPosition)
{
   if (0 != _listener)
   {
      _listener->OnScrollEnd(touchPosition);
   }

   Reset();
   _startPosition = 0;
}


void SwipeCalculator::OnSwipeStart(const Candera::Vector2& touchPosition)
{
   if (_startPosition == 0)
   {
      _startPosition = GetPosition(touchPosition);
   }

   AdjustInitialSwipedDistance();

   bool nextState(true);

   if (0 != _listener)
   {
      nextState = _listener->OnSwipeStart(_remainingSwipeDistance);
   }

   if (!nextState)
   {
      Reset();
   }

   _startSwiping = true;
}


void SwipeCalculator::OnSwipe(int timestamp)
{
   if (_state != SwipingState::Swiping)
   {
      SetState(SwipingState::Swiping);
   }

   UInt32 deltaTime = GetTouchInterval(timestamp);

   if (0 != deltaTime)
   {
      AdjustSwipingParameters();
      Float initialVelocity = _velocity;
      Float distance = 0.0F;
      if (_velocity > 0)
      {
         distance = _velocity * deltaTime - _swipingAcceleration * deltaTime * deltaTime * 0.5f;
         _velocity = _velocity - _swipingAcceleration * deltaTime;
      }
      else
      {
         distance = _velocity * deltaTime + _swipingAcceleration * deltaTime * deltaTime * 0.5f;
         _velocity = _velocity + _swipingAcceleration * deltaTime;
      }

      _isVelocityAdjustable = ((initialVelocity * _velocity) > 0);
      if (!_isVelocityAdjustable)
      {
         distance = _remainingSwipeDistance;
      }

      bool forward = true;
      Candera::Float detectDirectionDistance = _startPosition + distance - _detectDirectionPosition;
      if (detectDirectionDistance > 0.0f)
      {
         forward = false;
      }

      _detectDirectionPosition = _startPosition + distance;

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SwipeCalculator::OnSwipe velocity = %f, forward = %d, distance = %f, remainingDistance = %f", _velocity, forward, distance, _remainingSwipeDistance));

      bool nextState(true);

      if ((0 != _listener) && _isVelocityAdjustable)
      {
         nextState = _listener->OnSwipe(distance, forward);
      }

      _itemSpeedAnimationManager.RegisterSize(_velocity, _state);
      if (!nextState)
      {
         Reset();
      }

      _startPosition += distance;
      _startTime += deltaTime;
   }
}


void SwipeCalculator::OnSwipeEnd()
{
   if (0 != _listener)
   {
      _listener->OnSwipeEnd();
   }

   Reset();
   _startPosition = 0;
}


void SwipeCalculator::Reset()
{
   SetState(SwipingState::Idle);
   _itemSpeedAnimationManager.RegisterSize(0, SwipingState::Idle);
}


Courier::Float SwipeCalculator::GetPosition(const Vector2& touchPoint) const
{
   if (0 == _scrollable)
   {
      return 0;
   }

   Courier::Float position = 0;

   switch (_scrollable->GetScrollingOrientation())
   {
      case Candera::Horizontal:
         position = touchPoint.GetX();
         break;
      case Candera::Vertical:
         position = touchPoint.GetY();
         break;
      default:
         position = 0;
         break;
   }

   return position;
}


bool SwipeCalculator::ValidPoint(const hmibase::input::gesture::GestureEvent& gestureData) const
{
   return (gestureData._isPoint1Valid);
}


bool SwipeCalculator::ValidDirection(const hmibase::input::gesture::GestureEvent& gestureData) const
{
   return ((0 != _scrollable && _scrollable->GetScrollingOrientation() == Candera::Horizontal) ? hmibase::input::gesture::GestureEvent::DIR_HORIZONTAL == gestureData._gestureDirection :
           hmibase::input::gesture::GestureEvent::DIR_VERTICAL == gestureData._gestureDirection);
}


bool SwipeCalculator::ValidVelocity(const hmibase::input::gesture::GestureEvent& gestureData) const
{
   return (gestureData._isPoint1VelocityValid);
}


FeatStd::Float SwipeCalculator::GetVelocity(const hmibase::input::gesture::Vector2D& velocity2D) const
{
   return ((0 != _scrollable && _scrollable->GetScrollingOrientation() == Candera::Horizontal) ? static_cast<FeatStd::Float>(velocity2D.x) : static_cast<FeatStd::Float>(velocity2D.y));
}


Candera::Float SwipeCalculator::GetDistance(Courier::Float pos) const
{
   return pos - _startPosition;
}


Candera::UInt32 SwipeCalculator::GetTouchInterval(Candera::UInt32 now) const
{
   if (now > _startTime)
   {
      return (now - _startTime);
   }
   else
   {
      return 0;
   }
}


FeatStd::UInt32 SwipeCalculator::GetTimeNow() const
{
   FeatStd::UInt32 now(0);

   if (0 != _timeDispatcher)
   {
      now = _timeDispatcher->GetTimeMs();
   }

   return now;
}


void SwipeCalculator::AdjustSwipingParameters()
{
   if ((0 != _swipingAdjuster) && (0 != _velocity))
   {
      Float velocitySignum(Signum(_velocity));
      const Float doubleAcceleration(2.0F * _swipingAcceleration);
      Float calculatedSwipingDistance(velocitySignum * _velocity * _velocity / doubleAcceleration);

      if (Math::Absolute(calculatedSwipingDistance) > _maxSwipingDistance)
      {
         calculatedSwipingDistance = velocitySignum * _maxSwipingDistance;
      }

      _remainingSwipeDistance = _swipingAdjuster->AdjustSwipedDistance(calculatedSwipingDistance);

      velocitySignum = Signum(_remainingSwipeDistance);

      _velocity = velocitySignum * Math::SquareRoot(doubleAcceleration * Math::Absolute(_remainingSwipeDistance));
   }
}


void SwipeCalculator::AdjustInitialSwipedDistance()
{
   if ((0 != _swipingAdjuster) && (0 != _velocity))
   {
      const Float velocitySignum(Signum(_velocity));
      const Float doubleAcceleration(2 * _swipingAcceleration);
      Float calculatedSwipingDistance(velocitySignum * _velocity * _velocity / doubleAcceleration);

      if (Math::Absolute(calculatedSwipingDistance) > _maxSwipingDistance)
      {
         calculatedSwipingDistance = velocitySignum * _maxSwipingDistance;
      }

      _remainingSwipeDistance = _swipingAdjuster->AdjustInitialSwipedDistance(calculatedSwipingDistance);

      _velocity = velocitySignum * Math::SquareRoot(doubleAcceleration * Math::Absolute(_remainingSwipeDistance));
   }
}


void SwipeCalculator::SetState(SwipingState::Enum state)
{
   _previousState = _state;
   _state = state;

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Swiping: state changed from %d to %d", _previousState, _state));

   if (0 != _listener)
   {
      _listener->OnStateChanged(state);
   }
}
