/* ***************************************************************************************
* FILE:          PageSwipingBehavior.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PageSwipingBehavior is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include <Widgets/2D/List/generated/ListMessages.h>
#include "widget2D_std_if.h"
#include "PageSwipingBehavior.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/PageSwipingBehavior.cpp.trc.h"
#endif

using namespace FeatStd;
using namespace Candera;
using namespace Courier;

PageSwipingBehavior::PageSwipingBehavior(Swiper& swiper, ListWidget2D& listWidget, const OrientationOperator& orientationOperator, const PageSwipingBehaviorInfo& info) :
   _swiper(swiper),
   _listWidget(listWidget),
   _orientationOperator(orientationOperator),
   _info(info),
   _initialStartIndex(0),
   _previousPageStartIndex(0),
   _nextPageStartIndex(0)
{
}


void PageSwipingBehavior::BeginGestureImpl(const Candera::Vector2& position)
{
   _initialStartIndex = _listWidget.GetStartIndex();
   _scrollStartPosition = position;
   const PagesInfo& pagesInfo(_listWidget.GetPagesInfo());
   Int32 pageIndex;

   pagesInfo.RetrieveStartIndexOfPage(0, _initialStartIndex);

   _previousPageStartIndex = _initialStartIndex;
   _nextPageStartIndex = _initialStartIndex;

   if (pagesInfo.RetrieveStartIndexOfPage(-1, pageIndex))
   {
      _previousPageStartIndex = pageIndex;
   }

   if (pagesInfo.RetrieveStartIndexOfPage(1, pageIndex))
   {
      _nextPageStartIndex = pageIndex;
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PageSwipingBehavior::BeginGestureImpl pos = (%f, %f), startIndex = %d", position.GetX(), position.GetY(), _initialStartIndex));
}


void PageSwipingBehavior::StartScrollImpl(const Candera::Vector2& pos)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PageSwipingBehavior::StartScrollImpl pos = (%f, %f), startIndex = %d", pos.GetX(), pos.GetY(), _initialStartIndex));
}


bool PageSwipingBehavior::ScrollImpl(FeatStd::Float movementDistance, bool forward)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PageSwipingBehavior::ScrollImpl movementDistance = %f, forward = %d", movementDistance, forward));
   bool scrolled(_swiper.ScrollList(movementDistance, forward));

   return scrolled;
}


void PageSwipingBehavior::EndScrollImpl(const Candera::Vector2& pos)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PageSwipingBehavior::EndScrollImpl pos=(%f, %f)", pos.GetX(), pos.GetY()));
   _scrollEndPosition = pos;

   const Float& initialPos(_orientationOperator.GetVectorComponent(_scrollStartPosition));
   const Float& finalPos(_orientationOperator.GetVectorComponent(_scrollEndPosition));
   const bool forward(finalPos < initialPos);
   const Float& distance(Math::Absolute(finalPos - initialPos));

   ScrollFixPage(forward, distance);
}


void PageSwipingBehavior::StartSwipeImpl(const FeatStd::Float& estimatedDistance)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PageSwipingBehavior::StartSwipeImpl estimatedDistance = %f, startIndex = %d", estimatedDistance, _initialStartIndex));
   const Float& initialPos(_orientationOperator.GetVectorComponent(_scrollStartPosition));
   const Float& finalPos(_orientationOperator.GetVectorComponent(_scrollEndPosition));
   const bool forward(estimatedDistance < 0);
   const Float& distance(Math::Absolute(finalPos - initialPos + estimatedDistance));

   if (estimatedDistance != 0)
   {
      ScrollFixPage(forward, distance);
   }

   RequestIdle();
}


bool PageSwipingBehavior::SwipeImpl(FeatStd::Float /*movementDistance*/, bool /*forward*/)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PageSwipingBehavior::SwipeImpl"));
   return true;
}


bool PageSwipingBehavior::ShouldSnapImpl() const
{
   return false;
}


void PageSwipingBehavior::ScrollFixPage(const bool forward, const FeatStd::Float distance)
{
   const Float& visibleSize(_orientationOperator.GetVectorComponent(_info.GetVisibleArea()));
   const FeatStd::Float scrollOffsetPercentage(_info.GetScrollOffsetPercentage());
   const bool thresholdMet(_info.IsThresholdMet(_orientationOperator, distance));

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PageSwipingBehavior::ScrollFixPage forward = %d, distance = %f, visibleSize = %f, offsetPercentage = %f", forward, distance, visibleSize, scrollOffsetPercentage));

   if (thresholdMet)
   {
      _listWidget.SetStartIndex(forward ? _nextPageStartIndex : _previousPageStartIndex);
      _listWidget.Invalidate();
   }
   else
   {
      Int32 startIndex(_initialStartIndex);
      if ((startIndex == 0) && (_listWidget.IsCircular()) && (!forward))
      {
         startIndex = _listWidget.GetMaxPosition();
      }
      _listWidget.SetStartIndex(startIndex);
      _listWidget.Invalidate();
   }
}
