/* ***************************************************************************************
* FILE:          DefaultOverscroller.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  DefaultOverscroller is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"

#include <Candera/Engine2D/Layout/Layouter.h>
#include "DefaultOverscroller.h"


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/DefaultOverscroller.cpp.trc.h"
#endif


using namespace Candera;
using namespace FeatStd;

DefaultOverscroller::DefaultOverscroller(const OrientationOperator& orientationOperator, TimeDispatcher& timeDispatcher) :
   _orientationOperator(orientationOperator),
   _timeDispatcher(timeDispatcher),
   _contentNode(0),
   _viewportSize(0.0F),
   _viewportRatio(0.0F),
   _viewportRatioInverse(0.0F),
   _function(0),
   _swipeOverscrollDistance(0.0F),
   _movedDistance(0.0F),
   _moving(false),
   _swiping(false),
   _overscrolledDistance(0.0F),
   _forward(false),
   _forwardStartedMovement(false),
   _relax(false),
   _listener(0),
   _shouldEnd(false),
   _relaxT0(0)
{
}


DefaultOverscroller::~DefaultOverscroller()
{
   _contentNode = 0;
   _function = 0;
   _listener = 0;
}


void DefaultOverscroller::SetViewportSize(Candera::Vector2 viewportSize)
{
   _viewportSize = _orientationOperator.GetVectorComponent(viewportSize);
   _viewportRatio = 0.001F * _viewportSize;
   if (_viewportSize != 0.0F)
   {
      _viewportRatioInverse = 1000.0F / _viewportSize;
   }
   else
   {
      _viewportRatioInverse = 0.0F;
   }
}


void DefaultOverscroller::SetListener(OverscrollerListener* listener)
{
   _listener = listener;
}


void DefaultOverscroller::Move(FeatStd::Float movingDistance, bool forward)
{
   if (_swiping && (0 != _contentNode))
   {
      if (!_moving)
      {
         _forwardStartedMovement = forward;
      }
      _moving = true;

      const Float alreadyMovedDistance(_movedDistance);
      const Float previousSwipeOverscrollDistance(_swipeOverscrollDistance);

      _swipeOverscrollDistance -= movingDistance;
      _movedDistance += movingDistance;

      if ((forward != _forwardStartedMovement) && ((alreadyMovedDistance * _movedDistance) <= 0.0F))
      {
         _movedDistance = 0.0F;
      }

      if ((previousSwipeOverscrollDistance * _swipeOverscrollDistance) <= 0.0F)
      {
         _swipeOverscrollDistance = 0.0F;
      }
   }
}


FeatStd::Float DefaultOverscroller::GetMovedDistance() const
{
   return _movedDistance;
}


void DefaultOverscroller::SetInitialSwipeOverscrollDistance(FeatStd::Float distance)
{
   if (IsSwipeAllowed())
   {
      const Float signum((distance < 0.0F) ? -1.0F : 1.0F);

      if ((signum * distance) > _viewportSize)
      {
         distance = signum * _viewportSize;
      }

      _swipeOverscrollDistance = distance;
   }
   else
   {
      _swipeOverscrollDistance = 0.0F;
   }
}


FeatStd::Float DefaultOverscroller::GetExtraDistanceForSwipe() const
{
   return _swipeOverscrollDistance;
}


bool DefaultOverscroller::Update(bool idleRequested)
{
   bool invalidate(false);
   _shouldEnd = _shouldEnd || idleRequested;
   UpdateMovementEnd();

   if (_moving)
   {
      Overscroll();
      UpdateMargin();
      invalidate = true;
   }
   else if (_relax)
   {
      Relax();
      UpdateMargin();
      invalidate = true;
   }
   else
   {
      // do nothing
   }

   return invalidate;
}


void DefaultOverscroller::BeginGesture(const Candera::Vector2& position)
{
   _relax = false;
   _moving = false;
   _swiping = true;
   _shouldEnd = false;
   _movedDistance = 0.0F;
}


void DefaultOverscroller::StartScroll(const Candera::Vector2& pos)
{
   _shouldEnd = !IsDragAllowed();
}


void DefaultOverscroller::EndScroll(const Candera::Vector2& pos)
{
   _shouldEnd = true;
}


void DefaultOverscroller::StartSwipe(const Candera::Float estimatedDistance)
{
   _shouldEnd = !IsSwipeAllowed();
}


void DefaultOverscroller::EndSwipe()
{
   _shouldEnd = true;
}


FeatStd::Float DefaultOverscroller::Overscroll()
{
   Float viewportPerMile(0);
   if (0 != _function)
   {
      viewportPerMile = _function->Overscroll(_movedDistance * _viewportRatioInverse);
      _overscrolledDistance = _viewportRatio * viewportPerMile;
   }

   return viewportPerMile;
}


FeatStd::Float DefaultOverscroller::Relax()
{
   Float viewportPerMile(0);
   if (0 != _function)
   {
      UInt32 now(_timeDispatcher.GetTimeMs());
      _relax = _function->Relax(_overscrolledDistance * _viewportRatioInverse, now - _relaxT0, viewportPerMile);

      _overscrolledDistance = _viewportRatio * viewportPerMile;
   }

   return viewportPerMile;
}


void DefaultOverscroller::UpdateMargin()
{
   Margin margin(Layouter::GetMargin(*_contentNode));
   Int16 previousMargin(_orientationOperator.GetValue(margin.GetLeft(), margin.GetTop()));
   Int16 previousMarginOrthogonal(_orientationOperator.GetValue(margin.GetTop(), margin.GetLeft()));

   previousMargin += _overscrolledDistance;

   _orientationOperator.SetMargin(*_contentNode, previousMargin, previousMarginOrthogonal, true, false);

   NotifyListener();
}


void DefaultOverscroller::UpdateMovementEnd()
{
   if (_shouldEnd)
   {
      if (_moving)
      {
         _relax = true;
         _relaxT0 = _timeDispatcher.GetTimeMs();
      }
      _moving = false;
      _swiping = false;
   }
}


void DefaultOverscroller::NotifyListener()
{
   if (0 != _listener)
   {
      const Float p((_viewportSize > 0.0F) ? Math::Absolute(_overscrolledDistance / _viewportSize) : 0.0F);
      if (_overscrolledDistance > 0)
      {
         _listener->OnPreOverscroll(p);
      }
      else
      {
         _listener->OnPostOverscroll(p);
      }
   }
}


bool DefaultOverscroller::IsDragAllowed() const
{
   return (0 != _function) && (_function->IsOnDragEnabled());
}


bool DefaultOverscroller::IsSwipeAllowed() const
{
   return (0 != _function) && (_function->IsOnSwipeEnabled());
}
