/* ***************************************************************************************
* FILE:          ItemSnapperHelper.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ItemSnapperHelper is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"
#include "ItemSnapperHelper.h"
#include <Candera/System/Container/Vector.h>

#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
#include <Candera/EngineBase/Layout/Layouter.h>
#else
#include <Candera/Engine2D/Layout/Layouter.h>
#endif

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ItemSnapperHelper.cpp.trc.h"
#endif
using namespace Candera;

FeatStd::Float ItemSnapperHelper::GetDeltaToCenter(const ItemSnapper::SnappingInfo& snappingInfo) const
{
   bool found = false;
   FeatStd::UInt8 cnt = 0;
   FeatStd::Float centerPos = snappingInfo._viewportSize / 2;
   FeatStd::Float delta = snappingInfo._viewportSize;

   ItemSnapper::ItemsInfoType effectiveSizes;
   ItemSnapper::ItemsInfoType effectivePositions;
   GetEffectiveNodeSizes(snappingInfo, effectiveSizes);
   GetEffectiveNodePositions(snappingInfo, true, effectivePositions);

   while ((!found) && (cnt < snappingInfo._nodeSizes.Size()))
   {
      Float itemPos = effectivePositions[cnt];
      Float itemSize = effectiveSizes[cnt];
      Float itemCenterPos =  itemPos + itemSize / 2;

      FeatStd::Float newDelta = centerPos - itemCenterPos;

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ItemSnapperHelper::GetDeltaToCenter delta = %f, itemPos=%f, itemSize=%f, itemCenterPos=%f, newDelta=%f", delta, itemPos, itemSize, itemCenterPos, newDelta));

      found = (Math::Absolute(newDelta) > Math::Absolute(delta));
      if (!found)
      {
         delta = newDelta;
      }

      cnt++;
   }

   return delta;
}


FeatStd::Int32 ItemSnapperHelper::GetCenterItemIndex(const ItemSnapper::SnappingInfo& snappingInfo) const
{
   bool found = false;
   FeatStd::SizeType cnt = 0;
   FeatStd::SizeType centerIndex = 0;
   FeatStd::Float centerPos = snappingInfo._viewportSize / 2;
   FeatStd::Float delta = snappingInfo._viewportSize;

   ItemSnapper::ItemsInfoType effectiveSizes;
   ItemSnapper::ItemsInfoType effectivePositions;
   GetEffectiveNodeSizes(snappingInfo, effectiveSizes);
   GetEffectiveNodePositions(snappingInfo, true, effectivePositions);

   while ((!found) && (cnt < snappingInfo._nodeSizes.Size()))
   {
      Float itemPos = effectivePositions[cnt];
      Float itemSize = effectiveSizes[cnt];
      Float itemCenterPos = itemPos + itemSize / 2;

      FeatStd::Float newDelta = centerPos - itemCenterPos;

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ItemSnapperHelper::GetDeltaToCenter delta = %f, itemPos=%f, itemSize=%f, itemCenterPos=%f, newDelta=%f", delta, itemPos, itemSize, itemCenterPos, newDelta));

      found = (Math::Absolute(newDelta) > Math::Absolute(delta));
      if (!found)
      {
         delta = newDelta;
      }
      else
      {
         centerIndex = cnt - 1;
      }

      cnt++;
   }

   return centerIndex;
}


void ItemSnapperHelper::GetEffectiveNodePositions(const ItemSnapper::SnappingInfo& snappingInfo, bool considerChildMargins, Candera::Internal::Vector<FeatStd::Float>& effectivePositions) const
{
   const Margin& margin = Layouter::GetMargin(snappingInfo._itemsNode);
   for (int i = 0; i < (int)snappingInfo._nodePositions.Size(); i++)
   {
      effectivePositions.Add(snappingInfo._nodePositions[i] + snappingInfo._orientationOperator.GetVectorComponent(Vector2(margin.GetLeft(), margin.GetTop())));
   }

   if (considerChildMargins)
   {
      Candera::Node2D* child(snappingInfo._itemsNode.GetFirstChild());
      FeatStd::UInt8 i = 0;
      while (0 != child)
      {
         if (child->GetFirstChild() != 0)
         {
            const Margin& margin = Layouter::GetMargin(const_cast<Node2D&>(*child->GetFirstChild()));
            effectivePositions[i] += snappingInfo._orientationOperator.GetVectorComponent(Vector2(margin.GetLeft(), margin.GetTop()));
            i++;
         }

         child = child->GetNextSibling();
      }
   }
}


void ItemSnapperHelper::GetEffectiveNodeSizes(const ItemSnapper::SnappingInfo& snappingInfo, Candera::Internal::Vector<FeatStd::Float>& effectiveSizes) const
{
   Candera::Node2D* child(snappingInfo._itemsNode.GetFirstChild());
   FeatStd::UInt8 i = 0;
   while (0 != child)
   {
      if (child->GetFirstChild() != 0)
      {
         const Margin& margin = Layouter::GetMargin(const_cast<Node2D&>(*child->GetFirstChild()));
         effectiveSizes.Add(snappingInfo._nodeSizes[i] - snappingInfo._orientationOperator.GetVectorComponent(Vector2(Float(margin.GetLeft() + margin.GetRight()), Float(margin.GetTop() + margin.GetBottom()))));
         i++;
      }
      child = child->GetNextSibling();
   }
}
