/* ***************************************************************************************
* FILE:          CollapsingOverlayLayouter.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  CollapsingOverlayLayouter is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "CollapsingOverlayLayouter.h"

#if defined SESA_ARABIC_LAYOUT_FIX
#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
#include "Candera/EngineBase/Layout/ArabicLayouterPatch.h"
#else
#include "Candera/Engine2D/Layout/ArabicLayouterPatch.h"
#endif
#endif


Candera::Vector2 CollapsingOverlayLayouter::OnMeasure(Candera::Node2D& node, const Candera::Vector2& clientArea)
{
   Candera::Float width = 0.0F;
   Candera::Float height = 0.0F;
   Candera::Vector2 childArea = (node.IsEffectiveRenderingEnabled()) ? clientArea : Candera::Vector2();
   Candera::Node2D* child = node.GetFirstChild();
   while (child != 0)
   {
      Layouter* childLayouter = child->GetLayouter();
      if (childLayouter != 0)
      {
         childLayouter->Measure(*child, childArea);
         Candera::Vector2 preferredChildSize = childLayouter->GetClientSize(*child);
         if (width < preferredChildSize.GetX())
         {
            width = preferredChildSize.GetX();
         }
         if (height < preferredChildSize.GetY())
         {
            height = preferredChildSize.GetY();
         }
      }
      child = child->GetNextSibling();
   }
   return Candera::Vector2(width, height);
}


#if defined SESA_ARABIC_LAYOUT_FIX
void CollapsingOverlayLayouter::OnArrange(Candera::Node2D& node, const Candera::Rectangle& clientArea)
{
   if (Candera::ArabicLayouterPatch::IsSceneEnabled(node))
   {
      Candera::Float clientWidth = clientArea.GetWidth();
      Candera::Float clientHeight = clientArea.GetHeight();
      Candera::Float actualWidth = 0.0F;
      Candera::Float actualHeight = 0.0F;
      if (!node.IsEffectiveRenderingEnabled())
      {
         clientWidth = 0.0F;
         clientHeight = 0.0F;
      }
      Candera::Node2D* child = node.GetFirstChild();
      while (child != 0)
      {
         Candera::Layouter* childLayouter = child->GetLayouter();
         if (childLayouter != 0)
         {
            Candera::Rectangle childArea(0.0F, 0.0F, clientWidth, clientHeight);
            childLayouter->Arrange(*child, childArea);
            const Candera::Margin& margin = Layouter::GetMargin(*child);
            Candera::Vector2 topLeftMargin(static_cast<Candera::Float>((GetLanguageSensitiveDirection(*child) == LayoutAlignment::LayoutDirection::RightToLeftDirection) ? margin.GetRight() : margin.GetLeft()), margin.GetBottom());
            Candera::Vector2 childSize = GetArrangeActualSize() + child->GetPosition() - topLeftMargin;
            if (actualWidth < childSize.GetX())
            {
               actualWidth = childSize.GetX();
            }
            if (actualHeight < childSize.GetY())
            {
               actualHeight = childSize.GetY();
            }
         }

         child = child->GetNextSibling();
      }
      SetArrangeActualSize((!node.IsEffectiveRenderingEnabled()) ? Candera::Vector2() : Candera::Vector2(actualWidth, actualHeight));
   }
   else
   {
      Candera::Float clientWidth = clientArea.GetWidth();
      Candera::Float clientHeight = clientArea.GetHeight();
      if (!node.IsEffectiveRenderingEnabled())
      {
         clientWidth = 0.0F;
         clientHeight = 0.0F;
      }
      Candera::Node2D* child = node.GetFirstChild();
      while (child != 0)
      {
         Layouter* childLayouter = child->GetLayouter();
         if (childLayouter != 0)
         {
            Candera::Rectangle childArea(0.0F, 0.0F, clientWidth, clientHeight);
            AlignRectangle(childArea, childArea, *child);
            childLayouter->Arrange(*child, childArea);
         }
         child = child->GetNextSibling();
      }
      node.SetPosition(clientArea.GetPosition());
   }
}


#else
void CollapsingOverlayLayouter::OnArrange(Candera::Node2D& node, const Candera::Rectangle& clientArea)
{
   Candera::Float clientWidth = clientArea.GetWidth();
   Candera::Float clientHeight = clientArea.GetHeight();
   if (!node.IsEffectiveRenderingEnabled())
   {
      clientWidth = 0.0F;
      clientHeight = 0.0F;
   }
   Candera::Node2D* child = node.GetFirstChild();
   while (child != 0)
   {
      Layouter* childLayouter = child->GetLayouter();
      if (childLayouter != 0)
      {
         Candera::Rectangle childArea(0.0F, 0.0F, clientWidth, clientHeight);
         AlignRectangle(childArea, childArea, *child);
         childLayouter->Arrange(*child, childArea);
      }
      child = child->GetNextSibling();
   }
   node.SetPosition(clientArea.GetPosition());
}


#endif

CollapsingOverlayLayouter* CollapsingOverlayLayouter::GetInstance()
{
   FEATSTD_UNSYNCED_STATIC_OBJECT(CollapsingOverlayLayouter, s_collapsingOverlayLayouter);
   return &s_collapsingOverlayLayouter;
}
