/* ***************************************************************************************
* FILE:          ListDataWrapper.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ListDataWrapper is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "Widgets/2D/BaseWidget2D.h"
#include "ListDataWrapper.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ListDataWrapper.cpp.trc.h"
#endif


using namespace Candera;


tSharedPtrDataProvider ListDataWrapper::s_emptyProvider = ListDataProvider::newBackEndInterface(0, 0, 0, 0);
tSharedPtrListDataItem ListDataWrapper::s_emptyItem(0);

ListDataWrapper::ListDataWrapper() :
   m_numberOfItems(0),
   m_bufferSize(0),
   m_scrollingType(Candera::DefaultScrolling),
   m_virtualListSize(0),
   m_startIndex(0),
   m_dataProvider(s_emptyProvider)
{
}


ListDataWrapper::~ListDataWrapper()
{
}


void ListDataWrapper::Clear()
{
   m_numberOfItems = 0;
   m_bufferSize = 0;
   m_scrollingType = Candera::DefaultScrolling;
   m_virtualListSize = 0;
   m_startIndex = 0;
   m_dataProvider = s_emptyProvider;
}


bool ListDataWrapper::PostListDataRequestMsg(UInt32 listId, UInt32 startIndex, UInt32 bufferSize, UInt32 numberOfItems, Candera::ListScrollingType scrollingType, bool isDataCached, bool& requestPending)
{
   Int32 si(startIndex);

   m_numberOfItems = numberOfItems;
   m_bufferSize = bufferSize;
   m_scrollingType = scrollingType;

   const bool isCircularScrolling = (scrollingType == Candera::CircularScrolling);
   UInt32 windowSize = numberOfItems + bufferSize + bufferSize;

   const Candera::UInt32 virtualListSize = GetVirtualListSize();
   if (IsDataAvailable() && (windowSize > virtualListSize))
   {
      si = 0;
      windowSize = virtualListSize;
   }
   else if (isCircularScrolling)
   {
      AdjustParametersCircular(si, windowSize);
   }
   else
   {
      AdjustParametersNonCircular(si, windowSize);
   }

   ListDateProviderReqMsg* msg = COURIER_MESSAGE_NEW(ListDateProviderReqMsg)(listId, si, windowSize, isCircularScrolling, isDataCached);
   if (0 != msg)
   {
      if (etg_bIsTraceActive(TR_CLASS_HMI_LISTPROVIDER, ETG_LEVEL_USER_1))
      {
         ETG_TRACE_USR1_CLS_DCL((TR_CLASS_HMI_LISTPROVIDER, APP_TRACECLASS_ID(),
                                 "ListDateProviderReqMsg resquest send for listId=%u, startIndex=%u, windowsize=%u circular=%s",
                                 listId, si, windowSize, isCircularScrolling ? "true" : "false"));
      }
      else
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(),
                             "ListDateProviderReqMsg resquest send for listId=%u, startIndex=%u, windowsize=%u circular=%s",
                             listId, si, windowSize, isCircularScrolling ? "true" : "false"));
      }
      static_cast<void>(msg->Post());
      requestPending = true;
   }

   return true;
}


void ListDataWrapper::AdjustParametersNonCircular(FeatStd::Int32& startIndex, FeatStd::UInt32& windowSize) const
{
   const Int32 bufferSize(static_cast<Int32>(m_bufferSize));
   Int32 dataStartIndex((startIndex <= bufferSize) ? (0) : (startIndex - bufferSize));

   if (IsDataAvailable())
   {
      const Candera::UInt32 virtualListSize(m_dataProvider->virtualListSize());

      // calculate delta to see if the last requested index is greater than the list size and
      // adjust the start index and windowSize accordingly
      const Int32 delta((dataStartIndex + static_cast<Int32>(windowSize)) - virtualListSize);
      if (delta > 0)
      {
         if (delta > dataStartIndex)
         {
            dataStartIndex = 0;
            windowSize = virtualListSize;
         }
         else
         {
            dataStartIndex -= delta;
         }
      }
   }

   startIndex = dataStartIndex;
}


void ListDataWrapper::AdjustParametersCircular(FeatStd::Int32& startIndex, FeatStd::UInt32& /*windowSize*/) const
{
   if (IsDataAvailable())
   {
      const Candera::UInt32 virtualListSize = m_dataProvider->virtualListSize();
      if ((startIndex + m_bufferSize > virtualListSize) && (virtualListSize > m_bufferSize))
      {
         startIndex -= virtualListSize;
      }
   }
   else
   {
      if (startIndex < 0)
      {
         startIndex = 0;
      }
   }

   startIndex -= m_bufferSize;
}


void ListDataWrapper::OnListDataResultMsg(const ListDateProviderResMsg& listDataResultMsg)
{
   tSharedPtrDataProvider newDataProvider(listDataResultMsg.GetListDateProvider());
   if ((newDataProvider.PointsToNull())
         || (newDataProvider->listId() == 0)
         // fr83hi: allow DataModel to send a dataprovider with empty content ( ListSize=0, windowelementsize=0 size=0)
         || ((newDataProvider->virtualListSize() == 0 && newDataProvider->windowElementSize() != 0))  // fr83hi: merged ??
         || ((newDataProvider->virtualListSize() != 0 && newDataProvider->windowElementSize() == 0))
         || ((m_scrollingType != Candera::CircularScrolling) && (newDataProvider->startIndex() > Int32(newDataProvider->virtualListSize()))))
   {
      ETG_TRACE_ERR(("FlexListWidget2D: -------------------ListDateProviderResMsg"));
      if (newDataProvider == NULL)
      {
         ETG_TRACE_ERR(("FlexListWidget2D:  received with NULL"));
      }
      else
      {
         if (newDataProvider.PointsToNull())
         {
            ETG_TRACE_ERR(("FlexListWidget2D:  PointsToNull"));
         }
         else
         {
            ETG_TRACE_ERR(("FlexListWidget2D:  received ListID       %d", newDataProvider->listId()));
            ETG_TRACE_ERR(("FlexListWidget2D:  received ListSize     %d", newDataProvider->listSize()));
            ETG_TRACE_ERR(("FlexListWidget2D:  received startIndex   %d", newDataProvider->startIndex()));
            ETG_TRACE_ERR(("FlexListWidget2D:  received focusIndex   %d", newDataProvider->getFocusIndex()));
            ETG_TRACE_ERR(("FlexListWidget2D:  received winElmSize   %d", newDataProvider->windowElementSize()));
            ETG_TRACE_ERR(("FlexListWidget2D:  received virtListSize %d", newDataProvider->virtualListSize()));
         }
      }

      m_dataProvider = s_emptyProvider;
   }
   else
   {
      m_startIndex = newDataProvider->startIndex();
      m_virtualListSize = newDataProvider->virtualListSize();
      m_dataProvider = newDataProvider;
   }
}


void ListDataWrapper::OnListDataUpdateMsg(const ListDataProviderUpdMsg& listDataUpdateMsg, ListDataProviderUpdater::ChangedItemsType& changedItems)
{
   if (!listDataUpdateMsg.GetUpdater().PointsToNull())
   {
      if (IsDataAvailable())
      {
         listDataUpdateMsg.GetUpdater()->update(m_dataProvider, changedItems);
      }
      else
      {
         ETG_TRACE_ERR(("No data provider available for ListId %d to apply the update", listDataUpdateMsg.GetUpdater()->listId()));
      }
   }
}


bool ListDataWrapper::IsDataAvailable() const
{
   return (m_dataProvider != s_emptyProvider) && (m_dataProvider->virtualListSize() > 0);
}


Candera::Int32 ListDataWrapper::GetStartIndex() const
{
   return m_startIndex;
}


Candera::UInt32 ListDataWrapper::GetWindowElementSize() const
{
   return m_dataProvider->windowElementSize();
}


Candera::UInt32 ListDataWrapper::GetListId() const
{
   return m_dataProvider->listId();
}


Candera::UInt32 ListDataWrapper::GetUnmodifiedListSize() const
{
   return m_dataProvider->virtualListSize();
}


Candera::UInt32 ListDataWrapper::GetVirtualListSize() const
{
   return m_virtualListSize;
}


FeatStd::SizeType ListDataWrapper::GetSequenceNumber() const
{
   return m_dataProvider->getSequenceNumber();
}


tSharedPtrListDataItem& ListDataWrapper::operator[](size_t index)
{
   return IsIndexValid(index) ? ((*m_dataProvider)[index]) : s_emptyItem;
}


const tSharedPtrListDataItem& ListDataWrapper::operator[](size_t index) const
{
   return IsIndexValid(index) ? ((*m_dataProvider)[index]) : s_emptyItem;
}


bool ListDataWrapper::IsIndexValid(size_t index) const
{
   return /*index is unsigned and thus always greater than 0*/ (index < m_dataProvider->windowElementSize());
}


void ListDataWrapper::UpdateSetListChangeSetIndex(Candera::Int32 firstVisibleIndex)
{
   if (IsDataAvailable())
   {
      m_dataProvider->setListChangeSetIndex(firstVisibleIndex);
   }
}
