/* ***************************************************************************************
* FILE:          ListDataProviderBuilder.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ListDataProviderBuilder is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#if !defined(ListDataProviderBuilder_H)
#define ListDataProviderBuilder_H

#include <stdint.h>
#include "ServersData.h"
#include "ListDataItem.h"
#include "DataBindingUpdater.h"
#include "ListDataProvider.h"
#include "Widgets/2D/List/generated/ListMessages.h"
#include "Courier/Platform/Memory.h"

// Provides support to initialize the list data required by the FlexListWidget2D.
// See the documentation provided for this widget to understand how ListDataProvider and DataItem work.
// DataItemContext is used by the widget to identify the template node in the scene for a particular list item.
// A default DataItemContext can be specified when the builder is created and for each item it is possible to overwrite it.
// The data for each row is provided sing a vector which contain elements of type int, string or vector.
// Nodes inside each template use a binding mechanism between the elements of the vector and various subnodes inside the template.
/// <code>
/// </code>


class ListDataProviderBuilder
{
   public:
      class ListItem
      {
            friend class ListDataProviderBuilder;

         public:
            static const ServersDataItemFlags Unused              = 0;        // unused
            static const ServersDataItemFlags Expandable          = 0x01;     // expanded, used  with dropdown
            static const ServersDataItemFlags IsExpanded          = 0x02;     // expanded or collapsed
            static const ServersDataItemFlags FullComparison      = 0x04;     // items are compared based on their complete state; otherwise, only their
            // type, handles, flags are taken into consideration

            // Adds a new id entry in the data vector for this item.
            // Id entries are special strings which can be set as button widget names
            // and when the ButtonReactionMsg is received
            // the sender info in the message can be obtained to retrieve
            // the hdlRow, hdlCol and flags specified as parameters of this method as well as the list id of the parent list.
            ListItem& AddId(ServersDataItemHdlRow hdlRow,
                            ServersDataItemHdlCol hdlCol = 0ul,
                            ServersDataItemFlags flags = 0ul);
            // Adds a new int entry in the data vector for this item.
            ListItem& AddData(Candera::UInt32 intValue);
            // Adds a new string entry in the data vector for this item.
            ListItem& AddData(const Candera::String& strValue);
            // Adds a new vector entry in the data vector for this item.
            ListItem& AddData(const std::vector<tSharedPtrListDataItem>& vecValue);

            // Adds the specified binding updater for this item.
            // The updater must be dynamically allocated in the user code and will be deallocated by the widget.
            _FEATSTD_DEPRECATED("This method will be made private. Use AddDataBindingUpdater() instead.",
                                ListItem& AddDataBindingUpdaterPtr(Courier::IDataBindingUpdater* dataBindingUpdater)
                               );

            // Adds a new data binding updater for this item which is initialized with the specified data.
            template<typename T, typename TData>
            ListItem& AddDataBindingUpdater(const TData& data)
            {
               Courier::DataBindingUpdater<T>* ptr = COURIER_NEW(Courier::DataBindingUpdater<T>)(data);
               if (ptr != NULL)
               {
                  (void)AddDataBindingUpdaterPtrImpl(ptr);
               }
               return *this;
            }

            // Adds a new uninitialized data binding updater for this item.
            // The updater is returned so that it can be initialized.
            template<typename T>
            Courier::DataBindingUpdater<T>* AddDataBindingUpdater()
            {
               Courier::DataBindingUpdater<T>* ptr = COURIER_NEW(Courier::DataBindingUpdater<T>)();
               if (ptr != NULL)
               {
                  (void)AddDataBindingUpdaterPtrImpl(ptr);
               }
               return ptr;
            }

            // Called by CreateDataProvider to create the DataItem which should be sent to the widget.
            tSharedPtrListDataItem CreateDataItem() const;

            // Modifies indexToUpdate which is used only for list data provider updater
            ListItem& SetIndexToUpdate(Candera::Int32 index)
            {
               _indexToUpdate = index;
               return *this;
            }

            // Returns indexToUpdate which is used only for list data provider updater
            Candera::Int32 GetIndexToUpdate() const
            {
               return _indexToUpdate;
            }

         private:
            ListItem(const ListItem& other);
            ListItem& operator=(const ListItem& other);
            ListItem(const ListDataProviderBuilder& builder,
                     ServersDataItemHdlRow hdlRow,
                     ServersDataItemHdlCol hdlCol = 0ul,
                     DataItemContext dataItemContext = "",
                     ServersDataItemFlags flags = 0ul,
                     Candera::Int32 indexToUpdate = 0);

            ListItem& AddDataBindingUpdaterPtrImpl(Courier::IDataBindingUpdater* dataBindingUpdater);

            ServersDataItemFlags GetFirstItemFlags() const;

            const ListDataProviderBuilder& _builder;
            unsigned int _hdlRow;
            DataItemContext _dataItemContext;
            tSharedPtrListDataItemVector _data;

            //used only for list data provider updater to identify the item to replace in the list data provider.
            //it has a sign in order to support circular lists which require currently negative indexes.
            Candera::Int32 _indexToUpdate;
      };

      // Creates a new builder for a specific listId using the specified default DataItemContext.
      ListDataProviderBuilder(unsigned int listId, DataItemContext defaultDataItemContext = "");
      ~ListDataProviderBuilder();

      unsigned int GetListId() const
      {
         return _listId;
      }
      const char*  GetDefaultDataItemContext() const
      {
         return _defaultDataItemContext;
      }

      // Adds a new item and initializes the entry 0 in the data vector for it.
      // - dataItemContext allows to overwrite the default one in the builder.
      // - indexToUpdate is used only for list data provider updater to identify the item to replace in the list data provider.
      ListItem& AddItem(ServersDataItemHdlRow hdlRow,
                        ServersDataItemHdlCol hdlCol = 0ul,
                        DataItemContext dataItemContext = "",
                        ServersDataItemFlags flags = 0ul,
                        Candera::Int32 indexToUpdate = 0);

      // Adds a new expandable item and initializes the entry 0 in the data vector for it.
      // - expandableListId identifies the expandable list and will be set for this DataItem in HdlCol.
      // - extraFlags provide additional flags in addition to DataItem::Expandable and will be set for the DataItem.
      // - indexToUpdate is used only for list data provider updater to identify the item to replace in the list data provider.
      ListItem& AddExpandableItem(ServersDataItemHdlRow hdlRow,
                                  unsigned int expandableListId,
                                  DataItemContext dataItemContext = "",
                                  ServersDataItemFlags extraFlags = 0ul,
                                  Candera::Int32 indexToUpdate = 0);

      // Creates a list data provider which contains items of type DataItem necessary to render data in a FlexListWidget2D.
      tSharedPtrDataProvider CreateDataProvider(int startIndex = 0, unsigned int virtualListSize = 0);
      // Creates a vector of DataItems which can be used as sub items using ListItem::AddData(vector).
      tSharedPtrListDataItemVector CreateDataItemVector();
      // Creates a list data provider updater which can be used to update a list data provider.
      tSharedPtrDataProviderUpdater CreateDataProviderUpdater();

   private:
      unsigned int _listId;
      DataItemContext _defaultDataItemContext;
      std::vector<ListItem*> _items;
};


//
// Helper class to enumerate the items, takes care about circular/wrap-around list handling
//
class ListDateProviderReqItemEnumerator
{
   public:
      ListDateProviderReqItemEnumerator(const ListDateProviderReqMsg& oMsg, uint32_t virtualListSize);
      ListDateProviderReqItemEnumerator(int32_t startIndex, uint32_t windowElementSize, uint32_t virtualListSize, bool circular);

      bool getItemIndex(uint32_t& index);
      int32_t getStartIndex() const
      {
         return _startIndex;
      }
      uint32_t getVirtualListSize() const
      {
         return _maxitems;
      }
   private:
      void init(int32_t startIndex, uint32_t windowElementSize, uint32_t virtualListSize, bool circular);
      int32_t _itemIndex;
      int32_t _startIndex;
      int32_t _lastIndex;
      uint32_t _maxitems;
};


#endif
