/* ***************************************************************************************
* FILE:          ListDataProvider.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ListDataProvider is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef _ListDataProvider_H
#define _ListDataProvider_H

FEATSTD_LINT_FILE(967, "ListDataProvider.h", "false positive - file has valid include guards.")

#include "Widgets/2D/List/Data/ListDataItem.h"
#include <Courier/Messaging/Message.h>
#include <string>
#include <map>
#include <set>

class ListDataProvider;
typedef Candera::MemoryManagement::SharedPointer<ListDataProvider> tSharedPtrDataProvider;


// interface to list widget and b-logic
class ListDataProvider
{
   public:

      static tSharedPtrDataProvider newBackEndInterface(Candera::UInt32 listId, Candera::Int32 startIndex, Candera::UInt32 windowElementSize, Candera::UInt32 virtualListSize);

      virtual ~ListDataProvider();

      /** Every newly allocated ListDataProvider gets a sequence number generated which can be used to identify different versions of the data.
      * The sequence number of the data used to prepare the visible items in the list is provided in various update messages posted by the list.
      */
      FeatStd::SizeType getSequenceNumber() const;

      Candera::UInt32 listId() const;

      Candera::Int32 startIndex() const;

      Candera::UInt32 windowElementSize() const;

      Candera::UInt32 listSize() const;

      Candera::UInt32 virtualListSize() const;

      Candera::Int32 getFocusIndex() const;

      bool isFocusIndexEnabled() const;

      void setFocusIndex(Candera::Int32 focusIndex);

      void setListChangeSetIndex(FeatStd::Int32 listChangeSetIndex);

      FeatStd::Int32 getListChangeSetIndex() const;

      void setCacheOnOff(bool enable);

      bool isCacheEnabled() const;

      tSharedPtrListDataItem& operator[](size_t index);

      const tSharedPtrListDataItem& operator[](size_t index) const;

      std::string dumpContent() const;
      static std::string dumpPayload(const Courier::Message& msg);

      tSharedPtrDataProvider clone() const;

      ServersDataItemHdlRow searchIdentifier(Courier::Identifier& ident) const;
      tSharedPtrListDataItem searchIdentifier(unsigned hdlRow, unsigned hdlCol) const;

      bool validDataItem(unsigned ii) const;

      bool elementsInList(Candera::Int32 searchIndex, Candera::UInt32 windowElementSize, bool bWithValidData) const;

      void Retain();

      void Release();

   private:

      ListDataProvider(Candera::UInt32 listId, Candera::Int32 startIndex, Candera::UInt32 windowElementSize, Candera::UInt32 virtualListSize);

      std::vector <tSharedPtrListDataItem> _lst;             // list elements
      Candera::UInt32                  _listId;          // list identifier
      Candera::Int32                   _startIndex;      // index number of first index in list
      Candera::UInt32                  _virtualListSize; // max numbers for scrollbar min/max and last entry for request. Corresponds to the size if all available data items were added to the list.
      Candera::Int32                   _focusIndex;      // 0..n, -1=no focus modification
      Candera::Int32                   _listChangeSetIndex;
      bool                             _cached;
      Courier::Platform::AtomicOp::Atomic _refCnt; // reference counter for shared pointer
      FeatStd::SizeType                _sequenceNumber;
};


class ListDataProviderUpdater;
typedef Candera::MemoryManagement::SharedPointer<ListDataProviderUpdater> tSharedPtrDataProviderUpdater;

/* Provides support to replace items contained into a list data provider. It does not insert or remove items.
* Create a ListDataProviderUpdater using ListDataProviderBuilder and post it with a ListDataProviderUpdMsg.
* The list widget will consume the message, update its data provider and refresh the screen.
*/
class ListDataProviderUpdater
{
   public:
      FEATSTD_TYPEDEF_SHARED_POINTER(ListDataProviderUpdater);

      virtual ~ListDataProviderUpdater() {}

      /* Creates a new data provider updater associated with the specified list id. */
      static tSharedPtrDataProviderUpdater create(Candera::UInt32 listId);

      /* Identifies the list to be updated. */
      Candera::UInt32 listId() const;

      /* Adds a new item which is associated with a specific index.
      * Index has a sign in order to support circular lists which require currently both negative and positive indexes.
      */
      void addItem(Candera::Int32 index, tSharedPtrListDataItem item);

      typedef std::set<Candera::Int32> ChangedItemsType;
      /* Applies the updates to the specified data provider. */
      void update(tSharedPtrDataProvider dataProvider, ChangedItemsType& changedItems);

   private:
      FEATSTD_SHARED_POINTER_DECLARATION();

      ListDataProviderUpdater(Candera::UInt32 listId);

      Candera::UInt32 _listId;

      typedef std::map<Candera::Int32, tSharedPtrListDataItem> ItemsType;
      ItemsType _items;
};


#endif
