/* ***************************************************************************************
* FILE:          ListDataProvider.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ListDataProvider is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "ListDataProvider.h"
#include "Widgets/2D/List/generated/ListMessages.h"


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ListDataProvider.cpp.trc.h"
#endif


std::string ListDataProvider::dumpPayload(const Courier::Message& msg)
{
#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
   FeatStd::Internal::FixedSizeString<1000> buf;
   FeatStd::FixedSizeStringBuffer<1000> strBuf(buf);
   strBuf.AppendObject(msg);
   return buf.CStr();
#else
   return "Courier Message Payload appender for ListDataProvider not enabled!";
#endif
}


tSharedPtrDataProvider ListDataProvider::clone() const
{
   tSharedPtrDataProvider dataClone = newBackEndInterface(_listId, _startIndex, _lst.size(), _virtualListSize);

   for (unsigned ii = 0; ii < _lst.size(); ++ii)
   {
      (*dataClone)[ii] = (*this)[ii];
   }
   return dataClone;
}


ServersDataItemHdlRow ListDataProvider::searchIdentifier(Courier::Identifier& ident) const
{
   for (unsigned ii = 0; ii < _lst.size(); ++ii)
   {
      tSharedPtrListDataItem it = (*this)[ii];
      if (it->getIdent() == ident)
      {
         return it->getHdlRow();
      }
   }
   return 0;
}


tSharedPtrListDataItem ListDataProvider::searchIdentifier(unsigned hdlRow, unsigned hdlCol) const
{
   char buf[40];

   SNPRINTF(buf, sizeof(buf), "ListItem_%u_%u_%u", listId(), hdlRow, hdlCol);
   Courier::Identifier ident = Courier::Identifier(buf);

   for (Candera::UInt32 i = 0; i < (*this).listSize(); i++)
   {
      tSharedPtrListDataItem it = (*this)[i];
      if (it->isVector() && (*it).getHdlRow() == hdlRow)
      {
         for (size_t n = 0; n < it->size(); n++)
         {
            ListDataItem* dataItem = Candera::Dynamic_Cast<ListDataItem*>((*it)[n].GetPointerToSharedInstance());
            if (0 != dataItem)
            {
               if ((0 == n) && (dataItem->getIdent() != ident))
               {
                  return tSharedPtrListDataItem();
               }
               if (dataItem->type() == ListDataItem::DataBindingUpdater)
               {
                  return tSharedPtrListDataItem(dataItem);
               }
            }
         }
      }
   }
   return tSharedPtrListDataItem();
}


bool ListDataProvider::validDataItem(unsigned ii) const
{
   return (!((*this)[ii].PointsToNull()));
}


bool ListDataProvider::elementsInList(Candera::Int32 searchIndex, Candera::UInt32 windowElementSize, bool bWithValidData) const
{
   if (listSize() > 0)
   {
      if (searchIndex >= _startIndex && (searchIndex + windowElementSize) <= (_startIndex + listSize()))
      {
         if (!bWithValidData)
         {
            for (unsigned ii = 0; ii < _lst.size(); ++ii)
            {
               if (!validDataItem(ii))
               {
                  return false;
               }
            }
         }
         return true;
      }
   }
   return false;
}


void ListDataProvider::Retain()
{
   Courier::Platform::AtomicOp::Inc(_refCnt);
}


void ListDataProvider::Release()
{
   if (Courier::Platform::AtomicOp::Dec(_refCnt) == 0)
   {
      FEATSTD_DELETE(this);
   }
}


class ListDataSequenceNumberGenerator
{
   public:
      static size_t generate()
      {
         static ListDataSequenceNumberGenerator _generator;

         size_t seqNr = static_cast<size_t>(Courier::Platform::AtomicOp::Inc(_generator._nextSeqNr));
         return seqNr;
      }

   private:
      ListDataSequenceNumberGenerator()
      {
         *_nextSeqNr = 0;
      }
      Courier::Platform::AtomicOp::Atomic _nextSeqNr;
};


ListDataProvider::ListDataProvider(Candera::UInt32 listId, Candera::Int32 startIndex, Candera::UInt32 windowElementSize, Candera::UInt32 virtualListSize)
{
   _listId = listId;
   _startIndex = startIndex;
   _lst.resize(windowElementSize);
   _virtualListSize = virtualListSize;
   _focusIndex = -1;
   _listChangeSetIndex = -1;
   _cached = true; // all lists by default are cached, can be  disabled by setCacheOnOff
   *_refCnt = 0;
   _sequenceNumber = ListDataSequenceNumberGenerator::generate();
}


size_t ListDataProvider::getSequenceNumber() const
{
   return _sequenceNumber;
}


const tSharedPtrListDataItem& ListDataProvider::operator[](size_t index) const
{
   return _lst[index];
}


tSharedPtrListDataItem& ListDataProvider::operator[](size_t index)
{
   return _lst[index];
}


tSharedPtrDataProvider ListDataProvider::newBackEndInterface(Candera::UInt32 listId, Candera::Int32 startIndex, Candera::UInt32 windowElementSize, Candera::UInt32 virtualListSize)
{
   return tSharedPtrDataProvider(FEATSTD_NEW(ListDataProvider)(listId, startIndex, windowElementSize, virtualListSize));
}


ListDataProvider::~ListDataProvider()
{
   _lst.clear();
}


Candera::UInt32 ListDataProvider::listId() const
{
   return _listId;
}


Candera::Int32 ListDataProvider::startIndex() const
{
   return _startIndex;
}


Candera::UInt32 ListDataProvider::windowElementSize() const
{
   return _lst.size();
}


Candera::UInt32 ListDataProvider::listSize() const
{
   return _lst.size();
}


Candera::UInt32 ListDataProvider::virtualListSize() const
{
   return _virtualListSize;
}


Candera::Int32 ListDataProvider::getFocusIndex() const
{
   return _focusIndex;
}


bool ListDataProvider::isFocusIndexEnabled() const
{
   return (_focusIndex >= 0);
}


void ListDataProvider::setFocusIndex(Candera::Int32 focusIndex)
{
   _focusIndex = focusIndex;
}


void ListDataProvider::setListChangeSetIndex(FeatStd::Int32 listChangeSetIndex)
{
   _listChangeSetIndex = listChangeSetIndex;
}


FeatStd::Int32 ListDataProvider::getListChangeSetIndex() const
{
   return _listChangeSetIndex;
}


void ListDataProvider::setCacheOnOff(bool cacheEnable)
{
   _cached = cacheEnable;
}


bool ListDataProvider::isCacheEnabled() const
{
   return _cached;
}


std::string ListDataProvider::dumpContent() const
{
   ETG_TRACE_USR1_CLS_DCL((TR_CLASS_HMI_LISTPROVIDER, APP_TRACECLASS_ID(), "=== DUMP content of ListDataProvider with ListID=%d ===========================", listId()));
   ETG_TRACE_USR1_CLS_DCL((TR_CLASS_HMI_LISTPROVIDER, APP_TRACECLASS_ID(), "===  startIndex=%d listSize=%d windowSize=%d virtualListSize=%d focusIndex=%d",
                           startIndex(), listSize(), windowElementSize(), virtualListSize(), getFocusIndex()));

   for (unsigned ii = 0; ii < _lst.size(); ++ii)
   {
      tSharedPtrListDataItem it = (*this)[ii];
      if (!it.PointsToNull())
      {
         it->dumpContent(ii);
      }
   }
   return "";
}


ListDataProviderUpdater::ListDataProviderUpdater(Candera::UInt32 listId) : _listId(listId)
{
}


tSharedPtrDataProviderUpdater ListDataProviderUpdater::create(Candera::UInt32 listId)
{
   return tSharedPtrDataProviderUpdater(CANDERA_NEW(ListDataProviderUpdater)(listId));
}


Candera::UInt32 ListDataProviderUpdater::listId() const
{
   return _listId;
}


void ListDataProviderUpdater::addItem(Candera::Int32 index, tSharedPtrListDataItem item)
{
   _items[index] = item;
}


void ListDataProviderUpdater::update(tSharedPtrDataProvider dataProvider, ChangedItemsType& changedItems)
{
   if (!dataProvider.PointsToNull() && (dataProvider->listId() == listId()))
   {
      ETG_TRACE_USR1_CLS_DCL((TR_CLASS_HMI_LISTPROVIDER, APP_TRACECLASS_ID(), "update list data provider with ListId=%u, StartIndex=%d, ListSize=%u",
                              listId(), dataProvider->startIndex(), dataProvider->listSize()));

      for (ItemsType::iterator it = _items.begin(); it != _items.end(); ++it)
      {
         Candera::Int32 index = (*it).first;
         size_t relativeIndex = static_cast<size_t>(index - dataProvider->startIndex());
         if ((index >= dataProvider->startIndex()) && (relativeIndex < dataProvider->listSize()))
         {
            ETG_TRACE_USR1_CLS_DCL((TR_CLASS_HMI_LISTPROVIDER, APP_TRACECLASS_ID(), "replacing item with index=%d (relativeIndex=%u) in data provider with ListId=%u",
                                    index, relativeIndex, listId()));

            (*dataProvider)[relativeIndex] = (*it).second;
            changedItems.insert(index);
         }
         else
         {
            ETG_TRACE_USR1_CLS_DCL((TR_CLASS_HMI_LISTPROVIDER, APP_TRACECLASS_ID(), "new item with index=%d is outside range of data provider with ListId=%u, StartIndex=%d, ListSize=%u",
                                    index, listId(), dataProvider->startIndex(), dataProvider->listSize()));
         }
      }
   }
}
