/* ***************************************************************************************
* FILE:          SwipingListContentUpdater.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SwipingListContentUpdater is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include <Widgets/2D/List/Content/AddChecker.h>
#include <Widgets/2D/List/Content/PageInfoCollector.h>
#include "Widgets/2D/ControlTemplate/ControlTemplate.h"
#include "Widgets/2D/List/Content/ItemSizesContainer.h"
#include "Widgets/2D/List/Content/Adders/OrthoGridAddingStrategy.h"
#include "Widgets/2D/List/Content/Adders/GridAddingStrategy.h"
#include "Widgets/2D/List/Content/Adders/StackAddingStrategy.h"
#include <Candera/Engine2D/Layout/GridLayouter.h>
#include <Widgets/2D/ControlTemplate/ItemLayouter.h>
#include <Widgets/2D/List/Content/ListContentAdder.h>

#include "SwipingListContentUpdater.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/SwipingListContentUpdater.cpp.trc.h"
#endif

using namespace Candera;

SwipingListContentUpdater::SwipingListContentUpdater(const OwnerInfoRetriever& ownerInfoRetriever,
      OrientationOperator& directOrientationOperator,
      OrientationOperator& orthogonalOrientationOperator,
      ControlTemplateInstanceContainer& itemsInstanceContainer,
      ISizeContainer& sizesContainer,
      ITemplateRetriever& templateRetriever,
      const ContentProvider& contentProvider,
      ItemSpeedAnimationManager& itemSpeedAnimationManager) :
   ListContentUpdater(itemsInstanceContainer, templateRetriever, contentProvider),
   _scrollableContentAdder(directOrientationOperator, orthogonalOrientationOperator, itemsInstanceContainer, templateRetriever, contentProvider, itemSpeedAnimationManager, ListContentAdder::ForwardAndReverse),
   _paddingContentAdderPrevious(directOrientationOperator, orthogonalOrientationOperator, itemsInstanceContainer, templateRetriever, contentProvider, itemSpeedAnimationManager, ListContentAdder::Reverse),
   _paddingContentAdderNext(directOrientationOperator, orthogonalOrientationOperator, itemsInstanceContainer, templateRetriever, contentProvider, itemSpeedAnimationManager, ListContentAdder::Forward),
   _ownerInfoRetriever(ownerInfoRetriever),
   _swipingStateChecker(0),
   _itemSizesContainer(0),
   _listItemAdder(0),
   _itemsGroupLayouter(0),
   _pixelWiseAnimationManager(),
   _pixelWiseAnimationTime(0),
   _shortPixelwiseAnimationTime(0),
   _animTargetPosition(0.0F),
   _indexOfFirstCompletelyVisibleElement(0),
   _firstItemIndex(0),
   _previousLogicalStartIndex(0),
   _previousEndIndex(0),
   _positionLimit(Math::MaxFloat()),
   _firstUpdateAfterNewData(true),
   _currentActualStartIndex(0),
   _orientationOperator(directOrientationOperator),
   _orthogonalOrientationOperator(orthogonalOrientationOperator),
   _numberOfCompleteVisibleItems(0),
   _firstVisibleItemSize(0.0F),
   _circular(false),
   _animationManagerListener(0),
   _fixedPages(false),
   _visibleArea(),
   _pixelwise(false),
   _shouldAnalyzeTemplates(false),
   _elementsShownOnlyOnce(true),
   _addingStrategy(AddingStrategy::SharedPointer(0)),
   _sizesContainer(sizesContainer),
   _coverflow(false),
   _dynamicGrid(false),
   _layoutingListener(0),
   _expandTriggerManager(0),
   _pagewiseLockOut(false),
   _privateTimeDispatcher(0),
   _listAlignment(Candera::ListAlignment::Begin),
   _oldVirtualListSize(0),
   _preparedItemsBufferSize(0),
   _preparedItemsUpdateTriggerOffset(0),
   _itemLayouter(sizesContainer),
   _normalizer(0),
   _lastItemIndex(0),
   _dynamicGridEmptyCellsPolicy(DynamicGridEmptyCellsPolicyType::DistributeToAllItems),
   _listener(0),
   _lastAdditionData(),
   _listInfo(),
   _itemSpeedAnimationManager(itemSpeedAnimationManager)
{
}


SwipingListContentUpdater::~SwipingListContentUpdater()
{
   if (_itemsGroupLayouter != 0)
   {
      // the instance pointed by _itemsGroupLayouter will be destroyed by the asset builder
      _itemsGroupLayouter = 0;
   }

   _swipingStateChecker = 0;
   _itemSizesContainer = 0;
   _animationManagerListener = 0;
   _layoutingListener = 0;
   _privateTimeDispatcher = Animation::AnimationTimeDispatcher::SharedPointer(0);
   _listener = 0;
}


void SwipingListContentUpdater::Init(SwipingStateChecker* swipingStateChecker,
                                     ItemSizesContainer* itemSizesContainer,
                                     LayoutingListener* layoutingListener,
                                     Candera::Node2D* itemsGroup,
                                     const ListWidget2DBase::NumberOfItemsType& configuredNumberOfItems,
                                     TimeDispatcher* timeDispatcher)
{
   DisposeContent();

   Base::Init(itemsGroup, configuredNumberOfItems);
   _layoutingListener = layoutingListener;

   UpdateLayouter();
   UpdateAddingStrategy();
   UpdateAdder();

   _pixelWiseAnimationManager.Init(timeDispatcher);
   _pixelWiseAnimationManager.SetListener(this);

   _swipingStateChecker = swipingStateChecker;
   _itemSizesContainer = itemSizesContainer;
}


void SwipingListContentUpdater::UpdateLayouter()
{
   Candera::Node2D* itemsGroup(GetItemsGroup());
   if (0 != itemsGroup)
   {
      Candera::Layouter* originalLayouter(itemsGroup->GetLayouter());

      if (0 != _itemsGroupLayouter)
      {
         originalLayouter = _itemsGroupLayouter->GetOriginalLayouter();

         FEATSTD_DELETE(_itemsGroupLayouter);
         _itemsGroupLayouter = 0;
      }

      _itemsGroupLayouter = FEATSTD_NEW(FlexListLayouter)(originalLayouter);
      if (0 != _itemsGroupLayouter)
      {
         itemsGroup->SetLayouter(_itemsGroupLayouter);
         _itemsGroupLayouter->RegisterListener(_layoutingListener);
      }
   }
}


void SwipingListContentUpdater::UpdateAddingStrategy()
{
   if (0 != _itemsGroupLayouter)
   {
      Candera::Layouter* originalLayouter(_itemsGroupLayouter->GetOriginalLayouter());
      GridLayouter* gridLayouter(dynamic_cast<GridLayouter*>(originalLayouter));
      if (0 != gridLayouter)
      {
         const UInt8 rows(gridLayouter->GetRowCount());
         const UInt8 columns(gridLayouter->GetColumnCount());

         if (_dynamicGrid)
         {
            const OrthoGridAddingStrategy::SharedPointer& strategy(OrthoGridAddingStrategy::Create(_sizesContainer, _orientationOperator, _orthogonalOrientationOperator, rows, columns));
            strategy->SetDynamicGridEmptyCellsPolicy(_dynamicGridEmptyCellsPolicy);
            _addingStrategy = strategy;
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SwipingListContentUpdater::UpdateAdderFactory dynamic grid with rows=%d, columns=%d", rows, columns));
         }
         else
         {
            _addingStrategy = GridAddingStrategy::Create(_sizesContainer, _orientationOperator, _orthogonalOrientationOperator, rows, columns);
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SwipingListContentUpdater::UpdateAdderFactory horizontal grid with rows=%d, columns=%d", rows, columns));
         }
      }
      else
      {
         _addingStrategy = StackAddingStrategy::Create(_sizesContainer, _orientationOperator, _orthogonalOrientationOperator, _coverflow);
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SwipingListContentUpdater::UpdateAdderFactory stack"));
      }
   }

   if (!_addingStrategy.PointsToNull())
   {
      _addingStrategy->SetNormalizer(_normalizer);
   }
}


void SwipingListContentUpdater::UpdateAdder()
{
   if (!_addingStrategy.PointsToNull())
   {
      _listItemAdder = _addingStrategy->CreateItemAdder(GetItemsGroup());
      _paddingListItemAdderPrevious = _addingStrategy->CreateItemAdder(GetItemsGroup());
      _paddingListItemAdderNext = _addingStrategy->CreateItemAdder(GetItemsGroup());

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SwipingListContentUpdater::UpdateAdder created adder"));
      _shouldAnalyzeTemplates = true;
   }
}


void SwipingListContentUpdater::Cache()
{
   if (!_addingStrategy.PointsToNull())
   {
      _addingStrategy->Cache(GetContentProvider(), _templateRetriever);
   }
}


void SwipingListContentUpdater::UpdateMaxPositions()
{
   MaxIndexManager::MaxIndexData maxPosData(
      _lastAdditionData.invalidItemsAdded,
      _coverflow,
      _lastAdditionData.pixelWiseEnabled,
      _lastAdditionData.endReached,
      _lastAdditionData.lastItemCompletelyVisible,
      _lastAdditionData.firstItemCompletelyVisible,
      _lastAdditionData.numberOfItemsInFirstStructure,
      _lastAdditionData.startIndex,
      _lastAdditionData.previousVisibleItems,
      _numberOfCompleteVisibleItems,
      _lastAdditionData.size,
      _lastAdditionData.firstItemVisibleSize,
      _lastAdditionData.lastItemInvisibleSize,
      _lastAdditionData.configuredNumberOfItems);
   _maxIndexManager.UpdateMaxIndex(maxPosData);

   CalculatePositionLimit(_lastAdditionData.position + _lastAdditionData.reversePositionCorrection, _lastAdditionData.scrollableSize, _lastAdditionData.endReached, _lastAdditionData.fillViewport);
}


void SwipingListContentUpdater::DisposeContent()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SwipingListContentUpdater::DisposeContent"));

   if (!_listItemAdder.PointsToNull())
   {
      _listItemAdder->Clear();
   }

   Base::DisposeContent();
}


void SwipingListContentUpdater::SetPixelWiseAnimationTime(const ListWidget2DBase::PixelWiseAnimationTimeType& time)
{
   _pixelWiseAnimationTime = time;
}


void SwipingListContentUpdater::SetShortPixelWiseAnimationTime(const ListWidget2DBase::PixelWiseAnimationTimeType& time)
{
   _shortPixelwiseAnimationTime = time;
}


void SwipingListContentUpdater::SetAnimationManagerListenr(FlexPixelWiseAnimationManagerListener* listener)
{
   _animationManagerListener = listener;
}


void SwipingListContentUpdater::SetCircular(bool val)
{
   _circular = val;
   _maxIndexManager.SetCircular(val);
}


void SwipingListContentUpdater::SetFixedPages(bool val)
{
   _fixedPages = val;
   _maxIndexManager.SetFixedPages(_fixedPages);
}


void SwipingListContentUpdater::SetPixelwise(bool val)
{
   _pixelwise = val;
}


void SwipingListContentUpdater::SetCoverflow(bool val)
{
   _coverflow = val;

   ResetMaximumPositions();
}


void SwipingListContentUpdater::ForcePosition(Candera::UInt32 newIndex, Candera::UInt32 /*previousIndex*/)
{
   //note: old index should probably be fetched from the animation manager to avoid errors? Test this.
   //Float currentPosition = _pixelWiseAnimationManager.GetPixelPosition();

   _pixelWiseAnimationManager.RemapPosition(Float(newIndex));
}


UInt32 SwipingListContentUpdater::GetMaxPosition() const
{
   return UInt32(_maxIndexManager.GetMaxIndex());
}


void SwipingListContentUpdater::SetTargetStartIndex(ListWidget2DBase::StartIndexType startIndex)
{
   if (!_addingStrategy.PointsToNull())
   {
      ContentProvider contentProvider(GetContentProvider());
      if ((_templateRetriever.IsValid()) && contentProvider.HasItems())
      {
         if (startIndex < 0)
         {
            if (IsCircularNeeded())
            {
               Float currentPosition(_pixelWiseAnimationManager.GetPosition());

               if (currentPosition == 0)
               {
                  currentPosition += GetMaximumPosition();

                  _pixelWiseAnimationManager.SetPosition(currentPosition);
               }

               startIndex += contentProvider.GetVirtualListSize();
            }
            else
            {
               startIndex = 0;
            }
         }

         _animTargetPosition = _addingStrategy->CalculateStartingPosition(_templateRetriever, contentProvider, startIndex);
         if ((!IsCircularNeeded()) && (_animTargetPosition > _positionLimit))
         {
            _animTargetPosition = _positionLimit;
         }
      }
   }
}


void SwipingListContentUpdater::RefreshOnDataChanged()
{
   ResetMaximumPositions();

   _firstUpdateAfterNewData = true;

   Int32 size(0);
   ContentProvider contentProvider(GetContentProvider());

   if (contentProvider.HasItems())
   {
      size = contentProvider.GetVirtualListSize();
   }

   if (!_addingStrategy.PointsToNull())
   {
      _addingStrategy->ProcessDataChanged(size);
   }

   AnalyzeTemplates();
   Cache();
}


void SwipingListContentUpdater::ResetMaximumPositions()
{
   Int32 size(0);
   ContentProvider contentProvider(GetContentProvider());

   if (contentProvider.HasItems())
   {
      size = contentProvider.GetVirtualListSize();
   }

   _maxIndexManager.ResetMaxPosition(size, true);

   if (_oldVirtualListSize < size)
   {
      _positionLimit = _maxIndexManager.GetMaxIndex();
   }
}


void SwipingListContentUpdater::RefreshOnListItemExpandReqMsg()
{
   ResetMaximumPositions();
}


FeatStd::Int32 SwipingListContentUpdater::GetNumberOfCompletelyVisibleItems() const
{
   return Int32(_numberOfCompleteVisibleItems);
}


FeatStd::UInt32 SwipingListContentUpdater::GetIndexOfFirstCompletelyVisibleElement() const
{
   return UInt32(_indexOfFirstCompletelyVisibleElement);
}


FeatStd::Int32 SwipingListContentUpdater::GetFirstItemIndex() const
{
   return _firstItemIndex;
}


FeatStd::Int32 SwipingListContentUpdater::GetLastItemIndex() const
{
   return _lastItemIndex;
}


const PagesInfo& SwipingListContentUpdater::GetPagesInfo() const
{
   static ListPagesInfo emptyPagesInfo;

   if (!_addingStrategy.PointsToNull())
   {
      return _addingStrategy->GetPagesInfo();
   }

   return emptyPagesInfo;
}


void SwipingListContentUpdater::SetTimeDispatcher(const Candera::Animation::AnimationTimeDispatcher::SharedPointer& val)
{
   _privateTimeDispatcher = val;
}


Int32 SwipingListContentUpdater::AddListContent(ListWidget2DBase::FocusedIndexType& focusedIndex, const Candera::Float position, const Candera::Vector2& visibleAreaInPixels, const ListWidget2DBase::PixelWiseScrollingEnabledType& pixelWiseEnabled)
{
   Int32 startIndex(0);

   if ((!_listItemAdder.PointsToNull()) && (!_addingStrategy.PointsToNull()) && GetContentProvider().HasItems() && (_templateRetriever.IsValid()) && (0 != _itemSizesContainer) && (GetContentProvider().GetVirtualListSize() > 0))
   {
      const UInt32 previousVisibleItems = _numberOfCompleteVisibleItems;

      _scrollableContentAdder.Set(_listItemAdder,
                                  _addingStrategy,
                                  &_maxIndexManager,
                                  _listener,
                                  _expandTriggerManager);

      _itemSpeedAnimationManager.ComputeCurrentSize();
      startIndex = _scrollableContentAdder.AddContent(position, _scrollableArea, _listInfo, _pixelwise);

      // Correct target position if number of elements changed
      if (_scrollableContentAdder.CorrectCurrentPosition())
      {
         const Float firstItemMargin(Float(_listItemAdder->GetFirstItemMargin()));
         const Courier::Vector<Float>& sizes(_listItemAdder->GetSizes());

         const Float firstItemMarginNormalized(_normalizer->Normalize(firstItemMargin, sizes));
         const Float targetPosition(static_cast<Float>(startIndex) - firstItemMarginNormalized);
         SetTargetPosition(targetPosition);
         SetCurrentPosition(targetPosition);

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListContentAdder::AddListContent reverse added targetPosition=%f", targetPosition));
      }

      if (Candera::Node2D* itemsGroup = GetItemsGroup())
      {
         _orientationOperator.SetAlignment(*itemsGroup, _listAlignment);
         _orientationOperator.SetMargin(*itemsGroup, _listItemAdder->GetBeginingMargin(), 0, true, false);
         _orientationOperator.SetMargin(*itemsGroup, _listItemAdder->GetEndingMargin(), 0, false, true);
      }

      // item adding has finished; retrieve info data from the adder
      _numberOfCompleteVisibleItems = _listItemAdder->GetNumberOfCompleteVisibleItems();

      _firstVisibleItemSize = _listItemAdder->GetBeginingSize();

      _itemSizesContainer->SetItemsSizes(_listItemAdder->GetSizes());

      const Float firstItemMargin(Float(_listItemAdder->GetFirstItemMargin()));
      const Float lastItemMargin(Float(_listItemAdder->GetLastItemMargin()));

      const bool lastItemCompletelyVisible(lastItemMargin >= 0.0F);
      const bool firstItemCompletelyVisible(firstItemMargin >= 0.0F);

      const Int32 numberOfItemsInFirstStructure(_listItemAdder->GetNumberOfItemsInFirstStructure());

      const Float firstItemVisibleSize(_firstVisibleItemSize + (firstItemMargin < 0.0F ? firstItemMargin : 0.0F));
      const Float lastItemInvisibleSize((lastItemMargin < 0.0F) ? (-lastItemMargin) : 0.0F);

      const Float sizeOfAddedElements(_listItemAdder->GetSizeOfAddedElements());

      const bool fillViewport(pixelWiseEnabled || (0 == GetConfiguredNumberOfItems()));
      const Float scrollableSize(_orientationOperator.GetVectorComponent(_scrollableArea));

      _lastAdditionData =
      {
         _scrollableContentAdder.GetInvalidItemsAdded(),
         IsCircularNeeded(),
         lastItemCompletelyVisible,
         firstItemCompletelyVisible,
         pixelWiseEnabled,
         _scrollableContentAdder.EndReached(),
         fillViewport,
         GetConfiguredNumberOfItems(),
         numberOfItemsInFirstStructure,
         GetContentProvider().GetVirtualListSize(),
         previousVisibleItems,
         static_cast<FeatStd::Float>(startIndex),
         position,
         scrollableSize,
         _scrollableContentAdder.GetReversedPositionCorrection(),
         firstItemVisibleSize,
         lastItemInvisibleSize,
         firstItemMargin,
         lastItemMargin,
         sizeOfAddedElements
      };

      _firstItemIndex = startIndex;
      _lastItemIndex = _firstItemIndex + _listItemAdder->GetNumberOfAddedItems() - 1;

      if (!firstItemCompletelyVisible)
      {
         startIndex += numberOfItemsInFirstStructure;
      }

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListContentAdder::AddListContent circularNeeded=%d, pixelWiseEnabled=%d, endReached=%d, firstItemMargin=%f, "
                          "lastItemMargin=%f, numberOfItemsInFirstStructure=%d, startIndex=%d, previousVisibleItems=%d, _numberOfCompleteVisibleItems=%d",
                          IsCircularNeeded(), pixelWiseEnabled, _scrollableContentAdder.EndReached(), firstItemMargin, lastItemMargin, numberOfItemsInFirstStructure, startIndex, previousVisibleItems, _numberOfCompleteVisibleItems));
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListContentAdder::AddListContent visibleSize=%f, sizeOfAddedElements=%f, firstItemVisibleSize=%f, lastItemInvisibleSize=%f, configuredNumberOfItems=%d",
                          scrollableSize, sizeOfAddedElements, firstItemVisibleSize, lastItemInvisibleSize, GetConfiguredNumberOfItems()));

      CheckContentUpdated(_firstItemIndex, _lastItemIndex);
   }

   _oldVirtualListSize = GetContentProvider().GetVirtualListSize();

   return startIndex;
}


bool SwipingListContentUpdater::IsCircularNeeded() const
{
   return _circular && (_elementsShownOnlyOnce || _coverflow);
}


void SwipingListContentUpdater::CalculatePositionLimit(const Float position, const Float scrollableSize, const bool endReached, const bool fillViewport)
{
   const Float maximumPosition(GetMaximumPosition());
   _positionLimit = maximumPosition;

   bool adjust(endReached && fillViewport);
   adjust = adjust && (!IsCircularNeeded());
   adjust = adjust && (!_listItemAdder.PointsToNull());
   adjust = adjust && (0 != _itemSizesContainer);
   adjust = adjust && (0 != _normalizer);
   adjust = adjust && (!_fixedPages);
   adjust = adjust && (!_coverflow);

   if (adjust)
   {
      const Float completeSizeOfAddedElements(_listItemAdder->GetSizeOfAddedElements());
      const Float delta(scrollableSize - completeSizeOfAddedElements);
      bool itemsLessThanViewport((delta > 0.0F) && (position < 1.0F));

      if (!itemsLessThanViewport)
      {
         const ItemSizesContainer::SizesVector& itemsSizes(_itemSizesContainer->GetItemsSizes());

         Float normalizedDelta = _normalizer->Normalize(delta, itemsSizes);
         _positionLimit = Math::Floor(position) - normalizedDelta;
      }
   }
}


void SwipingListContentUpdater::CheckContentUpdated(const FeatStd::Int32 firstItemIndex, const FeatStd::Int32 lastItemIndex)
{
   bool contentUpdated = ((_previousLogicalStartIndex != firstItemIndex) || (_previousEndIndex != lastItemIndex));

   if (contentUpdated)
   {
      NotifyContentUpdated(firstItemIndex, _previousLogicalStartIndex);
      _previousLogicalStartIndex = firstItemIndex;
      _previousEndIndex = lastItemIndex;
      ListVisualContentUpdateMsg* msg = COURIER_MESSAGE_NEW(ListVisualContentUpdateMsg)(_ownerInfoRetriever.GetOwnerId(), GetFirstItemIndex(), GetIndexOfFirstCompletelyVisibleElement(), GetNumberOfVisibleItems(), GetNumberOfCompletelyVisibleItems(), GetConfiguredNumberOfItems());
      if (msg != 0)
      {
         msg->Post();
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ListWidget2D: Post ListContentUpdateMsg listId=%u, firstVisibleIndex=%d, FirstCompleteVisibleIndex=%d, NumberOfVisibleItems=%d, NumberOfCompleteVisibleItems=%d, ConfiguredNumberOfItems=%d",
                             _ownerInfoRetriever.GetOwnerId(), GetFirstItemIndex(), GetIndexOfFirstCompletelyVisibleElement(), GetNumberOfVisibleItems(), GetNumberOfCompletelyVisibleItems(), GetConfiguredNumberOfItems()));
      }
   }
}


Courier::ViewScene2D* SwipingListContentUpdater::GetCurrentViewScene()
{
   Courier::ViewScene2D* viewScene(0);
   BaseWidget2D* owner(_ownerInfoRetriever.GetOwner());

   if (0 != owner)
   {
      Courier::View* parentView = owner->GetParentView();
      if (0 != parentView)
      {
         viewScene = parentView->ToViewScene2D();
      }
   }

   return viewScene;
}


bool SwipingListContentUpdater::UpdateListContent(ListWidget2DBase::StartIndexType& /*startIndex*/, ListWidget2DBase::FocusedIndexType& focusedIndex,
      ListWidget2DBase::NumberOfItemsType& numberOfItems,
      const Candera::Vector2& visibleAreaInPixels,
      ScrollingSpeed scrollingSpeed)
{
   DisposeContent();
   AnalyzeTemplates();

   _visibleArea = visibleAreaInPixels;
   _scrollableArea = Vector2((visibleAreaInPixels.GetX() - (_paddingArea.GetLeft() + _paddingArea.GetRight())),
                             (visibleAreaInPixels.GetY() - (_paddingArea.GetTop() + _paddingArea.GetBottom())));

   _listInfo =
   {
      GetItemsGroup(),
      IsGrid(),
      _coverflow,
      _fixedPages,
      _circular,
      GetConfiguredNumberOfItems(),
      _positionLimit,
      &_itemLayouter,
      _scrollAnimationInput
   };

   bool itemsInvalid = false;
   _focusedItem = 0;
   ContentProvider contentProvider(GetContentProvider());
   bool isInvalidationNeeded = false;

   if ((0 != GetItemsGroup()) && (0 != _swipingStateChecker) && (contentProvider.HasItems()))
   {
      if (_swipingStateChecker->IsIdle())
      {
         UpdateAnimationTime(scrollingSpeed);

         if (_animTargetPosition > _positionLimit)
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SwipingListContentUpdater::UpdateListContent limiting targetPosition %f to %f", _animTargetPosition, _positionLimit));
            _animTargetPosition = _positionLimit;
         }

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SwipingListContentUpdater::UpdateListContent animated position from %f", _pixelWiseAnimationManager.GetPosition()));
         isInvalidationNeeded = _pixelWiseAnimationManager.RunAnimationIfNeeded(_animTargetPosition);
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SwipingListContentUpdater::UpdateListContent animated position to %f; target=%f", _pixelWiseAnimationManager.GetPosition(), _animTargetPosition));
      }
      else if (!(_swipingStateChecker->IsTouched() || _swipingStateChecker->IsFocused()))
      {
         isInvalidationNeeded = true;
      }

      FeatStd::Float currentPosition(GetCurrentPosition());

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SwipingListContentUpdater::UpdateListContent adding elements position=%f", currentPosition));

      Int32 newStartIndex = AddListContent(focusedIndex, currentPosition, visibleAreaInPixels, _pixelwise);
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SwipingListContentUpdater::UpdateListContent added elements newStartIndex=%d", newStartIndex));

      itemsInvalid = itemsInvalid || (_indexOfFirstCompletelyVisibleElement != newStartIndex);
      if ((focusedIndex < 0) || (focusedIndex < _currentActualStartIndex))
      {
         focusedIndex = _currentActualStartIndex;
      }
      else if ((focusedIndex > contentProvider.GetVirtualListSize()) || (focusedIndex > _currentActualStartIndex + static_cast<FeatStd::Int32>(_numberOfCompleteVisibleItems)))
      {
         focusedIndex = _currentActualStartIndex + _numberOfCompleteVisibleItems;
      }

      _indexOfFirstCompletelyVisibleElement = newStartIndex;

      if (!_coverflow)
      {
         CollectPagesInfo();
      }

      if (_numberOfCompleteVisibleItems != numberOfItems)
      {
         numberOfItems = _numberOfCompleteVisibleItems;
         itemsInvalid = true;
      }

      itemsInvalid = itemsInvalid || isInvalidationNeeded;

      //synchronous notification after widgets have been added/removed for a potential client like focus manager
      if ((!itemsInvalid) && (ListContentUpdater::GetContentUpdateCallback() != 0) && (contentProvider.HasItems()))
      {
         ListContentUpdater::GetContentUpdateCallback()(contentProvider.GetListId(), true);
      }
   }

   _firstUpdateAfterNewData = false;

   if (_addingStrategy->SupportsViewportPadding())
   {
      AddPreviousPaddingItems();
      AddNextPaddingItems();

      LayoutingAdjuster* layoutingAdjuster = _addingStrategy->GetLayoutingAdjuster();
      if (layoutingAdjuster != 0 && (!_listItemAdder.PointsToNull()) && (!_paddingListItemAdderPrevious.PointsToNull()) && (!_paddingListItemAdderNext.PointsToNull()))
      {
         FeatStd::Int32 totalNoOfStructures = _listItemAdder->GetNumberOfStructures() + _paddingListItemAdderPrevious->GetNumberOfStructures() + _paddingListItemAdderNext->GetNumberOfStructures();
         layoutingAdjuster->Adjust(GetItemsGroup(), totalNoOfStructures, _orientationOperator);
      }
   }

   return itemsInvalid;
}


void SwipingListContentUpdater::CollectPagesInfo()
{
   if (!_addingStrategy.PointsToNull())
   {
      ContentProvider contentProvider(GetContentProvider());
      AddChecker addChecker(_pixelwise, GetConfiguredNumberOfItems());

      _addingStrategy->CollectPagesInfo(contentProvider, addChecker, _templateRetriever, _fixedPages, _firstUpdateAfterNewData, _indexOfFirstCompletelyVisibleElement, _numberOfCompleteVisibleItems, _circular, _scrollableArea);

      const FeatStd::Int32 virtualListSize(contentProvider.GetVirtualListSize());
      const FeatStd::Int32 windowElementSize(contentProvider.GetWindowElementSize());
      const bool allDataAvailable(virtualListSize <= windowElementSize);
      if (_firstUpdateAfterNewData && _fixedPages && allDataAvailable)
      {
         FeatStd::Int32 startIndexOfLastPage(_addingStrategy->GetPagesInfo().GetStartIndexOfLastPage());

         _maxIndexManager.SetIndexOfLastFixedPage(startIndexOfLastPage);
      }
   }
}


void SwipingListContentUpdater::AnalyzeTemplates()
{
   if (_shouldAnalyzeTemplates)
   {
      if (!_addingStrategy.PointsToNull())
      {
         _addingStrategy->AnalyzeTemplates(_templateRetriever);

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SwipingListContentUpdater::AnalyzeTemplates done"));
      }
      _shouldAnalyzeTemplates = false;
   }
}


void SwipingListContentUpdater::UpdateAnimationTime(ScrollingSpeed scrollingSpeed)
{
   switch (scrollingSpeed)
   {
      case Slow:
         _pixelWiseAnimationManager.SetAnimationTime(_pixelWiseAnimationTime);
         break;
      case Fast:
         _pixelWiseAnimationManager.SetAnimationTime(_shortPixelwiseAnimationTime);
         break;
      case Immediate:
         _pixelWiseAnimationManager.SetAnimationTime(0);
         break;
      default:
         _pixelWiseAnimationManager.SetAnimationTime(_pixelWiseAnimationTime);
         break;
   }
}


FeatStd::Float SwipingListContentUpdater::GetCurrentPosition() const
{
   return _pixelWiseAnimationManager.GetPosition();
}


FeatStd::Float SwipingListContentUpdater::GetMaximumPosition()
{
   Float maxPosition = 0.0F;

   ContentProvider contentProvider(GetContentProvider());
   if ((_templateRetriever.IsValid()) && (!_addingStrategy.PointsToNull()) && (contentProvider.HasItems()))
   {
      const Float maxIndex = Float(GetMaxPosition());
      maxPosition = _addingStrategy->AdjustMaxPosition(_templateRetriever, contentProvider, UInt32(maxIndex));
   }

   return maxPosition;
}


FeatStd::Float SwipingListContentUpdater::GetMaximumSwipingPosition() const
{
   return _positionLimit;
}


void SwipingListContentUpdater::SetCurrentPosition(FeatStd::Float val)
{
   _pixelWiseAnimationManager.SetPosition(LimitPosition(val));
}


void SwipingListContentUpdater::SetTargetPosition(FeatStd::Float val, bool finishCurrentAnimation)
{
   _animTargetPosition = LimitPosition(val);

   _pixelWiseAnimationManager.SetFinishCurrentAnimation(finishCurrentAnimation);
}


void SwipingListContentUpdater::SetMovementDelay(FeatStd::Int32 delay)
{
   if (delay < 0)
   {
      delay = 0;
   }
   _pixelWiseAnimationManager.SetDelay(static_cast<UInt32>(delay));
}


void SwipingListContentUpdater::SetDynamicGrid(bool flag)
{
   _dynamicGrid = flag;
   UpdateAddingStrategy();
   UpdateAdder();
}


bool SwipingListContentUpdater::IsGrid() const
{
   // dynamic_cast used as the Layouter do not support Candera::Dynamic_Cast
   return ((0 != _itemsGroupLayouter) && (dynamic_cast<GridLayouter*>(_itemsGroupLayouter->GetOriginalLayouter()) != 0));
}


FeatStd::Float SwipingListContentUpdater::LimitPosition(FeatStd::Float val)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SwipingListContentUpdater::LimitPosition from %f", val));

   const Float maximumPosition(_positionLimit);
   if (IsCircularNeeded())
   {
      if (0 != maximumPosition)
      {
         while (val < 0)
         {
            val += maximumPosition;
         }

         while (val > maximumPosition)
         {
            val -= maximumPosition;
         }
      }
   }
   else
   {
      if (val < 0)
      {
         val = 0;
      }

      if (val > maximumPosition)
      {
         val = maximumPosition;
      }
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SwipingListContentUpdater::LimitPosition to %f", val));

   return val;
}


void SwipingListContentUpdater::OnAfterTemplateGroupChanged()
{
   _shouldAnalyzeTemplates = true;
}


void SwipingListContentUpdater::OnAfterInvalidItemTemplateChanged()
{
   _shouldAnalyzeTemplates = true;
}


void SwipingListContentUpdater::OnAnimationFinished()
{
   if (0 != _animationManagerListener)
   {
      _animationManagerListener->OnAnimationFinished();
   }

   if (IsCircular())
   {
      const Float maximumPosition(GetMaximumPosition());
      _animTargetPosition = LimitPosition(_animTargetPosition);
      if (_animTargetPosition >= maximumPosition)
      {
         _animTargetPosition -= maximumPosition;
      }
      _pixelWiseAnimationManager.SetPosition(_animTargetPosition);
   }
}


bool SwipingListContentUpdater::IsCircular() const
{
   return IsCircularNeeded();
}


void SwipingListContentUpdater::Finalize()
{
   Base::Finalize();
}


void SwipingListContentUpdater::SetNormalizer(Normalizer* val)
{
   _normalizer = val;
   if (!_addingStrategy.PointsToNull())
   {
      _addingStrategy->SetNormalizer(_normalizer);
   }

   if (!_listItemAdder.PointsToNull())
   {
      _listItemAdder->SetNormalizer(_normalizer);
   }
}


void SwipingListContentUpdater::SetDynamicGridEmptyCellsPolicy(DynamicGridEmptyCellsPolicyType::Enum val)
{
   _dynamicGridEmptyCellsPolicy = val;
   UpdateAddingStrategy();
   UpdateAdder();
}


void SwipingListContentUpdater::OnCustomAnimationsStopped()
{
   ResetMaximumPositions();
}


bool SwipingListContentUpdater::IsLastCompletelyVisible() const
{
   bool retValue(false);
   if (!_listItemAdder.PointsToNull())
   {
      retValue = (_listItemAdder->GetLastItemMargin() >= 0);
   }
   return retValue;
}


void SwipingListContentUpdater::AddPreviousPaddingItems()
{
   if (!_paddingListItemAdderPrevious.PointsToNull())
   {
      Candera::Vector2 area(_orientationOperator.GetValue((Float)_paddingArea.GetLeft(), (Float)_visibleArea.GetX()),
                            _orientationOperator.GetValue((Float)_visibleArea.GetY(), (Float)_paddingArea.GetTop()));

      if (_orientationOperator.GetVectorComponent(area) > 0)
      {
         _paddingContentAdderPrevious.Set(_paddingListItemAdderPrevious, _addingStrategy, 0, 0, 0);
         _paddingContentAdderPrevious.AddContent(GetCurrentPosition(), area, _listInfo, true);

         FeatStd::Float addedItemSize = _paddingListItemAdderPrevious->GetSizeOfAddedElements();

         // top or left margin for previous elements not fitting in the padding area
         FeatStd::Float previousMargin = _orientationOperator.GetVectorComponent(area) - addedItemSize;
         Node2D* itemGroupNode = GetItemsGroup();

         if (itemGroupNode != 0)
         {
            Candera::Margin currentMargin = (_pixelwise) ? Layouter::GetMargin(*itemGroupNode) : Candera::Margin();

            Candera::Margin newMargin(currentMargin.GetLeft() + _orientationOperator.GetValue(previousMargin, 0.0f),
                                      currentMargin.GetTop() + _orientationOperator.GetValue(0.0f, previousMargin),
                                      currentMargin.GetRight(), currentMargin.GetBottom());

            Layouter::SetMargin(*itemGroupNode, newMargin);
         }
      }
   }
}


void SwipingListContentUpdater::AddNextPaddingItems()
{
   Candera::Vector2 area(_orientationOperator.GetValue((Float)_paddingArea.GetRight(), (Float)_visibleArea.GetX()),
                         _orientationOperator.GetValue((Float)_visibleArea.GetY(), (Float)_paddingArea.GetBottom()));

   if (_orientationOperator.GetVectorComponent(area) > 0 && !_scrollableContentAdder.EndReached())
   {
      _paddingListItemAdderNext->SetStructureOffset(_listItemAdder->GetNumberOfStructures() + _paddingListItemAdderPrevious->GetNumberOfStructures());
      _paddingContentAdderNext.Set(_paddingListItemAdderNext, _addingStrategy, 0, 0, 0);
      FeatStd::Float startPosition = GetCurrentPosition() + _listItemAdder->GetNumberOfStructures();
      _paddingContentAdderNext.AddContent(startPosition, area, _listInfo, true);
   }
}
