/* ***************************************************************************************
* FILE:          PageInfoCollector.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PageInfoCollector is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"

#include "PageInfoCollector.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/PageInfoCollector.cpp.trc.h"
#endif


using namespace FeatStd;
using namespace Candera;


PageInfoCollector::PageInfoCollector(ListPagesInfo& pagesInfo,
                                     const ITemplateRetriever& templateRetriever,
                                     const AddChecker& addChecker,
                                     ItemAdder::SharedPointer itemAdder,
                                     const ContentProvider& contentProvider,
                                     FeatStd::Int32 visualStartIndex,
                                     FeatStd::Int32 numberOfVisibleItems,
                                     bool circular) :
   _pagesInfo(pagesInfo),
   _templateRetriever(templateRetriever),
   _addChecker(addChecker),
   _itemAdder(itemAdder),
   _contentProvider(contentProvider),
   _visualStartIndex(visualStartIndex),
   _numberOfVisibleItems(numberOfVisibleItems),
   _circular(circular)
{
}


void PageInfoCollector::CollectPageInfo(const Candera::Vector2& visibleAreaInPixels)
{
   _pagesInfo.Clear();
   _pagesInfo.RecordPageStartIndex(0, _visualStartIndex);
   _pagesInfo.RecordCurrentPageSize(_numberOfVisibleItems);

   if (_contentProvider.HasItems())
   {
      if (!_itemAdder.PointsToNull())
      {
         const Int32 windowElementSize(_contentProvider.GetWindowElementSize());
         const bool allDataAvailable(_contentProvider.GetWindowElementSize() == _contentProvider.GetVirtualListSize());
         const Int32 dataStartIndex(_contentProvider.GetStartIndex());

         const Int32 startIndex((_circular && allDataAvailable) ? dataStartIndex - windowElementSize : dataStartIndex);
         const Int32 pastEndIndex(dataStartIndex + windowElementSize);

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PageInfoCollector::CollectPageInfo collecting from %d to %d", startIndex, pastEndIndex));

         CollectPageInfoForPrevious(*_itemAdder, startIndex, visibleAreaInPixels);
         CollectPageInfoForNext(*_itemAdder, pastEndIndex, visibleAreaInPixels);
      }
   }
}


void PageInfoCollector::CollectPageInfoForPrevious(ItemAdder& itemAdder, const Int32 startIndex, const Candera::Vector2& visibleAreaInPixels)
{
   itemAdder.Clear();

   bool vacantVisibleArea(true);
   bool shouldAdd(true);
   Int32 pageIndex(-1);
   Int32 searchStartIndex(_visualStartIndex - 1);
   Int32 i(searchStartIndex);
   itemAdder.SetReverseMode(true);

   itemAdder.BeginAdding(visibleAreaInPixels, 0.0F);

   while (i >= startIndex)
   {
      const tSharedPtrListDataItem& current = _contentProvider.GetItem(i);
      Node2D* templateNode(_templateRetriever.GetTemplate(current));

      --i;

      if (0 != templateNode)
      {
         vacantVisibleArea = itemAdder.AddItem(*templateNode, *templateNode);

         shouldAdd = _addChecker.ShouldAdd(itemAdder, vacantVisibleArea);
         if (!shouldAdd || (i == -1))
         {
            itemAdder.FinishReverseAdding();
            itemAdder.FinishAdding();
            const Int32 numberOfCompleteVisibleItems(static_cast<Int32>(itemAdder.GetNumberOfCompleteVisibleItems()));

            searchStartIndex -= numberOfCompleteVisibleItems;
            if (numberOfCompleteVisibleItems == 0)
            {
               searchStartIndex -= 1;
            }
            i = searchStartIndex;

            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PageInfoCollector::CollectPageInfoForPrevious recording page=%d, startIndex %d", pageIndex, searchStartIndex + 1));
            _pagesInfo.RecordPageStartIndex(pageIndex, searchStartIndex + 1);
            --pageIndex;

            itemAdder.BeginAdding(visibleAreaInPixels, 0.0F);
         }
      }
   }
}


void PageInfoCollector::CollectPageInfoForNext(ItemAdder& itemAdder, const Int32 pastEndIndex, const Candera::Vector2& visibleAreaInPixels)
{
   itemAdder.Clear();

   bool vacantVisibleArea(true);
   bool shouldAdd(true);
   Int32 pageIndex(0);
   Int32 searchStartIndex(_visualStartIndex);
   Int32 i(searchStartIndex);
   itemAdder.SetReverseMode(false);

   itemAdder.BeginAdding(visibleAreaInPixels, 0.0F);

   while (i < pastEndIndex)
   {
      const Int32 currentItemIndex(i);
      const tSharedPtrListDataItem& current = _contentProvider.GetItem(currentItemIndex);
      Node2D* templateNode(_templateRetriever.GetTemplate(current));

      ++i;

      if (0 != templateNode)
      {
         vacantVisibleArea = itemAdder.AddItem(*templateNode, *templateNode);

         shouldAdd = _addChecker.ShouldAdd(itemAdder, vacantVisibleArea);
         bool finishAdding((!shouldAdd) && (_circular || (i <= pastEndIndex)));
         finishAdding = finishAdding || (i == pastEndIndex);
         if (finishAdding)
         {
            itemAdder.FinishDirectAdding();
            itemAdder.FinishAdding();

            const Int32 numberOfCompleteVisibleItems(static_cast<Int32>(itemAdder.GetNumberOfCompleteVisibleItems()));

            searchStartIndex += numberOfCompleteVisibleItems;
            if (numberOfCompleteVisibleItems == 0)
            {
               searchStartIndex += 1;
            }

            if (_circular || (searchStartIndex < pastEndIndex))
            {
               i = searchStartIndex;
               if (0 == pageIndex)
               {
                  _pagesInfo.RecordCurrentPageSize(numberOfCompleteVisibleItems);
               }

               ++pageIndex;
               ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PageInfoCollector::CollectPageInfoForNext recording page=%d, startIndex %d", pageIndex, searchStartIndex));
               _pagesInfo.RecordPageStartIndex(pageIndex, searchStartIndex);
            }
            itemAdder.BeginAdding(visibleAreaInPixels, 0.0F);
         }
      }
   }
}
