/* ***************************************************************************************
* FILE:          ListPagesInfo.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ListPagesInfo is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include <FeatStd/Util/BinarySearch.h>

#include "ListPagesInfo.h"

using namespace FeatStd;
using namespace FeatStd::Internal;

ListPagesInfo::ListPagesInfo() :
   _currentPageSize(0)
{
}


ListPagesInfo::~ListPagesInfo()
{
}


void ListPagesInfo::Clear()
{
   _pageStartIndices.Clear();
   _currentPageSize = 0;
}


void ListPagesInfo::RecordCurrentPageSize(FeatStd::Int32 val)
{
   _currentPageSize = val;
}


void ListPagesInfo::RecordPageStartIndex(FeatStd::Int32 pageIndex, FeatStd::Int32 val)
{
   Int32 insertAt;

   Int size(_pageStartIndices.Size());
   Int32 index(BinarySearch<PageComparer, Info, Int32, PageStartIndices>(Info(pageIndex, val), _pageStartIndices, size, &insertAt));

   if (index == size)
   {
      _pageStartIndices.Insert(insertAt, Info(pageIndex, val));
   }
   else
   {
      _pageStartIndices[index]._pageIndex = pageIndex;
      _pageStartIndices[index]._startIndex = val;
   }
}


FeatStd::Int ListPagesInfo::PageComparer::Compare(Info left, Info right)
{
   return left._pageIndex - right._pageIndex;
}


FeatStd::Int ListPagesInfo::StartIndexComparer::Compare(Info left, Info right)
{
   return left._startIndex - right._startIndex;
}


bool ListPagesInfo::RetrieveStartIndexOfPage(const FeatStd::Int32 pageIndex, FeatStd::Int32& startIndex) const
{
   bool retVal = false;

   Int32 insertAt;
   Int size(_pageStartIndices.Size());
   Int32 index(BinarySearch<PageComparer, Info, Int32, PageStartIndices>(Info(pageIndex, 0), _pageStartIndices, size, &insertAt));

   if (index != size)
   {
      retVal = true;
      startIndex = _pageStartIndices[index]._startIndex;
   }
   else if ((0 <= insertAt) && (insertAt <= size) && (size > 0))
   {
      Int32 pos = (insertAt == size) ? (insertAt - 1) : insertAt;
      startIndex = _pageStartIndices[pos]._startIndex - (_pageStartIndices[pos]._pageIndex - pageIndex) * _currentPageSize;
   }

   return retVal;
}


FeatStd::Int32 ListPagesInfo::GetPage(FeatStd::Int32 startIndex) const
{
   FeatStd::Int32 page(0);
   Int32 insertAt;
   Int size(_pageStartIndices.Size());
   Int32 index(BinarySearch<StartIndexComparer, Info, Int32, PageStartIndices>(Info(0, startIndex), _pageStartIndices, size, &insertAt));

   if (index != size)
   {
      page = _pageStartIndices[index]._pageIndex;
   }
   else if ((0 < insertAt) && (insertAt <= size))
   {
      page = _pageStartIndices[insertAt - 1]._pageIndex;
   }
   else
   {
      page = 0;
   }

   return page;
}


void ListPagesInfo::Shift(FeatStd::Int32 offset)
{
   for (Int32 i(0); i < (Int32) _pageStartIndices.Size(); ++i)
   {
      ListPagesInfo::Info& pageInfo(_pageStartIndices[i]);
      pageInfo._pageIndex += offset;
   }
}


FeatStd::Int32 ListPagesInfo::GetStartIndexOfLastPage() const
{
   Int32 startIndex(0);

   if (_pageStartIndices.Size() > 0)
   {
      startIndex = _pageStartIndices[_pageStartIndices.Size() - 1]._startIndex;
   }
   return startIndex;
}
