/* ***************************************************************************************
* FILE:          GridTemplateSpanAnalyzer.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  GridTemplateSpanAnalyzer is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"

#include "GridTemplateSpanAnalyzer.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/GridTemplateSpanAnalyzer.cpp.trc.h"
#endif

using namespace Candera;

GridTemplateSpanAnalyzer::GridTemplateSpanAnalyzer(ISizeContainer& sizesContainer) :
   _minSize(Math::MaxFloat(), Math::MaxFloat()),
   _sizesContainer(sizesContainer),
   _maxSpan(1.0F, 1.0F)
{
}


GridTemplateSpanAnalyzer::SharedPointer GridTemplateSpanAnalyzer::Create(ISizeContainer& sizesContainer)
{
   return SharedPointer(FEATSTD_NEW(GridTemplateSpanAnalyzer)(sizesContainer));
}


GridTemplateSpanAnalyzer::~GridTemplateSpanAnalyzer()
{
}


void GridTemplateSpanAnalyzer::Analyze(const ITemplateRetriever& templateRetriever)
{
   ProcessingStarted();
   templateRetriever.Accept(this);
   ProcessingEnded();
}


Candera::Vector2 GridTemplateSpanAnalyzer::GetSpan(const Candera::Node2D* t) const
{
   Vector2 span(1.0F, 1.0F);
   SizeCache::const_iterator cache(_templatesCache.find(t));

   if (cache != _templatesCache.end())
   {
      span = cache->second;
   }

   return span;
}


Candera::Vector2 GridTemplateSpanAnalyzer::GetSpanUnitSize() const
{
   return _minSize;
}


Candera::Vector2 GridTemplateSpanAnalyzer::GetMaxSpan() const
{
   return _maxSpan;
}


void GridTemplateSpanAnalyzer::Visit(Candera::Node2D& t)
{
   ProcessTemplate(t);
}


void GridTemplateSpanAnalyzer::ProcessingStarted()
{
   _templatesCache.clear();
   _minSize = Vector2(Math::MaxFloat(), Math::MaxFloat());
   _maxSpan = Vector2(1.0F, 1.0f);
}


void GridTemplateSpanAnalyzer::ProcessTemplate(Candera::Node2D& t)
{
   Candera::Vector2 size = _sizesContainer.MeasureClientSize(&t);

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GridTemplateSpanAnalyzer::ProcessTemplate with size=(%f, %f), name='%s'", size.GetX(), size.GetY(), t.GetName()));

   FEATSTD_LINT_NEXT_EXPRESSION(1963, "Intended use of value_type to avoid explicit type conversion.");
   static_cast<void>(_templatesCache.insert(SizeCache::value_type(&t, size)));

   if ((!Math::FloatAlmostEqual(size.GetX(), 0.0F)) && (!Math::FloatAlmostEqual(size.GetY(), 0.0F)))
   {
      if (size.GetX() < _minSize.GetX())
      {
         _minSize.SetX(size.GetX());
      }

      if (size.GetY() < _minSize.GetY())
      {
         _minSize.SetY(size.GetY());
      }
   }
}


void GridTemplateSpanAnalyzer::ProcessingEnded()
{
   // replace the cached size with the calculated span

   if ((!Math::FloatAlmostEqual(_minSize.GetX(), 0.0F)) && (!Math::FloatAlmostEqual(_minSize.GetY(), 0.0F)))
   {
      const Vector2 inverseMinSize = Vector2(1.0F / _minSize.GetX(), 1.0F / _minSize.GetY());

      SizeCache::iterator it(_templatesCache.begin());
      for (; it != _templatesCache.end(); ++it)
      {
         Vector2& cachedVal = (*it).second;

         Float xSpan(Math::Ceil(cachedVal.GetX() * inverseMinSize.GetX()));
         Float ySpan(Math::Ceil(cachedVal.GetY() * inverseMinSize.GetY()));

         if (xSpan > _maxSpan.GetX())
         {
            _maxSpan.SetX(xSpan);
         }

         if (ySpan > _maxSpan.GetY())
         {
            _maxSpan.SetY(ySpan);
         }

         cachedVal.SetX(xSpan);
         cachedVal.SetY(ySpan);

         if (Math::FloatAlmostEqual(cachedVal.GetX(), 0.0F))
         {
            cachedVal.SetX(1.0F);
         }

         if (Math::FloatAlmostEqual(cachedVal.GetY(), 0.0F))
         {
            cachedVal.SetY(1.0F);
         }
      }
   }
   else
   {
      _templatesCache.clear();
   }
}
