/* ***************************************************************************************
* FILE:          StackItemAdder.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  StackItemAdder is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "StackItemAdder.h"

using namespace Candera;


CANDERA_RTTI_DEFINITION(StackItemAdder);


// StackItemAdder
StackItemAdder::StackItemAdder(ISizeContainer& sizesContainer, Candera::Node2D* groupNode, const OrientationOperator& directOrientationOperator, const OrientationOperator& orthogonalOrientationOperator, const bool& coverflow) :
   Base(sizesContainer, groupNode, directOrientationOperator, orthogonalOrientationOperator),
   _isFirstItemFound(false),
   _isCoverflow(coverflow)
{
}


StackItemAdder::SharedPointer StackItemAdder::Create(ISizeContainer& sizesContainer, Candera::Node2D* groupNode, const OrientationOperator& directOrientationOperator, const OrientationOperator& orthogonalOrientationOperator, const bool& coverflow)
{
   return SharedPointer(FEATSTD_NEW(StackItemAdder)(sizesContainer, groupNode, directOrientationOperator, orthogonalOrientationOperator, coverflow));
}


StackItemAdder::~StackItemAdder()
{
}


bool StackItemAdder::AddItem(Candera::Node2D& /*templateNode*/, Candera::Node2D& childNode)
{
   const Float itemSize(InsertItem(childNode));

   if (_isCoverflow && itemSize == 0 && GetCurrentSize() > 0)
   {
      return false;
   }

   PreAdd(childNode, itemSize);
   Add(childNode);
   PostAdd(childNode, itemSize);

   const Float visibleSize(_directOrientationOperator.GetVectorComponent(GetVisibleArea()));
   return (GetCurrentSize() < visibleSize);
}


void StackItemAdder::Clear()
{
   _isFirstItemFound = false;

   Base::Clear();
}


Candera::UInt32 StackItemAdder::GetScrollingIncrement() const
{
   return 1;
}


Candera::Int32 StackItemAdder::GetNumberOfItemsInFirstStructure() const
{
   return 1;
}


Candera::Int32 StackItemAdder::GetNumberOfStructures() const
{
   return GetNumberOfAddedItems();
}


void StackItemAdder::PreAdd(Candera::Node2D& /*childNode*/, const Float itemSize)
{
   CollectSize(itemSize);
   IncreaseSize(itemSize);

   if (!_isFirstItemFound)
   {
      _isFirstItemFound = true;
      _beginingSize = itemSize;
   }
}


void StackItemAdder::Add(Candera::Node2D& /*childNode*/)
{
}


void StackItemAdder::PostAdd(Candera::Node2D& childNode, const Float itemSize)
{
   if (_beginingMargin != 0)
   {
      _numberOfCompleteVisibleItems += 1;
   }

   UpdateBeginingMargin(GetSizes());

   if (_beginingMargin != 0)
   {
      _numberOfCompleteVisibleItems -= 1;
   }

   _endingSize = itemSize;
}


void StackItemAdder::OnFinishDirectAdding()
{
   const Float visibleSize(_directOrientationOperator.GetVectorComponent(GetVisibleArea()));
   const Float completeSize(GetSizeOfAddedElements());
   const Float currentSize(GetCurrentSize());

   if (currentSize > visibleSize)
   {
      _endingMargin = Int16(visibleSize - currentSize);
   }
   else if (completeSize >= visibleSize)
   {
      _beginingMargin = Int16(visibleSize - completeSize);
   }
}


void StackItemAdder::OnFinishReverseAdding()
{
   const Float visibleSize(_directOrientationOperator.GetVectorComponent(GetVisibleArea()));
   const Float completeSize(GetSizeOfAddedElements());

   _endingMargin = 0;
   if (completeSize >= visibleSize)
   {
      _beginingMargin = Int16(visibleSize - completeSize);
   }
}


void StackItemAdder::OnFinishAdding()
{
   if (_isCoverflow)
   {
      _beginingMargin = 0;
      _endingMargin = 0;
   }

   const Int16 margin = 0 - (_beginingMargin + _endingMargin);

   if ((margin > _endingSize) || ((_beginingMargin == 0) != (_endingMargin == 0)))
   {
      _numberOfCompleteVisibleItems -= 1;
   }
}
