/* ***************************************************************************************
* FILE:          PositionsCache.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PositionsCache is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include <FeatStd/Util/BinarySearch.h>
#include <FeatStd/Platform/Math.h>

#include "PositionsCache.h"

using namespace FeatStd;
using namespace FeatStd::Internal;

PositionsCache::CacheData::CacheData(const FeatStd::Int32& structureIndex, const FeatStd::Int32& firstIndex) :
   _structureIndex(structureIndex),
   _firstIndex(firstIndex)
{
}


PositionsCache::PositionsCache(FeatStd::UInt32 cacheSize /*= 10*/) :
   _cache(),
   _cacheStep(1)
{
   if (cacheSize <= 1)
   {
      cacheSize = 2;
   }

   _cache.Reserve(cacheSize);

   Clear();
}


struct StructureIndexComparer
{
   static inline Int Compare(PositionsCache::CacheData left, PositionsCache::CacheData right)
   {
      return (left.GetStructureIndex() == right.GetStructureIndex()) ? 0 : ((left.GetStructureIndex() < right.GetStructureIndex()) ? -1 : 1);
   }
};


struct FirstIndexComparer
{
   static inline Int Compare(PositionsCache::CacheData left, PositionsCache::CacheData right)
   {
      return (left.GetFirstIndex() == right.GetFirstIndex()) ? 0 : ((left.GetFirstIndex() < right.GetFirstIndex()) ? -1 : 1);
   }
};


template<typename Comparer>
PositionsCache::CacheData PositionsCache::Find(const CacheData& key) const
{
   CacheData data(0, 0);

   Int32 insertAtIndex;
   Int32 foundIndex = BinarySearch<Comparer, CacheData, Int32, Positions>(key, _cache, _cache.GetCapacity(), &insertAtIndex);

   if (foundIndex != (Int32) _cache.GetCapacity())
   {
      //found exact element; use it
      data = _cache[foundIndex];
   }
   else if (insertAtIndex != 0)
   {
      // element not found; use the previous position to where the element should be inserted into
      data = _cache[insertAtIndex - 1];
   }

   return data;
}


PositionsCache::CacheData PositionsCache::FindByStartIndex(const FeatStd::Int32& requestedStartIndex) const
{
   return Find<FirstIndexComparer>(CacheData(0, requestedStartIndex));
}


PositionsCache::CacheData PositionsCache::FindByStructureIndex(const FeatStd::Int32& requestedStructureIndex) const
{
   return Find<StructureIndexComparer>(CacheData(requestedStructureIndex, 0));
}


void PositionsCache::Cache(const CacheData& data)
{
   const Int32 index = data.GetFirstIndex() / _cacheStep;

   if (_cache[index].GetFirstIndex() < data.GetFirstIndex())
   {
      _cache[index] = data;

      EnsureSorting();
   }
}


void PositionsCache::EnsureSorting()
{
   Int32 i = 0;
   CacheData lastCachedData(0, 0);
   for (; i < (Int32) _cache.GetCapacity(); ++i)
   {
      if (_cache[i].GetFirstIndex() > lastCachedData.GetFirstIndex())
      {
         lastCachedData = _cache[i];
      }
      else
      {
         _cache[i] = lastCachedData;
      }
   }
}


void PositionsCache::Clear()
{
   for (Int32 i = 0; i < (Int32) _cache.GetCapacity(); ++i)
   {
      _cache[i] = CacheData(0, 0);
   }
}


void PositionsCache::UpdateCachStep(FeatStd::Int32 maxStartIndex)
{
   if (_cache.GetCapacity() != 1)
   {
      _cacheStep = static_cast<Int32>(Math::Ceil(static_cast<Float>(maxStartIndex) / (_cache.GetCapacity() - 1)));
      if (_cacheStep <= 0)
      {
         _cacheStep = 1;
      }
   }
   else
   {
      _cacheStep = 1;
   }
}
