/* ***************************************************************************************
* FILE:          OrthoItemArranger.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  OrthoItemArranger is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/OrthoItemArranger.cpp.trc.h"
#endif

#include "OrthoItemArranger.h"

using namespace Candera;

OrthoItemArranger::OrthoItemArranger(const OrientationOperator& directOrientationOperator,
                                     const OrientationOperator& orthogonalOrientationOperator,
                                     UInt32& numberOfCompleteVisibleItems,
                                     OrthoGridPageInfoCollection& pagesInfo,
                                     OrthoGridPageInfo& page0,
                                     OrthoGridPageInfo& page1,
                                     Candera::Node2D* firstItem,
                                     Int32 maxOrthogonalItemsCount,
                                     Int32 itemIndex,
                                     Int32 structureOffset,
                                     bool newPage0,
                                     bool newPage1) :
   _directOrientationOperator(directOrientationOperator),
   _orthogonalOrientationOperator(orthogonalOrientationOperator),
   _numberOfCompleteVisibleItems(numberOfCompleteVisibleItems),
   _pagesInfo(pagesInfo),
   _page0(page0),
   _page1(page1),
   _relPos(0),
   _orthoPos(0),
   _maxRelPos(0),
   _item(firstItem),
   _maxOrthogonalItemsCount(maxOrthogonalItemsCount),
   _itemIndex(itemIndex),
   _structureOffset(structureOffset),
   _newPage0(newPage0),
   _newPage1(newPage1)
{
}


void OrthoItemArranger::Layout(Courier::Vector<Int8> spans,
                               UInt32 structIndex,
                               UInt32 pageIndex,
                               Int32 numberOfCells,
                               Int32 startPosition,
                               Float deltaPosition,
                               UInt32 sizesCount)
{
   LayoutItems(spans, structIndex, pageIndex, numberOfCells, startPosition, deltaPosition);
   ComputeMaxRelPosition();

   const bool pageEnd(IsPageEnd(structIndex, pageIndex, sizesCount));
   EndPage0(pageEnd, pageIndex);
   EndPage1(pageEnd, pageIndex);
}


void OrthoItemArranger::FinishLayout(Int32 pageIndex)
{
   EndPage1(true, pageIndex);
}


bool OrthoItemArranger::IsPageEnd(UInt32 structIndex, UInt32 pageIndex, UInt32 sizesCount)
{
   bool val(false);

   if (_maxOrthogonalItemsCount != 0)
   {
      const Int32 nextStructPageIndex((structIndex + 1) / _maxOrthogonalItemsCount);
      val = (pageIndex != nextStructPageIndex) || ((structIndex + 1) == sizesCount);
   }

   return val;
}


void OrthoItemArranger::ComputeMaxRelPosition()
{
   // compute the maximum relative position
   if (_relPos > _maxRelPos)
   {
      _maxRelPos = _relPos;
   }
}


void OrthoItemArranger::LayoutItems(Internal::Vector<Int8>& spans,
                                    UInt32 structIndex,
                                    UInt32 pageIndex,
                                    Int32 numberOfCells,
                                    Int32 startPosition,
                                    Float deltaPosition)
{
   // start setting the row, column and span for each item; build page information
   if (_maxOrthogonalItemsCount != 0)
   {
      _orthoPos = static_cast<Int8>(structIndex % _maxOrthogonalItemsCount);
   }

   _relPos = static_cast<Int8>(pageIndex * numberOfCells);
   for (SizeType index(0); HasNextItem() && (index < spans.Size()); ++index)
   {
      const Int8 span(spans[index]);
      const Int32 absPos(startPosition + _relPos);

      _directOrientationOperator.SetRowOrColumn(*_item, _relPos + _structureOffset);
      _orthogonalOrientationOperator.SetRowOrColumn(*_item, _orthoPos);

      _directOrientationOperator.SetRowOrColumnSpan(*_item, static_cast<Int8>(span));
      _orthogonalOrientationOperator.SetRowOrColumnSpan(*_item, 1);

      if (_newPage0 && (pageIndex == 0))
      {
         _page0.RetainItemLayout(absPos, _orthoPos, span, 1, _itemIndex);
      }
      else if (_newPage1 && (pageIndex == 1))
      {
         _page1.RetainItemLayout(absPos, _orthoPos, span, 1, _itemIndex);
      }

      _item = _item->GetNextSibling();
      ++_itemIndex;

      if ((_relPos >= deltaPosition) && ((_relPos + span) <= (deltaPosition + numberOfCells)))
      {
         ++_numberOfCompleteVisibleItems;
      }

      _relPos += span;
   }
}


void OrthoItemArranger::EndPage0(bool pageEnd, UInt32 pageIndex)
{
   if (_newPage0 && pageEnd && (pageIndex == 0))
   {
      _newPage0 = false;
      _page0.SetEndItemIndex(_itemIndex - 1);
      _pagesInfo.AddPage(_page0);
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OrthoItemArranger::EndPage0 adding page 0: startIndex = %d, endIndex = %d, startPosition = %d, endPosition = %d", _page0.GetStartItemIndex(), _page0.GetEndItemIndex(), _page0.GetStartStructIndex(), _page0.GetEndStructIndex()));
   }
}


void OrthoItemArranger::EndPage1(bool pageEnd, UInt32 pageIndex)
{
   if (_newPage1 && pageEnd && (pageIndex == 1))
   {
      _newPage1 = false;
      _page1.SetStartItemIndex(_page0.GetEndItemIndex() + 1);
      _page1.SetEndItemIndex(_itemIndex - 1);
      _pagesInfo.AddPage(_page1);
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OrthoItemArranger::EndPage1 adding page 1: startIndex = %d, endIndex = %d, startPosition = %d, endPosition = %d", _page1.GetStartItemIndex(), _page1.GetEndItemIndex(), _page1.GetStartStructIndex(), _page1.GetEndStructIndex()));
   }
}
