/* ***************************************************************************************
* FILE:          OrthoOrthoGridAddingStrategy.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  OrthoGridAddingStrategy is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "Widgets/2D/List/Content/GridTemplateSpanAnalyzer.h"

#include "OrthoGridAddingStrategy.h"
#include "GridPositionsCalculator.h"
#include "Candera/System/Mathematics/Vector2.h"
#include "KeepEmptyOGEmptyCellsPolicy.h"
#include "DistributeToLastItemOGEmptyCellsPolicy.h"
#include "DistributeToAllItemsOGEmptyCellsPolicy.h"

using namespace FeatStd;
using namespace Candera;


OrthoGridAddingStrategy::SharedPointer OrthoGridAddingStrategy::Create(ISizeContainer& sizesContainer, const OrientationOperator& directOrientationOperator, const OrientationOperator& orthogonalOrientationOperator, const FeatStd::UInt8 rows, const FeatStd::UInt8 columns)
{
   return OrthoGridAddingStrategy::SharedPointer(FEATSTD_NEW(OrthoGridAddingStrategy)(sizesContainer, directOrientationOperator, orthogonalOrientationOperator, rows, columns));
}


void OrthoGridAddingStrategy::SetDynamicGridEmptyCellsPolicy(DynamicGridEmptyCellsPolicyType::Enum val)
{
   _dynamicGridEmptyCellsPolicy = val;
}


ItemAdder::SharedPointer OrthoGridAddingStrategy::CreateItemAdderImpl(Candera::Node2D* groupNode)
{
   return CreateOrthoGridItemAdder(groupNode);
}


void OrthoGridAddingStrategy::ProcessDataCHangedImpl(FeatStd::Int32 dataSize)
{
   _positionsCache.Clear();
   _positionsCache.UpdateCachStep(dataSize);

   _pagesInfo.Clear();
}


void OrthoGridAddingStrategy::AnalyzeTemplatesImpl(const ITemplateRetriever& templateRetriever)
{
   if (!_templateSpanAnalyzer.PointsToNull())
   {
      _templateSpanAnalyzer->Analyze(templateRetriever);
   }
}


FeatStd::Int32 OrthoGridAddingStrategy::CalculateStartIndexImpl(const ITemplateRetriever& templateRetriever, const ContentProvider& contentProvider, const Candera::Float position)
{
   const Int32 size(contentProvider.GetVirtualListSize());
   FeatStd::Int32 startIndex(GetPositionsCalculator().CalculateStartIndex(templateRetriever, contentProvider, position));

   _positionsCache.Cache(PositionsCache::CacheData(static_cast<Int32>(position), size == 0 ? startIndex : startIndex % size));

   return startIndex;
}


FeatStd::Float OrthoGridAddingStrategy::AdjustMaxPositionImpl(const ITemplateRetriever& /*templateRetriever*/, const ContentProvider& /*contentProvider*/, const Candera::UInt32 /*maxIndex*/)
{
   // TODO: implement more accurately
   return _pagesInfo.GetMaxStructStartIndex();
}


OrthoGridAddingStrategy::OrthoGridAddingStrategy(ISizeContainer& sizesContainer, const OrientationOperator& directOrientationOperator, const OrientationOperator& orthogonalOrientationOperator, const FeatStd::UInt8 rows, const FeatStd::UInt8 columns) :
   Base(sizesContainer, directOrientationOperator, orthogonalOrientationOperator),
   _templateSpanAnalyzer(GridTemplateSpanAnalyzer::Create(sizesContainer)),
   _rows(rows),
   _columns(columns),
   _positionsCalculator(_pagesInfo),
   _dynamicGridEmptyCellsPolicy(DynamicGridEmptyCellsPolicyType::DistributeToAllItems)
{
   _gridOccupation.SetAbsoluteStart(0, 0);
}


const PositionsCalculator& OrthoGridAddingStrategy::GetPositionsCalculator() const
{
   return _positionsCalculator;
}


void OrthoGridAddingStrategy::CacheImpl(const ContentProvider& /*contentProvider*/, const ITemplateRetriever& /*templateRetriever*/)
{
}


OrthoGridItemAdder::SharedPointer OrthoGridAddingStrategy::CreateOrthoGridItemAdder(Candera::Node2D* groupNode)
{
   OrthoGridItemAdder::SharedPointer adder(OrthoGridItemAdder::Create(_templateSpanAnalyzer, _sizesContainer, groupNode, _pagesInfo, _directOrientationOperator, _orthogonalOrientationOperator, _rows, _columns));

   if (!adder.PointsToNull())
   {
      adder->SetGridOccupation(&_gridOccupation);
      adder->SetEmptyCellsPolicy(CreateEmptyCellsPolicy());
   }

   return adder;
}


const PagesInfo& OrthoGridAddingStrategy::GetPagesInfoImpl() const
{
   return _pagesInfo;
}


OrthoGridEmptyCellsPolicy::SharedPointer OrthoGridAddingStrategy::CreateEmptyCellsPolicy() const
{
   OrthoGridEmptyCellsPolicy::SharedPointer policy(0);
   switch (_dynamicGridEmptyCellsPolicy)
   {
      case DynamicGridEmptyCellsPolicyType::KeepEmpty:
         policy = KeepEmptyOGEmptyCellsPolicy::Create();
         break;
      case DynamicGridEmptyCellsPolicyType::DistributeToAllItems:
         policy = DistributeToAllItemsOGEmptyCellsPolicy::Create();
         break;
      case DynamicGridEmptyCellsPolicyType::DistributeToLastItem:
         policy = DistributeToLastItemOGEmptyCellsPolicy::Create();
         break;
      default:
         break;
   }
   return policy;
}
