/* ***************************************************************************************
* FILE:          ItemAdder.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ItemAdder is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "ItemAdder.h"
#include <Candera/System/Mathematics/Math.h>
#include <Widgets/2D/Common/NodeHelper.h>

using namespace Candera;

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ItemAdder.cpp.trc.h"
#endif

CANDERA_RTTI_BASECLASS_DEFINITION(ItemAdder);

ItemAdder::ItemAdder(ISizeContainer& sizesContainer, Candera::Node2D* groupNode, const OrientationOperator& directOrientationOperator, const OrientationOperator& orthogonalOrientationOperator) :
   _originalPosition(0.0F),
   _adjustedPosition(0.0F),
   _visibleArea(),
   _numberOfCompleteVisibleItems(0),
   _addedItems(0),
   _groupNode(groupNode),
   _directOrientationOperator(directOrientationOperator),
   _orthogonalOrientationOperator(orthogonalOrientationOperator),
   _beginingMargin(0),
   _endingMargin(0),
   _reverse(false),
   _currentSize(0.0F),
   _actualSize(0.0F),
   _sizesContainer(sizesContainer),
   _beginingSize(0.0F),
   _endingSize(0.0F),
   _normalizer(0),
   _zeroSizedItems(0),
   _structureOffset(0)
{
}


ItemAdder::~ItemAdder()
{
   _groupNode = 0;
   _normalizer = 0;
}


FeatStd::Float ItemAdder::BeginAdding(const Candera::Vector2& visibleArea, Candera::Float originalPosition)
{
   _originalPosition = originalPosition;
   _visibleArea = visibleArea;

   OnBeforeBeginAdding();
   _adjustedPosition = _originalPosition + GetPositionCorrection(originalPosition);
   OnBeginAdding();
   Clear();

   return _adjustedPosition;
}


void ItemAdder::BeginReverseAdding()
{
   OnBeginReverseAdding();
}


void ItemAdder::FinishDirectAdding()
{
   OnFinishDirectAdding();
}


void ItemAdder::FinishAdding()
{
   OnFinishAdding();
}


void ItemAdder::FinishReverseAdding()
{
   OnFinishReverseAdding();
}


Candera::Float ItemAdder::GetItemSize(Candera::Node2D& node)
{
   return _directOrientationOperator.GetVectorComponent(_sizesContainer.MeasureClientSize(&node));
}


Candera::UInt32 ItemAdder::GetNumberOfCompleteVisibleItems() const
{
   return _numberOfCompleteVisibleItems;
}


FeatStd::Float ItemAdder::InsertItem(Candera::Node2D& childNode)
{
   if (0 != _groupNode)
   {
      if (_reverse)
      {
         Node2D* firstChild = _groupNode->GetFirstChild();
         _groupNode->InsertBeforeChild(&childNode, firstChild);
      }
      else
      {
         static_cast<void>(_groupNode->AddChild(&childNode));
      }
   }

   ++_numberOfCompleteVisibleItems;
   ++_addedItems;

   const Float itemSize(GetItemSize(childNode));
   if (itemSize == 0.0F)
   {
      ++_zeroSizedItems;
   }
   return itemSize;
}


Candera::Layouter* ItemAdder::GetGroupNodeLayouter() const
{
   Candera::Layouter* layouter = 0;
   if (0 != _groupNode)
   {
      layouter = _groupNode->GetLayouter();
   }
   return layouter;
}


Candera::Float ItemAdder::GetOriginalPosition() const
{
   return _originalPosition;
}


Candera::Float ItemAdder::GetAdjustedPosition() const
{
   return _adjustedPosition;
}


Candera::Vector2 ItemAdder::GetVisibleArea() const
{
   return _visibleArea;
}


Candera::Int32 ItemAdder::GetAddedItems() const
{
   return _addedItems;
}


void ItemAdder::Clear()
{
   _numberOfCompleteVisibleItems = 0;
   _addedItems = 0;
   _zeroSizedItems = 0;
   _beginingMargin = 0;
   _endingMargin = 0;
   _beginingSize = 0.0F;
   _endingSize = 0.0F;
   _currentSize = 0.0F;
   _actualSize = 0.0F;
   _sizes.Clear();
}


Candera::Float ItemAdder::GetPostAddSizeHint(Candera::Node2D& childNode, const Candera::Vector2& visibleAreaInPixels)
{
   Courier::Vector<FeatStd::Float> sizes(GetSizes());
   Float currentSize(GetCurrentSize());
   if (sizes.Size() == 0)
   {
      const Vector2 size2D(NodeHelper::GetSize(&childNode, visibleAreaInPixels));
      const Float itemSize(_directOrientationOperator.GetVectorComponent(size2D));
      sizes.Add(itemSize);
      UpdateBeginingMargin(sizes);
      currentSize = GetCurrentSize();
      _currentSize = 0.0F;
      _beginingMargin = 0;
   }
   return currentSize;
}


Candera::Int16 ItemAdder::GetFirstItemMargin() const
{
   return _beginingMargin;
}


Candera::Int16 ItemAdder::GetLastItemMargin() const
{
   return _endingMargin;
}


void ItemAdder::SetReverseMode(bool reverse)
{
   _reverse = reverse;
}


Candera::Float ItemAdder::GetBeginingSize() const
{
   return _beginingSize;
}


Candera::Float ItemAdder::GetEndingSize() const
{
   return _endingSize;
}


Candera::Int16 ItemAdder::GetBeginingMargin() const
{
   return _beginingMargin;
}


Candera::Int16 ItemAdder::GetEndingMargin() const
{
   return _endingMargin;
}


Candera::Node2D* ItemAdder::GetGroupNode() const
{
   return _groupNode;
}


Candera::Int16 ItemAdder::ComputeStartMargin(const Courier::Vector<FeatStd::Float>& sizes, Candera::Float position) const
{
   const Float normalizedPosition(Math::Floor(position) - position);
   Float val(normalizedPosition);
   if (0 != _normalizer)
   {
      val = _normalizer->Denormalize(val, sizes);
   }
   return Int16(val);
}


const Int32 ItemAdder::GetNumberOfAddedItems() const
{
   return _addedItems;
}


Candera::Float ItemAdder::GetSizeOfAddedElements() const
{
   return _actualSize;
}


void ItemAdder::UpdateBeginingMargin(const Courier::Vector<FeatStd::Float>& sizes)
{
   const Candera::Int16 oldBeginingMargin(_beginingMargin);
   _beginingMargin = ComputeStartMargin(sizes, _adjustedPosition);

   _currentSize += _beginingMargin - oldBeginingMargin;

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ItemAdder::UpdateBeginingMargin adjustedPosition=%f, currentSize=%f, beginingMargin=%d, numberOfItems=%d", _adjustedPosition, _currentSize, _beginingMargin, _sizes.Size()));
}


void ItemAdder::CollectSize(FeatStd::Float size)
{
   if (_reverse)
   {
      _sizes.Insert(0, size);
   }
   else
   {
      _sizes.Add(size);
   }
}


void ItemAdder::ResetSize()
{
   _currentSize = 0;
   _actualSize = 0;
}


void ItemAdder::IncreaseSize(FeatStd::Float val)
{
   _currentSize += val;
   _actualSize += val;
}
