/* ***************************************************************************************
* FILE:          ItemSpeedAnimationManager.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ItemSpeedAnimationManager is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "ItemSpeedAnimationManager.h"

ItemSpeedAnimationManager::ItemSpeedAnimationManager() :
   _sizesCount(0),
   _currentPosition(-1),
   _currentSize(0),
   _isActive(false),
   _swipingState(SwipingState::Idle)
{};


FeatStd::Float ItemSpeedAnimationManager::GetSizeFromVelocity(FeatStd::Float velocity)
{
   if (velocity >= 0)
   {
      return (velocity * VELOCITY_MULTIPLIER);
   }
   else if (velocity < 0)
   {
      return (velocity * (-VELOCITY_MULTIPLIER));
   }
}


void ItemSpeedAnimationManager::ComputeCurrentSize()
{
   FeatStd::Float size(0);

   switch (_swipingState)
   {
      case SwipingState::Scrolling:
         size = GetSizeMed();
         _isActive = true;
         break;

      case SwipingState::Swiping:
         size = _sizes[0];
         _isActive = true;
         break;

      case SwipingState::Idle:
         if (_isActive)
         {
            if (_currentSize == 0)
            {
               // Reset
               _isActive = false;
               _sizesCount = 0;
               _currentPosition = -1;
            }
            else
            {
               InsertNewSize(0);
               size = GetSizeMed();
            }
         }
         break;

      default:
         break;
   }
   _currentSize = static_cast<FeatStd::UInt32>(size);
}


void ItemSpeedAnimationManager::RegisterSize(FeatStd::Float velocity, const SwipingState::Enum state)
{
   _swipingState = state;
   switch (_swipingState)
   {
      case SwipingState::Scrolling:
         InsertNewSize(GetSizeFromVelocity(velocity));
         break;

      case SwipingState::Swiping:
         _sizesCount = 1;
         _currentPosition = 0;
         _sizes[0] = GetSizeFromVelocity(velocity);
         break;

      default:
         // ignore velocity
         break;
   }
}


FeatStd::Float ItemSpeedAnimationManager::GetSizeMed()
{
   FeatStd::Float sum = 0;
   for (int i = 0; i < _sizesCount; i++)
   {
      sum += _sizes[i];
   }
   return (sum / _sizesCount);
}


void ItemSpeedAnimationManager::InsertNewSize(FeatStd::Float velocity)
{
   if (_currentPosition == (BUFFER_SIZE - 1))
   {
      _currentPosition = 0;
   }
   else
   {
      _currentPosition++;
   }

   _sizes[_currentPosition] = velocity;

   if (_sizesCount < BUFFER_SIZE)
   {
      _sizesCount++;
   }
}
