/* ***************************************************************************************
* FILE:          FlexListLimiterAnimationFactory.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FlexListLimiterAnimationFactory is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "FlexListLimiterAnimationFactory.h"
#include <Candera/EngineBase/Animation/AnimationPlayer.h>
#include <Candera/EngineBase/Animation/AnimationController.h>
#include <Candera/EngineBase/Animation/AnimationBlendedProperty.h>
#include <Candera/EngineBase/Animation/EaseInterpolationStrategy.h>
#include <Candera/EngineBase/Animation/BackEaseFunction.h>
#include <Candera/EngineBase/Animation/AnimationKeyframeSequence.h>

using namespace Candera;


const UInt32 c_marginBounceKeyframeCount = 3;
const UInt32 c_marginKeyframeValueElements = 4;

Candera::MemoryManagement::SharedPointer<Animation::AnimationPlayer> FlexListMarginBounceAnimationFactory::Create(Node2D* node, UInt32 durationMs, Candera::Vector2 bounceDirection)
{
   CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, "easeFunction is wrapped by the interpolation strategy.");

   Candera::MemoryManagement::SharedPointer<Animation::AnimationPlayer> result(0);

   bool created = false;
   MarginPropertySetter* marginPropertySetter = 0;
   Candera::MemoryManagement::SharedPointer<Animation::AnimationBlendedProperty> blendedProperty(0);
   Candera::MemoryManagement::SharedPointer<Animation::AnimationKeyframeSequence> keyFrameSequence(0);

   Animation::EaseInterpolationStrategy::SharedPointer interpolationStrategy;
   Animation::BackEaseFunction::SharedPointer easeFunction;
   Animation::AnimationController::SharedPointer animationController;

   Animation::SequenceTimeType* keyFrameTimes = 0;
   Float* keyFrameValues = 0;

   if ((node != 0) && (durationMs > 0))
   {
      result = Base::CreatePlainAnimation();
      if (!result.PointsToNull())
      {
         //Create AnimationBlendedProperty.
         marginPropertySetter = MarginPropertySetter::Create(node);
         blendedProperty = Animation::AnimationBlendedProperty::Create();

         if (!blendedProperty.PointsToNull())
         {
            blendedProperty->SetAnimationPropertySetter(Candera::Animation::AnimationPropertySetter::SharedPointer(marginPropertySetter)); //Takes ownership of marginPropertySetter* and disposes when needed.
         }

         //Create keyframe sequence.
         keyFrameSequence = Animation::AnimationKeyframeSequence::Create();

         //Create and setup interpolation strategy. We want to realize a bounce. Setup keyframes.
         interpolationStrategy = Animation::EaseInterpolationStrategy::Create();

         easeFunction = Animation::BackEaseFunction::SharedPointer(FEATSTD_NEW(Animation::BackEaseFunction));
         if ((!interpolationStrategy.PointsToNull()) && (!easeFunction.PointsToNull()) && (keyFrameSequence != 0))

         {
            easeFunction->SetAmplitude(1.0F);

            interpolationStrategy->SetEaseDirection(Animation::EaseInterpolationStrategy::EaseIn);

            interpolationStrategy->SetEaseFunction(easeFunction);

            keyFrameTimes = ComputeKeyFrameTimes(durationMs);
            keyFrameValues = ComputeKeyFrameValues(node, bounceDirection);

            if ((keyFrameTimes != 0) && (keyFrameValues != 0))
            {
               keyFrameSequence->SetKeyframes(c_marginKeyframeValueElements, c_marginBounceKeyframeCount,
                                              keyFrameTimes, Animation::KeyframeSequence::TimeTypeDisposer::Dispose,
                                              keyFrameValues, Animation::KeyframeSequence::ValuesDisposer::Dispose);

               //Set interpolation strategy to keyframesequence.

               keyFrameSequence->SetInterpolationStrategy(interpolationStrategy);

               //Almost there: Add keyframe sequence to blended property.
               static_cast<void>(blendedProperty->AddKeyframeSequence(keyFrameSequence));

               //Finally Create AnimationController.
               animationController = Candera::Animation::AnimationController::Create();
               if (animationController != 0)
               {
                  static_cast<void>(animationController->AddProperty(blendedProperty));
                  animationController->SetSequenceReferenceTime(0);
                  result->SetController(animationController);
                  created = true;
               }
            }
            else
            {
               if (keyFrameTimes != 0)
               {
                  Animation::KeyframeSequence::TimeTypeDisposer::Dispose(keyFrameTimes);
               }

               if (keyFrameValues != 0)
               {
                  Animation::KeyframeSequence::ValuesDisposer::Dispose(keyFrameValues);
               }
            }
         }
      }
   }

   if (!created)
   {
      _TODO("dispose animation has to be checked!")
      //Dispose(result, animationController, keyFrameSequence, keyFrameValues, keyFrameTimes, interpolationStrategy, easeFunction, blendedProperty, marginPropertySetter);
   }

   return result;
}


Candera::Animation::SequenceTimeType* FlexListMarginBounceAnimationFactory::ComputeKeyFrameTimes(Candera::UInt32 durationMs)
{
   Animation::SequenceTimeType* keyFrameTimes = FEATSTD_NEW_ARRAY(Animation::SequenceTimeType, c_marginBounceKeyframeCount);
   if (keyFrameTimes != 0)
   {
      for (UInt32 i = 0; i < c_marginBounceKeyframeCount; i++)
      {
         CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(774, "Using constant for configuration.")
         if ((c_marginBounceKeyframeCount - 1) > 0)
         {
            keyFrameTimes[i] = Int32(i * (durationMs / (c_marginBounceKeyframeCount - 1)));
         }
         else
         {
            keyFrameTimes[i] = 0;
         }
      }
   }

   return keyFrameTimes;
}


Candera::Float* FlexListMarginBounceAnimationFactory::ComputeKeyFrameValues(Candera::Node2D* node, Candera::Vector2 bounceDirection)
{
   UInt32 arraySize = c_marginBounceKeyframeCount * c_marginKeyframeValueElements;
   Float* result = FEATSTD_NEW_ARRAY(Float, c_marginBounceKeyframeCount * c_marginKeyframeValueElements);
   bool filled = false;
   if ((result != 0) && (node != 0))
   {
      Layouter* layouter = node->GetLayouter();
      if (layouter != 0)
      {
         const Margin& margin = Layouter::GetMargin(*node);
         Vector2 marginLeftTop = Vector2(static_cast<Float>(margin.GetLeft()), static_cast<Float>(margin.GetTop()));
         Vector2 marginRightBotton = Vector2(static_cast<Float>(margin.GetRight()), static_cast<Float>(margin.GetBottom()));

         CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(774, "Using constant for configuration.")
         if (arraySize >= c_marginKeyframeValueElements)
         {
            //Fill first and last element with original value.
            result[0] = marginLeftTop.GetX();
            result[1] = marginLeftTop.GetY();
            result[2] = marginRightBotton.GetX();
            result[3] = marginRightBotton.GetY();
            result[arraySize - c_marginKeyframeValueElements] = marginLeftTop.GetX();
            result[(arraySize - c_marginKeyframeValueElements) + 1] = marginLeftTop.GetY();
            result[(arraySize - c_marginKeyframeValueElements) + 2] = marginRightBotton.GetX();
            result[(arraySize - c_marginKeyframeValueElements) + 3] = marginRightBotton.GetY();
         }

         //for all intermediate elements calculate margin.
         for (UInt32 i = 1; i < c_marginBounceKeyframeCount - 1; i++)
         {
            Vector2 marginBounce = marginLeftTop + bounceDirection;
            result[c_marginKeyframeValueElements * i] = marginBounce.GetX();
            result[c_marginKeyframeValueElements * i + 1] = marginBounce.GetY();
            result[c_marginKeyframeValueElements * i + 2] = marginRightBotton.GetX();
            result[c_marginKeyframeValueElements * i + 3] = marginRightBotton.GetY();
         }

         filled = true;
      }
   }

   if ((!filled) && (result != 0))
   {
      Candera::MemoryPlatform::Set(result, 0, c_marginBounceKeyframeCount * c_marginKeyframeValueElements * sizeof(Float));
   }

   return result;
}


void FlexListMarginBounceAnimationFactory::Dispose(Candera::MemoryManagement::SharedPointer<Animation::AnimationPlayer>& result,
      Animation::AnimationController*& animationController, const Candera::MemoryManagement::SharedPointer<Animation::AnimationKeyframeSequence>& keyFrameSequence,
      Float*& keyFrameValues, Animation::SequenceTimeType*& keyFrameTimes, Animation::EaseInterpolationStrategy*& interpolationStrategy, Animation::BackEaseFunction*& easeFunction,
      const Candera::MemoryManagement::SharedPointer<Animation::AnimationBlendedProperty>& blendedProperty, MarginPropertySetter*& marginPropertySetter)
{
   //If the animation was created successfully, all referenced pointers will be disposed by the animation itself.
   //Otherwise cleanup needs to be done.
   //MarginPropertySetter* marginPropertySetter = 0;
   //MemoryManagement::SharedPointer<AnimationBlendedProperty> blendedProperty(0);
   //MemoryManagement::SharedPointer<AnimationKeyframeSequence> keyFrameSequence(0);

   if (!result.PointsToNull())
   {
      result->SetController(Animation::AnimationController::SharedPointer());

      result = Candera::MemoryManagement::SharedPointer<Animation::AnimationPlayer>(0); //deletes all references and discards player.
   }

   if (animationController != 0)
   {
      static_cast<void>(animationController->RemoveProperty(0));
      FEATSTD_DELETE(animationController);
      animationController = 0;
   }

   if (keyFrameSequence.PointsToNull())
   {
      if (keyFrameValues != 0)
      {
         FEATSTD_DELETE_ARRAY(keyFrameValues);
         keyFrameValues = 0;
      }

      if (keyFrameTimes != 0)
      {
         FEATSTD_DELETE_ARRAY(keyFrameTimes);
         keyFrameTimes = 0;
      }

      if (interpolationStrategy != 0)
      {
         interpolationStrategy->SetEaseFunction(Animation::AbstractEasingFunction::SharedPointer());
         FEATSTD_DELETE(interpolationStrategy);
         interpolationStrategy = 0;
      }

      if (easeFunction != 0)
      {
         FEATSTD_DELETE(easeFunction);
         easeFunction = 0;
      }
   }

   if (blendedProperty.PointsToNull())
   {
      FEATSTD_DELETE(marginPropertySetter);
      marginPropertySetter = 0;
   }
}
