/* ***************************************************************************************
* FILE:          ExpandAdjuster.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ExpandAdjuster is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"

#include "ExpandAdjuster.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ExpandAdjuster.cpp.trc.h"
#endif

using namespace Courier;

ExpandAdjuster::ExpandAdjuster(ItemSizesContainer* sizesContainer, PositioningHandler& positioner) :
   _sizesContainer(sizesContainer),
   _normalizer(0),
   _positioner(positioner),
   _initialMargin(0),
   _currentAdjustingItemIndex(-1)
{
}


ExpandAdjuster::~ExpandAdjuster()
{
   _sizesContainer = 0;
   _normalizer = 0;
}


void ExpandAdjuster::ExpandStarted(const AdjustData& data, FeatStd::Int16 endMargin)
{
   if (data._lastIsRunning)
   {
      if (data._lastIsForward)
      {
         // Used on all forward expandings
         _initialMargin = endMargin * (-1);
      }
      else
      {
         // Used when colapsing last element from the list
         _initialMargin = endMargin;
      }
   }
   else
   {
      // When not the last element is expanding/colapsing, there will be no initial margin
      _initialMargin = 0;
   }

   _currentAdjustingItemIndex = data._lastRunningIndex;
}


void ExpandAdjuster::PrepareAdjustment(const bool lastViewPort)
{
   if (lastViewPort)
   {
      // Reset max position when expanding reach last viewport
      _positioner.ResetMaximumPositions();
   }
}


bool ExpandAdjuster::Adjust(const AdjustData& data, FeatStd::Int16 endMargin)
{
   bool adjusted(false);

   // When last visible item is expanding or last item of the list is colapsing
   if (data._lastIsRunning && (data._lastIsForward || ((!data._lastIsForward) && data._lastViewport)))
   {
      Float delta = static_cast<Float>((endMargin * (-1)) - _initialMargin);
      const Float currentPosition(_positioner.GetCurrentPosition());
      const Float normalizedDelta(_normalizer->Normalize(delta, _sizesContainer->GetItemsSizes()));
      Float targetPosition(currentPosition + normalizedDelta);

      if (targetPosition > static_cast<Float>(_currentAdjustingItemIndex))
      {
         // When expanded item is larger than the viewPort, limit the expanding to the first visible pixels of that item
         targetPosition = static_cast<Float>(_currentAdjustingItemIndex);
      }
      if (currentPosition != targetPosition)
      {
         _positioner.SetCurrentPosition(targetPosition);
         _positioner.SetTargetPosition(targetPosition);

         adjusted = true;
      }
   }
   return adjusted;
}
