/* ***************************************************************************************
* FILE:          ImageEffectWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ImageEffectWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "CanderaPlatform/Device/Common/Effects/BitmapBrushBlend.h"
#include "ImageEffectWidget2D.h"

#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_IMAGE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ImageEffectWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(ImageEffectWidget2D);


ImageEffectWidget2D::ImageEffectWidget2D(): Base(),
   _invalid(true)
{
}


ImageEffectWidget2D::~ImageEffectWidget2D()
{
}


void ImageEffectWidget2D::OnChanged(::Candera::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   if (propertyId == BitmapPropertyId)
   {
      _invalid = true;
   }
}


bool ImageEffectWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const ImageEffectWidget2D* original = CLONEABLE_WIDGET_CAST<const ImageEffectWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }

      SetBitmap(original->GetBitmap());
      hmibase::widget::utils::EffectUtils::copyColor(original->GetNode(), GetNode());

      cloned = true;
   }
   return cloned;
}


void ImageEffectWidget2D::Update()
{
   Base::Update();

   if (_invalid && (GetNode() != NULL))
   {
      ImageEffectWidget2D::SetImage(*this, GetBitmap());

      _invalid = false;
   }
}


void ImageEffectWidget2D::SetImage(BaseWidget2D& widget, const Candera::MemoryManagement::SharedPointer<Candera::Image2D>& image)
{
   Candera::RenderNode* renderNode = Candera::Dynamic_Cast<Candera::RenderNode*>(widget.GetNode());
   if ((renderNode != NULL) && (renderNode->GetEffect(0) != NULL))
   {
      Candera::BitmapBrush* bitmapBrush = Candera::Dynamic_Cast<Candera::BitmapBrush*>(renderNode->GetEffect(0)->GetBrushEffect2D());
      if (bitmapBrush != NULL)
      {
         ImageEffectWidget2D::SetImage(widget, *bitmapBrush, image);
      }
   }
}


void ImageEffectWidget2D::SetImage(BaseWidget2D& widget, Candera::BitmapBrush& bitmapBrush, const Candera::MemoryManagement::SharedPointer<Candera::Image2D>& image)
{
   Candera::Image2D* oldImage = bitmapBrush.Image().Get();

   //if both bitmaps are NULL or they point to same image=> do nothing
   if (oldImage != image.GetPointerToSharedInstance())
   {
      if (image.PointsToNull())
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SetImage image=null %s", HMIBASE_TO_STRING_VW(&widget)));
      }
      else
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SetImage %s", HMIBASE_TO_STRING_VW(&widget)));
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SetImage image=%s", HMIBASE_TO_STRING((const Candera::Image2D*)image.GetPointerToSharedInstance())));
      }
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "SetImage node=%s", HMIBASE_TO_STRING_SPN2D(widget.GetNode())));

      bool invalidateLayout = false;
      //if one of the bitmaps is NULL and the other one non NULL or if size is different=>Invalidate layout
      if ((oldImage == NULL)
            || image.PointsToNull()
            || (oldImage->GetWidth() != image->GetWidth())
            || (oldImage->GetHeight() != image->GetHeight()))
      {
         invalidateLayout = true;
      }

      bitmapBrush.Unload();
      bitmapBrush.Image().Set(image);
      bitmapBrush.Upload();

      if (invalidateLayout)
      {
         //invalidate the layout because size changed
         Candera::Layouter::InvalidateLayout(widget.GetNode());
      }

      //this is for maximum compatibility, to be replaced with the invalidate conditioned by the invalidateLayout condition
      widget.Invalidate();
   }
}
