/* ***************************************************************************************
* FILE:          GestureWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  GestureWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "GestureWidget2D.h"

#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_GESTURE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/GestureWidget2D.cpp.trc.h"
#endif


namespace hmibase {
namespace widget {
namespace gesture {


CGI_WIDGET_RTTI_DEFINITION(GestureWidget2D);

using ::Candera::Char;
CANDERA_RTTI_DEFINITION(GestureController2D)
CANDERA_RTTI_DEFINITION(DefaultGestureController2D)


/*****************************************************************************/
GestureWidget2D::GestureWidget2D() : Base()
{
   //make the widget touchable
   SetTouchable(true);
}


/*****************************************************************************/
GestureWidget2D::~GestureWidget2D()
{
}


/*****************************************************************************/
bool GestureWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned = false;
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const GestureWidget2D* original = CLONEABLE_WIDGET_CAST<const GestureWidget2D*>(originalWidget);
      if (original != NULL)
      {
         cloned = true;
      }
   }
   return cloned;
}


/*****************************************************************************/
WidgetController* GestureWidget2D::GetDefaultController() const
{
   return &(DefaultGestureController2D::getInstance());
}


/*****************************************************************************/
bool GestureController2D::OnTapGesture(DelegateWidget& baseWidget, const hmibase::input::gesture::GestureEvent& gestureData)
{
   return postTapGestureMessage(baseWidget, gestureData);
}


/*****************************************************************************/
bool GestureController2D::OnDragGesture(DelegateWidget& baseWidget, const hmibase::input::gesture::GestureEvent& gestureData)
{
   return postDragGestureMessage(baseWidget, gestureData);
}


/*****************************************************************************/
bool GestureController2D::OnSwipeGesture(DelegateWidget& baseWidget, const hmibase::input::gesture::GestureEvent& gestureData)
{
   return postSwipeGestureMessage(baseWidget, gestureData);
}


/*****************************************************************************/
bool GestureController2D::OnPinchSpreadGesture(DelegateWidget& baseWidget, const hmibase::input::gesture::GestureEvent& gestureData)
{
   return postPinchSpreadGestureMessage(baseWidget, gestureData);
}


/*****************************************************************************/
bool GestureController2D::OnRotateGesture(DelegateWidget& baseWidget, const hmibase::input::gesture::GestureEvent& gestureData)
{
   return postRotateGestureMessage(baseWidget, gestureData);
}


/*****************************************************************************/
enGestureEvent::Enum DefaultGestureController2D::getGestureEvent(hmibase::input::gesture::GestureEvent::GestureState gestureState)
{
   switch (gestureState)
   {
      case hmibase::input::gesture::GestureEvent::ET_START:
         return enGestureEvent::Start;
      case hmibase::input::gesture::GestureEvent::ET_END:
         return enGestureEvent::End;
      case hmibase::input::gesture::GestureEvent::ET_ABORT:
         return enGestureEvent::Abort;
      case hmibase::input::gesture::GestureEvent::ET_MOVE:
         return enGestureEvent::Move;
      case hmibase::input::gesture::GestureEvent::ET_HOLD:
         return enGestureEvent::Hold;
      default:
         return enGestureEvent::Abort;
   }
}


/*****************************************************************************/
bool DefaultGestureController2D::postTapGestureMessage(DelegateWidget& widget, const hmibase::input::gesture::GestureEvent& gestureData)
{
   Courier::View* view = widget.GetParentView();
   if (view != NULL)
   {
      enTapEvent::Enum tapEvent(enTapEvent::Abort);
      switch (gestureData._gestureState)
      {
         //case hmibase::input::gesture::GestureEvent::ET_START:
         case hmibase::input::gesture::GestureEvent::ET_INITIATE:
            tapEvent = enTapEvent::Press;
            break;
         case hmibase::input::gesture::GestureEvent::ET_END:
            tapEvent = enTapEvent::Release;
            break;
         case hmibase::input::gesture::GestureEvent::ET_HOLD:
            tapEvent = enTapEvent::Hold;
            break;
         case hmibase::input::gesture::GestureEvent::ET_REPEAT:
            tapEvent = enTapEvent::Repeat;
            break;
         case hmibase::input::gesture::GestureEvent::ET_DT_START:
            tapEvent = enTapEvent::DoubleTap;
            break;
         case hmibase::input::gesture::GestureEvent::ET_ABORT:
            tapEvent = enTapEvent::Abort;
            break;
         default:
            break;
      }

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PostTapGestureMessage evt=[%d] p1=%12s %s",
                          ETG_CENUM(enTapEvent::Enum, tapEvent),
                          HMIBASE_TO_STRING(gestureData._pt1),
                          HMIBASE_TO_STRING_VW(&widget)));

      GestureMsg* msg = COURIER_MESSAGE_NEW(TapGestureMsg)(view->GetId(), Courier::Identifier(widget.GetLegacyName()), widget.GetUserData(), gestureData._nTimestamp,
                        tapEvent, gestureData._pt1.x, gestureData._pt1.y);
      if (msg != NULL)
      {
         return msg->Post();
      }
   }
   return false;
}


/*****************************************************************************/
bool DefaultGestureController2D::postDragGestureMessage(DelegateWidget& widget, const hmibase::input::gesture::GestureEvent& gestureData)
{
   Courier::View* view = widget.GetParentView();
   if (view != NULL)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PostDragGestureMessage evt=[%d] p1=%12s v1=%12s %s",
                          ETG_CENUM(enGestureEvent::Enum, getGestureEvent(gestureData._gestureState)),
                          HMIBASE_TO_STRING(gestureData._pt1),
                          HMIBASE_TO_STRING(gestureData._velocity1),
                          HMIBASE_TO_STRING_VW(&widget)));

      GestureMsg* msg = COURIER_MESSAGE_NEW(DragGestureMsg)(view->GetId(), Courier::Identifier(widget.GetLegacyName()), widget.GetUserData(), gestureData._nTimestamp,
                        getGestureEvent(gestureData._gestureState), gestureData._pt1.x, gestureData._pt1.y, gestureData._velocity1.x, gestureData._velocity1.y);
      if (msg != NULL)
      {
         return msg->Post();
      }
   }
   return false;
}


/*****************************************************************************/
bool DefaultGestureController2D::postSwipeGestureMessage(DelegateWidget& widget, const hmibase::input::gesture::GestureEvent& gestureData)
{
   Courier::View* view = widget.GetParentView();
   if (view != NULL)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PostSwipeGestureMessage evt=[%d] v1=%12s v2=%12s v3=%12s %s",
                          ETG_CENUM(enGestureEvent::Enum, getGestureEvent(gestureData._gestureState)),
                          HMIBASE_TO_STRING(gestureData._velocity1),
                          HMIBASE_TO_STRING(gestureData._velocity2),
                          HMIBASE_TO_STRING(gestureData._velocity3),
                          HMIBASE_TO_STRING_VW(&widget)));

      GestureMsg* msg = COURIER_MESSAGE_NEW(SwipeGestureMsg)(view->GetId(), Courier::Identifier(widget.GetLegacyName()), widget.GetUserData(), gestureData._nTimestamp,
                        gestureData._velocity1.x, gestureData._velocity1.y,
                        gestureData._velocity2.x, gestureData._velocity2.y,
                        gestureData._velocity3.x, gestureData._velocity3.y);
      if (msg != NULL)
      {
         return msg->Post();
      }
   }
   return false;
}


/*****************************************************************************/
bool DefaultGestureController2D::postPinchSpreadGestureMessage(DelegateWidget& widget, const hmibase::input::gesture::GestureEvent& gestureData)
{
   Courier::View* view = widget.GetParentView();
   if (view != NULL)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PostPinchSpreadGestureMessage evt=[%d] p1=%12s p2=%12s %s",
                          ETG_CENUM(enGestureEvent::Enum, getGestureEvent(gestureData._gestureState)),
                          HMIBASE_TO_STRING(gestureData._pt1),
                          HMIBASE_TO_STRING(gestureData._pt2),
                          HMIBASE_TO_STRING_VW(&widget)));

      GestureMsg* msg = COURIER_MESSAGE_NEW(PinchSpreadGestureMsg)(view->GetId(), Courier::Identifier(widget.GetLegacyName()), widget.GetUserData(), gestureData._nTimestamp,
                        getGestureEvent(gestureData._gestureState), gestureData._pt1.x, gestureData._pt1.y, gestureData._pt2.x, gestureData._pt2.y);
      if (msg != NULL)
      {
         return msg->Post();
      }
   }
   return false;
}


/*****************************************************************************/
bool DefaultGestureController2D::postRotateGestureMessage(DelegateWidget& widget, const hmibase::input::gesture::GestureEvent& gestureData)
{
   Courier::View* view = widget.GetParentView();
   if (view != NULL)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PostRotateGestureMessage evt=[%d] angle=[%d] p1=%12s p2=%12s %s",
                          ETG_CENUM(enGestureEvent::Enum, getGestureEvent(gestureData._gestureState)),
                          gestureData._rotateAngle,
                          HMIBASE_TO_STRING(gestureData._pt1),
                          HMIBASE_TO_STRING(gestureData._pt2),
                          HMIBASE_TO_STRING_VW(&widget)));

      GestureMsg* msg = COURIER_MESSAGE_NEW(RotateGestureMsg)(view->GetId(), Courier::Identifier(widget.GetLegacyName()), widget.GetUserData(), gestureData._nTimestamp,
                        getGestureEvent(gestureData._gestureState), gestureData._rotateAngle, gestureData._pt1.x, gestureData._pt1.y, gestureData._pt2.x, gestureData._pt2.y);
      if (msg != NULL)
      {
         return msg->Post();
      }
   }
   return false;
}


}
}


}
