/* ***************************************************************************************
* FILE:          FocusGroupWidget2D.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FocusGroupWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef __FOCUS_GROUP_WIDGET2D_H__
#define __FOCUS_GROUP_WIDGET2D_H__

#include "Focusable2D.h"
#include <Focus/FData.h>
#include <Widgets/2D/BaseWidget2D.h>
#include <Widgets/2D/ControlTemplate/ControlTemplateCloneableWidget.h>

#include <Candera/System/MetaInfo/BaseInternalMacros.h>
#include <Candera/System/MetaInfo/EnumDataTypeInternal.h>

CANDERA_INTERNAL_ENUM_METAINFO_BEGIN(::Focus::FAnchorMode::Enum, _Focus_FAnchorMode_Enum)
CANDERA_INTERNAL_ENUM_METAINFO_ITEM(None)
CANDERA_INTERNAL_ENUM_METAINFO_ITEM(Container)
CANDERA_INTERNAL_ENUM_METAINFO_ITEM(Content)
CANDERA_INTERNAL_ENUM_METAINFO_END()

class FocusGroupBase
{
   public:
      FocusGroupBase();
      virtual ~FocusGroupBase();

      /////////////////////////////////////////////////////////////////////////
      //configured
      bool getGroupConfigured() const
      {
         return _groupData.Configured;
      }
      void setGroupConfigured(bool value)
      {
         _groupData.Configured = value;
      }

      //wrap around
      bool getGroupWrapAround() const
      {
         return _groupData.WrapAround;
      }
      void setGroupWrapAround(bool value)
      {
         _groupData.WrapAround = value;
      }

      //preserve current focus
      bool getGroupPreserveFocus() const
      {
         return _groupData.PreserveFocus;
      }
      void setGroupPreserveFocus(bool value)
      {
         _groupData.PreserveFocus = value;
      }

      //default order
      typedef Focusable2DBase::FocusOrderType DefaultFocusOrderType;
      DefaultFocusOrderType getGroupDefaultOrder() const
      {
         return static_cast<DefaultFocusOrderType>(_groupData.DefaultOrder);
      }
      void setGroupDefaultOrder(DefaultFocusOrderType value)
      {
         _groupData.DefaultOrder = static_cast<Focus::FocusLayerType>(value);
      }

      //layer
      Focus::FocusLayerType getGroupLayer() const
      {
         return _groupData.Layer;
      }
      void setGroupLayer(Focus::FocusLayerType value)
      {
         _groupData.Layer = value;
      }

      //focus marker animation widget
      typedef Candera::Node2D* FocusMarkerNodeType;

      FocusMarkerNodeType getFocusMarkerNode() const
      {
         return _focusMarkerNode;
      }
      void setFocusMarkerNode(FocusMarkerNodeType value)
      {
         _focusMarkerNode = value;
      }

      //anchor mode
      Focus::FAnchorMode::Enum getGroupAnchorMode() const
      {
         return _groupData.AnchorMode;
      }
      void setGroupAnchorMode(Focus::FAnchorMode::Enum value)
      {
         _groupData.AnchorMode = value;
      }

      //anchor id
      Focus::FAnchorId getGroupAnchorId() const
      {
         return _groupData.AnchorId;
      }
      void setGroupAnchorId(Focus::FAnchorId value)
      {
         _groupData.AnchorId = value;
      }
      bool IsPropertyVisible_FocusAnchorId() const
      {
         return getGroupConfigured() && (getGroupAnchorMode() != Focus::FAnchorMode::None);
      }

      bool IsPropertyVisible_FocusGroupCommon() const
      {
         return getGroupConfigured() && (getGroupAnchorMode() != Focus::FAnchorMode::Content);
      }

      //interaction with focus manager
      void registerGroupToFocusManager(Focus::FManager& focusManager, Focus::FSession& session, Focus::FWidgetConfig& handle);

      void updateFocusGroup(BaseWidget2D& widget, Focusable2DBase& focusable);

   private:
      Focus::FGroupData _groupData;
      FocusMarkerNodeType _focusMarkerNode;
};


#define CdaProperty_FocusGroupConfigured()\
      CdaBindableProperty(ConfigureFocusGroup, bool, getGroupConfigured, setGroupConfigured)\
      CdaDescription("Focus group is configured explicitelly or uses default configuration.")\
      CdaCategory("Focus")\
      CdaBindablePropertyEnd()

#define CdaProperty_FocusGroupAnchorMode()\
      CdaProperty(FocusAnchorMode, Focus::FAnchorMode::Enum, getGroupAnchorMode, setGroupAnchorMode)\
      CdaDescription("")\
      CdaCategory("Focus")\
      CdaVisibilityTest(getGroupConfigured)\
      CdaPropertyEnd()

#define CdaProperty_FocusGroupAnchorId()\
      CdaBindableProperty(FocusAnchorId, Focus::FAnchorId, getGroupAnchorId, setGroupAnchorId)\
      CdaDescription("")\
      CdaCategory("Focus")\
      CdaVisibilityTest(IsPropertyVisible_FocusAnchorId)\
      CdaBindablePropertyEnd()

#define CdaProperty_FocusGroupWrapAround()\
      CdaBindableProperty(FocusWrapAround, bool, getGroupWrapAround, setGroupWrapAround)\
      CdaDescription("Focus wraps around to first/last element.")\
      CdaCategory("Focus")\
      CdaVisibilityTest(IsPropertyVisible_FocusGroupCommon)\
      CdaBindablePropertyEnd()

#define CdaProperty_FocusGroupPreserveFocus()\
      CdaBindableProperty(PreserveFocus, bool, getGroupPreserveFocus, setGroupPreserveFocus)\
      CdaDescription("Current focused element is preserved when group becomes inactive.")\
      CdaCategory("Focus")\
      CdaVisibilityTest(IsPropertyVisible_FocusGroupCommon)\
      CdaBindablePropertyEnd()

#define CdaProperty_FocusGroupDefaultFocusOrder()\
      CdaBindableProperty(DefaultFocusOrder, DefaultFocusOrderType, getGroupDefaultOrder, setGroupDefaultOrder)\
      CdaDescription("Focus order of the element which should become focused when this group becomes active.")\
      CdaCategory("Focus")\
      CdaVisibilityTest(IsPropertyVisible_FocusGroupCommon)\
      CdaBindablePropertyEnd()

#define CdaProperty_FocusGroupLayer()\
      CdaBindableProperty(FocusLayer, Focus::FocusLayerType, getGroupLayer, setGroupLayer)\
      CdaDescription("Specifies to which focus layer this group belongs. There should be distinct layers for main surface screens, subspeller, popups to prevent focusing of elements in the main screen when a popup is displayed. Only elements in the groups with the highest layer will be focused.")\
      CdaCategory("Focus")\
      CdaVisibilityTest(IsPropertyVisible_FocusGroupCommon)\
      CdaBindablePropertyEnd()

#define CdaProperty_FocusGroupMarkerAnimationWidget()\
      CdaProperty(FocusMarkerNode, FocusMarkerNodeType, getFocusMarkerNode, setFocusMarkerNode)\
      CdaDescription("Focus marker which can be used by a focus animation manager to animate the changing of the focus inside this group (and its descendant groups). The default focus animation manager will create a translation animation which will move the marker from the old focus position to the new focus position. Please refer to the focus group section inside the widget user guide and also to the widget focus management concept document for more details.")\
      CdaCategory("Focus")\
      CdaVisibilityTest(IsPropertyVisible_FocusGroupCommon)\
      CdaPropertyEnd()


class FocusGroupWidget2D : public BaseWidget2D, public FocusGroupBase, public Focusable2DBase
   IMPLEMENTS_CLONEABLE_WIDGET
{
   public:

      FocusGroupWidget2D();
      virtual ~FocusGroupWidget2D();

      CGI_WIDGET_RTTI_DECLARATION(FocusGroupWidget2D, BaseWidget2D);

      virtual bool CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap);
      virtual bool OnMessage(const Courier::Message& msg);
      virtual void Update() override;

      // @WidgetBaseClass
      CdaWidget2DDef(FocusGroupWidget2D, BaseWidget2D)
      CdaCategory("Common")
      CdaProperties()

      CdaProperty_AppearanceId()

      CdaProperty_Focusable()
      CdaProperty_FocusParentNode()
      CdaProperty_FocusControllerSet()
      CdaProperty_FocusOrder()

      CdaProperty_FocusGroupConfigured()
      CdaProperty_FocusGroupAnchorMode()
      CdaProperty_FocusGroupAnchorId()
      CdaProperty_FocusGroupWrapAround()
      CdaProperty_FocusGroupPreserveFocus()
      CdaProperty_FocusGroupDefaultFocusOrder()
      CdaProperty_FocusGroupLayer()
      CdaProperty_FocusGroupMarkerAnimationWidget()

      CdaPropertiesEnd()
      CdaWidgetDefEnd()

   protected:
      virtual void RegisterToFocusManagerImpl(Focus::FManager& focusManager, Focus::FSession& session, Focus::FWidgetConfig& handle);
};


#endif
