/* ***************************************************************************************
* FILE:          EditFieldWidget2D.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  EditFieldWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef HTC_EditField_Widget_2D_H
#define HTC_EditField_Widget_2D_H

#include "Widgets/2D/Touchable2D.h"
#include "Widgets/2D/EditField/EditFieldHelpers.h"
#include "Candera/System/GlobalizationBase/Culture.h"
#include <Widgets/2D/EditField/generated/EditFieldWidget2DMessages.h>
#ifdef USE_TEXTEFFECT_FOR_EDITFIELD
#include "Widgets/2D/TextEffect/TextEffectWidget.h"
#else
#include "Widgets/2D/Text/TextWidget2D.h"
#endif
#include "Widgets/2D/Text/TextLayoutingInfoListener.h"

typedef ::hmibase::widget::text::TextLayoutingInfoListener TextLayoutingListener;
typedef ::hmibase::widget::text::TextLayoutingInformation TextLayoutingData;

//Enums
//Start - Candera Style Enum section
#define ENUM_DATA_TYPE \
    ENUM_DATA_TYPE_BEGIN(enWidgetMode) \
        ENUM_DATA_TYPE_ITEM_VALUE(FreeTextMode, 0) \
        ENUM_DATA_TYPE_ITEM_VALUE(MatchTextMode, 1) \
    ENUM_DATA_TYPE_END(enWidgetMode)
#include <Candera/System/MetaInfo/EnumDataType.h>
//End - Candera Style enum section
enum enEditFieldState
{
   //Enabled
   enEFNormal,
   enEFFocused,
   enEFSelected,
   //Disabled
   enEFDisabled
};


//Macros
//Blink interval in msec
#define BLINK_INTERVAL 1000

#define EF_MIN_INDEX                0
//Moved these defines to EditFieldHelpers.h
//#define EF_MIN_TEXT_LENGTH            0
//#define EF_MAX_TEXT_LENGTH            255    // Maximum number of UTF8 Characters
#define REPEAT_CONST                0
#define CONST_FOR_AVG_OF_WIDTHS        2
#define CONST_FOR_MARGIN            2

#define MAX_BYTE_FOR_A_CHARACTER    4    // UTF8 character
#define MAX_BYTES                    ((EF_MAX_TEXT_LENGTH) * (MAX_BYTE_FOR_A_CHARACTER))

#define EF_CURSOR_WIDTH                1

class EditFieldWidget2D : public Touchable2D, public TextLayoutingListener
{
   public:

      typedef Candera::MemoryManagement::SharedPointer<Candera::TextRendering::SharedStyle> StyleType;
      typedef Candera::MemoryManagement::SharedPointer<Candera::Globalization::Culture> CulturePtr;

      EditFieldWidget2D();

      virtual ~EditFieldWidget2D();

      CGI_WIDGET_RTTI_DECLARATION(EditFieldWidget2D, Touchable2D);

      virtual void Update();

      virtual void InitWidget();

      virtual bool OnMessage(const Courier::Message& msg);

      virtual bool OnTouch(const Candera::Camera2D& camera2D, const Candera::Vector2& point);

      virtual void OnFocus();

      virtual void OnLostFocus();

      Candera::enWidgetMode coenGetMode() const;
      void vSetMode(Candera::enWidgetMode enMode);

      Candera::UInt32 cou16GetBgWidth() const;
      void vSetBgWidth(Candera::UInt32 u8BgWidth);

      Candera::UInt32 cou8GetBgHeight() const;
      void vSetBgHeight(Candera::UInt32 u8BgHeight);

      Candera::UInt32 cou8GetTextAreaMarginX() const;
      void vSetTextAreaMarginX(Candera::UInt32 u8TextAreaMarginX);

      Candera::UInt32 cou8GetTextAreaMarginY() const;
      void vSetTextAreaMarginY(Candera::UInt32 u8TextAreaMarginY);

      Candera::UInt32 cou8GetMaxTextLength() const;
      void vSetMaxTextLength(Candera::UInt32 u8MaxTextLength);

      Candera::String cosGetMaskChar() const;
      void vSetMaskChar(Candera::String sMaskChar);

      bool cobRestrictShiftOperations() const;
      void vSetRestrictShiftOperations(bool setRestriction);

      Candera::String cosGetInstructionText() const;
      void vSetInstructionText(Candera::String sInstructionText);

      Candera::Node2D* getBackgroundNode() const;
      void setBackgroundNode(Candera::Node2D*);

      Candera::Node2D* getCursorNode() const;
      void setCursorNode(Candera::Node2D*);

      Candera::Node2D* copGetEntryTextNode() const;
      void vSetEntryTextNode(Candera::Node2D*);

      Candera::Node2D* copGetMatchTextNode() const;
      void vSetMatchTextNode(Candera::Node2D*);

      // Kanji Conversion Requirement
      Candera::Node2D* copGetHighlightTextNode() const;
      void vSetHighlightTextNode(Candera::Node2D*);

      Candera::Node2D* copGetEndTextNode() const;
      void vSetEndTextNode(Candera::Node2D*);

      Candera::UInt32 cou8GetHighlightStartIndex() const;
      void vSetHighlightStartIndex(Candera::UInt32 highlightStartIndex);

      Candera::UInt32 cou8GetHighlightCount() const;
      void vSetHighlightCount(Candera::UInt32 u8HighlightCount);

      Candera::Color cooGetHighlightEnabledTextColor() const;
      void vSetHighlightEnabledTextColor(Candera::Color oHighlightEnabledTextColor);

      Candera::Color cooGetHighlightDisabledTextColor() const;
      void vSetHighlightDisabledTextColor(Candera::Color oHighlightDisabledTextColor);
      //End :: Kanji Conversion Requirement

      //Default text color enabled/disabled
      Candera::Color cooGetDefaultEnabledTextColor() const;
      void vSetDefaultEnabledTextColor(Candera::Color oDefaultEnabledTextColor);

      Candera::Color cooGetDefaultDisabledTextColor() const;
      void vSetDefaultDisabledTextColor(Candera::Color oDefaultDisabledTextColor);

      //Entry Text
      Candera::TextRendering::Font cooGetEntryTextFont() const;
      void vSetEntryTextFont(const Candera::TextRendering::Font& oEntryTextFont);

      Candera::Color cooGetEnabledEntryTextColor() const;
      void vSetEnabledEntryTextColor(Candera::Color oEnabledEntryTextColor);

      Candera::Color cooGetDisabledEntryTextColor() const;
      void vSetDisabledEntryTextColor(Candera::Color oDisabledEntryTextColor);

      //Match Text
      Candera::TextRendering::Font cooGetMatchTextFont() const;
      void vSetMatchTextFont(const Candera::TextRendering::Font& oMatchTextFont);

      Candera::Color cooGetEnabledMatchTextColor() const;
      void vSetEnabledMatchTextColor(Candera::Color oEnabledMatchTextColor);

      Candera::Color cooGetDisabledMatchTextColor() const;
      void vSetDisabledMatchTextColor(Candera::Color oDisabledMatchTextColor);

      //Instruction Text
      Candera::TextRendering::Font cooGetInstructionTextFont() const;
      void vSetInstructionTextFont(const Candera::TextRendering::Font& oInstructionTextFont);

      Candera::Color cooGetEnabledInstructionTextColor() const;
      void vSetEnabledInstructionTextColor(Candera::Color oEnabledInstructionTextColor);

      Candera::Color cooGetDisabledInstructionTextColor() const;
      void vSetDisabledInstructionTextColor(Candera::Color oDisabledInstructionTextColor);

      Candera::String cosGetCursorAnimation() const;
      void vSetCursorAnimation(Candera::String sCursorAnimation);

      Candera::UInt32 cou32GetBlinkInterval() const;
      void vSetBlinkInterval(Candera::UInt32 u32BlinkInterval);

      Candera::String cosGetEditFieldInfo() const;
      void vSetEditFieldInfo(Candera::String sEditFieldInfo);

      Candera::String cosGetEntryTextReceived() const;
      void vSetEntryTextReceived(Candera::String sEntryTextReceived);

      Candera::UInt32 cou16GetNewCursorPosition() const;
      void vSetNewCursorPosition(Candera::UInt32 u16NewCursorPos);

      //a parameter with value true signifies ParentView is rendering, false signifies not rendering
      virtual void OnParentViewRenderingEnabled(bool enable);
      Candera::String cosGetSuggestionText() const;
      void vSetSuggestionText(Candera::String sSuggestionText);

      Candera::String cosGetKeyReceived() const;
      void vSetKeyReceived(Candera::String sKey);

      Candera::String cosGetTruncationText() const;
      void vSetTruncationText(Candera::String sTruncationText);

      // overridden from Widget
      virtual void OnChanged(FeatStd::UInt32 propertyId);
      //To perform Visibility Test
      bool cobMatchMode() const;

      void SetTextStyle(const StyleType& style);
      const StyleType& GetTextStyle() const
      {
         return m_TextStyle;
      }

      void vSetMultiLineLayouting(const bool multilineLayouting);
      bool cobGetMultiLineLayouting() const;

      void vSetLineSpacing(const Candera::UInt32 lineSpacing);
      Candera::UInt32 couGetLineSpacing() const;

      void vSetMaximumNumberOfLines(const Candera::UInt32 maximumNumberOfLines);
      Candera::UInt32 couGetMaximumNumberOfLines() const;

      void OnLayoutingInfoChanged(TextLayoutingData& data);

      virtual void CultureChanged();
   private:
      CulturePtr m_cultureCurrent;

      Candera::enWidgetMode m_enWidgetMode;
      enEditFieldState m_enEFState;

      //Background width/height and TextArea offsets
      Candera::UInt32 m_u16BgWidth;
      Candera::UInt32 m_u8BgHeight;
      Candera::UInt32 m_u8TextAreaMarginX;
      Candera::UInt32 m_u8TextAreaMarginY;

      /*Node pointers for Background node,
      Cursor node, EntryText node, MatchText node*/

      Candera::Node2D* m_pBackgroundNode;
      Candera::Node2D* m_pCursorNode;
      Candera::Node2D* m_pEntryTextNode;
      Candera::Node2D* m_pMatchTextNode;

      //Kanji Conversion Requirement
      Candera::Node2D* m_pHighlightTextNode;
      Candera::Node2D* m_pEndTextNode;

      //Width/Height for TextArea were text will be rendered
      Candera::UInt32 m_u16TextAreaWidth;
      Candera::UInt32 m_u8TextAreaHeight;

      Candera::Vector2 m_v2EntryTextSize;
      Candera::Vector2 m_v2MatchTextSize;

      //Current and Maximum acceptable length of EditField String
      Candera::UInt32 m_u8CurStrLength;
      Candera::UInt32 m_u8MaxTextLength;

      //1st Character of m_sMaskChar is used for masking if specified
      Candera::String m_sMaskChar;
      //String used to display hint text
      Candera::String m_sInstructionText;

      //Members for Text styles- EntryText, MatchText, InstructionText
      Candera::Color m_oDefaultEnabledTextColor;
      Candera::Color m_oDefaultDisabledTextColor;

      Candera::Color m_oHighlightEnabledTextColor;
      Candera::Color m_oHighlightDisabledTextColor;

      Candera::TextRendering::Font m_oEntryTextFont;
      Candera::Color m_oEnabledEntryTextColor;
      Candera::Color m_oDisabledEntryTextColor;

      Candera::TextRendering::Font m_oMatchTextFont;
      Candera::Color m_oEnabledMatchTextColor;
      Candera::Color m_oDisabledMatchTextColor;

      Candera::TextRendering::Font m_oInstructionTextFont;
      Candera::Color m_oEnabledInstructionTextColor;
      Candera::Color m_oDisabledInstructionTextColor;

      //Candera path for cursor blinking animation
      Candera::String m_sCursorAnimation;
      //Animation property for Cursor Animation
      Courier::AnimationProperties m_oCursorAnimProp;
      //Time interval in msec for cursor blinking
      Candera::UInt32 m_u32BlinkInterval;
      Candera::UInt32 m_u16NewCursorPosition;

      //String to send binding property update to other components
      Candera::String m_sEditFieldInfo;
      //User entered String received from application (MatchTextMode)
      Candera::String m_sEntryTextReceived;
      //Suggestion String received from application (MatchTextMode)
      Candera::String m_sSuggestionText;
      //Masked form of EditField string,same as editfieldinfo string if no mask char
      Candera::String m_sMaskedText;
      //The visible text that is displayed in the edit field
      Candera::String m_sDisplayedText;

      bool m_bShiftRestricted;
      bool m_mixedCharHandlingRequired;

      // Bitmap nodes start
      typedef Candera::MemoryManagement::SharedPointer<Candera::Image2D> BitmapType;

      //! [DataBinding_BitmapPropertyGetterSetter]
      BitmapType _bitmapNormalBg;
      void SetNormalBgBitmap(const BitmapType& bitmap);
      const BitmapType& GetNormalBgBitmap() const
      {
         return _bitmapNormalBg;
      }

      BitmapType _bitmapSelectedBg;
      void SetSelectedBgBitmap(const BitmapType& bitmap);
      const BitmapType& GetSelectedBgBitmap() const
      {
         return _bitmapSelectedBg;
      }

      BitmapType _bitmapDisabledBg;
      void SetDisabledBgBitmap(const BitmapType& bitmap);
      const BitmapType& GetDisabledBgBitmap() const
      {
         return _bitmapDisabledBg;
      }

      BitmapType _bitmapCursorBg;
      void SetCursorBgBitmap(const BitmapType& bitmap);
      const BitmapType& GetCursorBgBitmap() const
      {
         return _bitmapCursorBg;
      }
      //! [DataBinding_BitmapPropertyGetterSetter]

      // Bitmap nodes end

#ifdef USE_TEXTEFFECT_FOR_EDITFIELD
      TextEffectWidget m_oEntryTextWidget;
      TextEffectWidget m_oMatchTextWidget;
#else
      TextWidget2D m_oEntryTextWidget;
      TextWidget2D m_oMatchTextWidget;

      //Kanji Conversion Req
      TextWidget2D m_oHighlightTextWidget;
      TextWidget2D m_oEndTextWidget;
      //End Kanji Conversion Req
#endif
      //m_u8CursorIndex - position of cursor or number of chars to the left of EditField string
      CursorPosition m_oCursorPosition;
      //Candera::UInt32 m_u8CursorIndex;
      //number of chars to the left of the visible EditField string(m_sDisplayedText)
      //Candera::UInt32 m_u8DisplayedCursorIndex;
      //m_u8VisibleStartIndex - index of first visible char in the EditField string
      Candera::UInt32 m_u8VisibleStartIndex;
      //m_u8VisibleCharCount - count of visible char in the EditField string
      Candera::UInt32 m_u8VisibleCharCount;

      //Kanji Conversion Req
      Candera::UInt32 m_uHighlightStartIndex;
      Candera::UInt32 m_uHighlightCount;
      Candera::UInt32 m_uHighlightVisibleStartIndex;
      Candera::UInt32 m_uHighlightVisibleCharCount;
      Candera::UInt32 m_uLeftVisibleStartIndex;
      Candera::UInt32 m_uLeftVisibleCharCount;
      Candera::UInt32 m_uEndVisibleStartIndex;
      Candera::UInt32 m_uEndVisibleCharCount;

      //End Kanji Conversion Req

      //Members for GetWidth() interface of TextEffectWidget
      Candera::TextRendering::LayoutingOptions oLayoutingOption;
      Candera::TextRendering::ShapingOptions oShapingOption;
      Candera::TextRendering::MeasuringOptions oMeasuringOption;

      Candera::String m_sKeyReceived;

      //Vector to store position of EntryTextNode
      Candera::Vector2 m_vEntryTextNodePos;

      //Kanji Req
      Candera::Vector2 m_vHighlightTextNodePos;
      Candera::Vector2 m_vEndTextNodePos;
      bool m_bHighlightDataChanged;

      //Vector to store default position of CursorNode when string is empty
      Candera::Vector2 m_vCursorDefaultPos;

      //boolean to determine if Widget is enabled
      bool m_bEditFieldEnabled;
      //boolean to determine if Widget state has changed
      bool m_bStateChanged;

      bool m_bUpdateBgImgWidget;
      bool m_bUpdateEntryTextWidget;
      bool m_bUpdateMatchTextWidget;
      bool m_bUpdateAdditionalTextWidget;

      bool m_bPropertyChanged;
      bool m_bInInstructionMode;
      bool m_bUpdateInstructionText;
      bool m_bUpdateSuggestionText;

      bool m_bInvalidateRequired;
      Candera::TChar m_pEditFieldText[(EF_MAX_TEXT_LENGTH * MAX_BYTE_FOR_A_CHARACTER) + 1];
      //Kanji Req :: TODO check allocation :how Much
      Candera::TChar m_pLeftText[(EF_MAX_TEXT_LENGTH * MAX_BYTE_FOR_A_CHARACTER) + 1];
      Candera::TChar m_pHighlightText[(EF_MAX_TEXT_LENGTH * MAX_BYTE_FOR_A_CHARACTER) + 1];
      Candera::TChar m_pEndText[(EF_MAX_TEXT_LENGTH * MAX_BYTE_FOR_A_CHARACTER) + 1];
      Candera::String m_sDisplayedLeftText;
      Candera::String m_sDisplayedHighightText;
      Candera::String m_sDisplayedEndText;

      Candera::String m_sElipsisText;
      StyleType m_TextStyle;
      bool m_bMultilineLayouting;
      Candera::UInt32 m_uLineSpacing;
      Candera::UInt32 m_uMaxNumberOfLines;
      Candera::UInt32 m_uNumberOfLines;  // To determine total number of line, same to m_uCurrentLine if chars are added in last
      Candera::UInt32 m_uCurrentLine;    // To determine line where insertion of characters is done
      Candera::UInt32 m_uLineHeightWithSpacing;
      std::vector<Candera::UInt32> m_charCountEachLine;
      std::vector<Candera::Int32> m_rectTopEachLine;
      Candera::UInt32 m_u16TextRectHeight;
      Candera::UInt32 m_u32TruncationTextDisplayedWidth;

      void vUpdateCursorPos();
      void vUpdateVisibleText();
      void vCreateMaskedText();

      //FreeTextMode Specific functions
      void vInsertChar();
      void vDeleteChar();
      void vDeleteAll();
      void vShortHoldDelete();
      void vUpdateIndexOnInsert();
      void vUpdateIndexOnDelete();
      void vShiftCursorLeft();
      void vShiftCursorRight();
      // MultiLine in Editfield
      void vUpdateIndexOnInsertMultiLine();
      void vUpdateVisibleTextMultiLine();
      Candera::UInt32 cou16ComputeHeight(Candera::String text, Candera::UInt32 u8Start, Candera::UInt32 u8nChars);
      void vUpdateIndexOnDeleteMultiLine();
      void vProcessOnTouchForMultiLine(Candera::Vector2 point);
      void vProcessCursorPosition();

      //MatchTextMode Specific functions
      void vUpdateEntryText();
      void vUpdateMatchText();

      //Kanji Req
      void vResetHighlightAndEndText();
      void vUpdateHighlightAndEndText();
      void vProcessEditFieldHightlightReqMsg(const ::hmibase::widget::editfield::EditFieldHightlightReqMsg& msg);
      void vProcessVisibleTextAndCharCount();
      void vUpdateAllText();
      void vAddElipsisToText(const Candera::TChar* text, Candera::String& elipsesAdded);
      void vUpdateAllTextOnDeleteChar();
      bool bPostEditFieldUpdMsg();

      Candera::UInt32 cou16ComputeWidth(Candera::String text, Candera::UInt32 u8Start, Candera::UInt32 u8nChars);

      Candera::UInt32 uCopyUTF8Data(Candera::Char* dstString, Candera::UInt32 dstBufferLength, Candera::UInt32 dstStringOffset, Candera::String srcString, Candera::UInt32 srcStringOffset, Candera::UInt32 CharCountToCopy) const;
      Candera::UInt32 u8UTFLength(Candera::String sText) const;

      void vConfgEntryTxtWidget();
      void vConfgMatchTxtWidget();
      void vProcessTextOnEditFieldAction(::hmibase::widget::editfield::enEditFieldTextAction::Enum action, Candera::String msgText);

      void setBgImage(const BitmapType& bitmap);

      void vUpdateCursorImage();
      void vUpdateTextNodePosDim();
      void vUpdateStateProperties();

      Candera::UInt32 u8CalculateUTF8Bytes(Candera::UInt32 uStartIndex, Candera::UInt32 nCharCount, Candera::String string) const;

      void vAddElipsisText(const Candera::TChar* text);
      FeatStd::UInt32 cou8GetTruncationTextDisplayedWidth();

      /** determine if the given character is an Arabic character
      @param chr the character to examine
      @return true if the character is in the Arabic code block, false otherwise */
      //TODO: the datatype 'Ucs2Char' should be used instead of 'wchar_t'
      bool IsArabicChar(wchar_t chr) const;

      void onLanguageChange();
      Candera::UInt32 calculateCursorOffset(bool, Candera::UInt32 currCharPos_Actual);
      Candera::UInt32 calculateCharCount(Candera::TChar*, Candera::TChar*, bool);
      Candera::UInt32 calculateOffeSetOnTouch(Candera::UInt32, bool isRightToLeft);

      // Candera Widget Properties
      CdaWidget2DDef(EditFieldWidget2D, Touchable2D)
      CdaDescription("Widget used to display text input given by user dynamically")
      CdaCategory("Input")

      CdaProperties()
      CdaProperty(WidgetTextMode, Candera::enWidgetMode, coenGetMode, vSetMode)
      CdaDescription("Operation Mode for the EditFieldWidget")
      CdaCategory("Private Property")
      CdaPropertyEnd()

      CdaProperty(BackgroundWidth, Candera::UInt32, cou16GetBgWidth, vSetBgWidth)
      CdaDescription("Width of EditField Background")
      CdaCategory("Private Property")
      CdaPropertyEnd()

      CdaProperty(BackgroundHeight, Candera::UInt32, cou8GetBgHeight, vSetBgHeight)
      CdaDescription("Height of EditField Background")
      CdaCategory("Private Property")
      CdaPropertyEnd()

      CdaProperty(TextAreaMarginX, Candera::UInt32, cou8GetTextAreaMarginX, vSetTextAreaMarginX)
      CdaDescription("X-Offset from Background Image  for TextArea")
      CdaCategory("Private Property")
      CdaPropertyEnd()

      CdaProperty(TextAreaMarginY, Candera::UInt32, cou8GetTextAreaMarginY, vSetTextAreaMarginY)
      CdaDescription("Y-Offset from Background Image for TextArea")
      CdaCategory("Private Property")
      CdaPropertyEnd()

      CdaProperty(MaxTextLength, Candera::UInt32, cou8GetMaxTextLength, vSetMaxTextLength)
      CdaDescription("Maximum acceptable length of EditField text")
      CdaCategory("Private Property")
      CdaPropertyEnd()

      CdaProperty(MaskChar, Candera::String, cosGetMaskChar, vSetMaskChar)
      CdaDescription("Mask char if masking is required, else leave it blank")
      CdaCategory("Private Property")
      CdaPropertyEnd()

      CdaProperty(RestrictShiftOperations, bool, cobRestrictShiftOperations, vSetRestrictShiftOperations)
      CdaDescription("Restricts the movement of cursor in highlight area only in EditField text")
      CdaCategory("Private Property")
      CdaPropertyEnd()

      CdaProperty(BackgroundNode, Candera::Node2D*, getBackgroundNode, setBackgroundNode)
      CdaDescription("Background image node")
      CdaCategory("Nodes")
      CdaPropertyEnd()

      CdaProperty(CursorNode, Candera::Node2D*, getCursorNode, setCursorNode)
      CdaDescription("Cursor image node")
      CdaCategory("Nodes")
      CdaPropertyEnd()

      CdaProperty(EntryTextNode, Candera::Node2D*, copGetEntryTextNode, vSetEntryTextNode)
      CdaDescription("EntryText node")
      CdaCategory("Nodes")
      CdaPropertyEnd()

      CdaProperty(MatchTextNode, Candera::Node2D*, copGetMatchTextNode, vSetMatchTextNode)
      CdaDescription("MatchText node")
      CdaCategory("Nodes")
      CdaPropertyEnd()

      //Kanji Converion Properties
      CdaProperty(HighlightTextNode, Candera::Node2D*, copGetHighlightTextNode, vSetHighlightTextNode)
      CdaDescription("HighlightText node")
      CdaCategory("Nodes")
      CdaPropertyEnd()

      CdaProperty(EndTextNode, Candera::Node2D*, copGetEndTextNode, vSetEndTextNode)
      CdaDescription("EndText node")
      CdaCategory("Nodes")
      CdaPropertyEnd()

      CdaBindableProperty(HighlightStartIndex, Candera::UInt32, cou8GetHighlightStartIndex, vSetHighlightStartIndex)
      CdaDescription("Bindable property to receive Highlight Start Index if Configured in free Mode")
      CdaCategory("BindableProperty")
      CdaBindablePropertyEnd()

      CdaBindableProperty(HighlightCount, Candera::UInt32, cou8GetHighlightCount, vSetHighlightCount)
      CdaDescription("Bindable property to receive Highlight Count if Configured in free Mode")
      CdaCategory("BindableProperty")
      CdaBindablePropertyEnd()
      //End kanji Conversion Properties

      // new bitmap properties start

      CdaProperty(NormalBgBitmap, BitmapType, GetNormalBgBitmap, SetNormalBgBitmap)
      CdaDescription("Enabled Background; Bitmap to be set on the BitmapBrushEffect inside the RenderNode")
      CdaCategory("Image Properties")
      CdaPropertyEnd()

      CdaBindableProperty(SelectedBgBitmap, BitmapType, GetSelectedBgBitmap, SetSelectedBgBitmap)
      CdaDescription("Selected Background; Bitmap to be set on the BitmapBrushEffect inside the RenderNode")
      CdaCategory("Image Properties")
      CdaPropertyEnd()

      CdaBindableProperty(DisabledBgBitmap, BitmapType, GetDisabledBgBitmap, SetDisabledBgBitmap)
      CdaDescription("Disabled Background; Bitmap to be set on the BitmapBrushEffect inside the RenderNode")
      CdaCategory("Image Properties")
      CdaPropertyEnd()

      CdaBindableProperty(CursorBgBitmap, BitmapType, GetCursorBgBitmap, SetCursorBgBitmap)
      CdaDescription("Cursor Bitmap; Bitmap to be set on the BitmapBrushEffect inside the RenderNode")
      CdaCategory("Image Properties")
      CdaPropertyEnd()

      // new bitmap properties end

      CdaProperty(DefaultEnabledTextColor, Candera::Color, cooGetDefaultEnabledTextColor, vSetDefaultEnabledTextColor)
      CdaDescription("Default Color for Text in Enabled state")
      CdaCategory("Text Properties")
      CdaPropertyEnd()

      CdaProperty(DefaultDisabledTextColor, Candera::Color, cooGetDefaultDisabledTextColor, vSetDefaultDisabledTextColor)
      CdaDescription("Default Color for Text in Disabled state")
      CdaCategory("Text Properties")
      CdaPropertyEnd()

      CdaProperty(HighlightEnabledTextColor, Candera::Color, cooGetHighlightEnabledTextColor, vSetHighlightEnabledTextColor)
      CdaDescription("Default Color for Highlight Text in Enabled state")
      CdaCategory("Text Properties")
      CdaPropertyEnd()

      CdaProperty(HighlightDisabledTextColor, Candera::Color, cooGetHighlightDisabledTextColor, vSetHighlightDisabledTextColor)
      CdaDescription("Default Color for Highlight Text in Disabled state")
      CdaCategory("Text Properties")
      CdaPropertyEnd()

      CdaProperty(TextStyle, StyleType, GetTextStyle, SetTextStyle)
      CdaDescription("The TextStyle used for texts like EntryText, matchText etc.Individual fonts wiil be used if TextStyle is empty.")
      CdaCategory("Text Properties")
      CdaPropertyEnd()

      CdaProperty(EntryTextFont, Candera::TextRendering::Font, cooGetEntryTextFont, vSetEntryTextFont)
      CdaDescription("Font for EntryText")
      CdaCategory("Text Properties")
      CdaPropertyEnd()

      CdaProperty(EnabledEntryTextColor, Candera::Color, cooGetEnabledEntryTextColor, vSetEnabledEntryTextColor)
      CdaDescription("Color for EntryText in enabled state")
      CdaCategory("Text Properties")
      CdaPropertyEnd()

      CdaProperty(DisabledEntryTextColor, Candera::Color, cooGetDisabledEntryTextColor, vSetDisabledEntryTextColor)
      CdaDescription("Color for EntryText in disabled state")
      CdaCategory("Text Properties")
      CdaPropertyEnd()

      CdaProperty(MatchTextFont, Candera::TextRendering::Font, cooGetMatchTextFont, vSetMatchTextFont)
      CdaDescription("Font for MatchText")
      CdaCategory("Text Properties")
      CdaVisibilityTest(cobMatchMode)
      CdaPropertyEnd()

      CdaProperty(EnabledMatchTextColor, Candera::Color, cooGetEnabledMatchTextColor, vSetEnabledMatchTextColor)
      CdaDescription("Color for MatchText in enabled state")
      CdaCategory("Text Properties")
      CdaVisibilityTest(cobMatchMode)
      CdaPropertyEnd()

      CdaProperty(DisabledMatchTextColor, Candera::Color, cooGetDisabledMatchTextColor, vSetDisabledMatchTextColor)
      CdaDescription("Color for MatchText  in disabled state")
      CdaCategory("Text Properties")
      CdaVisibilityTest(cobMatchMode)
      CdaPropertyEnd()

      CdaProperty(InstructionTextFont, Candera::TextRendering::Font, cooGetInstructionTextFont, vSetInstructionTextFont)
      CdaDescription("Font for InstructionText")
      CdaCategory("Text Properties")
      CdaPropertyEnd()

      CdaProperty(EnabledInstructionTextColor, Candera::Color, cooGetEnabledInstructionTextColor, vSetEnabledInstructionTextColor)
      CdaDescription("Color for InstructionText in enabled state")
      CdaCategory("Text Properties")
      CdaPropertyEnd()

      CdaProperty(DisabledInstructionTextColor, Candera::Color, cooGetDisabledInstructionTextColor, vSetDisabledInstructionTextColor)
      CdaDescription("Color for InstructionText in Disabled state")
      CdaCategory("Text Properties")
      CdaPropertyEnd()

      CdaProperty(MultiLineLayouting, bool, cobGetMultiLineLayouting, vSetMultiLineLayouting)
      CdaDescription("MultiLine Layouting for the Editfield to take input in multiple lines.")
      CdaCategory("Text Properties")
      CdaPropertyEnd()

      CdaProperty(MaximumNumberOfLines, Candera::UInt32, couGetMaximumNumberOfLines, vSetMaximumNumberOfLines)
      CdaDescription("MaximumNumberOfLines for the Editfield to take input in multiple lines upto value set in MaximumNumberOfLines")
      CdaCategory("Text Properties")
      CdaVisibilityTest(cobGetMultiLineLayouting)
      CdaPropertyEnd()

      CdaProperty(LineSpacing, Candera::UInt32, couGetLineSpacing, vSetLineSpacing)
      CdaDescription("LineSpacing in MultiLine Text for the Editfield")
      CdaCategory("Text Properties")
      CdaVisibilityTest(cobGetMultiLineLayouting)
      CdaPropertyEnd()

      CdaProperty(CursorAnimationPath, Candera::String, cosGetCursorAnimation, vSetCursorAnimation)
      CdaDescription("Specify the Candera path of the cursor animation to be used for blinking of cursor")
      CdaCategory("Cursor Animation Property")
      CdaPropertyEnd()

      CdaProperty(BlinkInterval, Candera::UInt32, cou32GetBlinkInterval, vSetBlinkInterval)
      CdaDescription("Specify the interval in msec for blinking of cursor")
      CdaCategory("Cursor Animation Property")
      CdaPropertyEnd()

      CdaBindableProperty(EditFieldInfo, Candera::String, cosGetEditFieldInfo, vSetEditFieldInfo)
      CdaDescription("Bindable property to send the EditField Text to other components")
      CdaCategory("BindableProperty")
      CdaBindablePropertyEnd()

      CdaBindableProperty(EntryTextReceived, Candera::String, cosGetEntryTextReceived, vSetEntryTextReceived)
      CdaDescription("Bindable property to receive Entry Text string if Configured in Match Mode")
      CdaCategory("BindableProperty")
      CdaVisibilityTest(cobMatchMode)
      CdaBindablePropertyEnd()

      CdaBindableProperty(NewCursorPosition, Candera::UInt32, cou16GetNewCursorPosition, vSetNewCursorPosition)
      CdaDescription("Bindable property to receive NewCursorPosition if Configured in free Mode")
      CdaCategory("BindableProperty")
      CdaBindablePropertyEnd()
      CdaBindableProperty(SuggestionText, Candera::String, cosGetSuggestionText, vSetSuggestionText)
      CdaDescription("Bindable property for suggestion text to be used if Configured in Match Mode")
      CdaCategory("BindableProperty")
      CdaVisibilityTest(cobMatchMode)
      CdaBindablePropertyEnd()

      CdaBindableProperty(KeyReceived, Candera::String, cosGetKeyReceived, vSetKeyReceived)
      CdaDescription("Bindable property to receive key press inputs for the EditField text")
      CdaCategory("BindableProperty")
      CdaBindablePropertyEnd()

      CdaBindableProperty(InstructionText, Candera::String, cosGetInstructionText, vSetInstructionText)
      CdaDescription("Bindable property to set the instruction text that will be displayed.")
      CdaCategory("BindableProperty")
      CdaBindablePropertyEnd()

      CdaBindableProperty(TruncationText, Candera::String, cosGetTruncationText, vSetTruncationText)
      CdaDescription("Bindable property to set the truncation text that will be displayed.")
      CdaCategory("Truncation")
      CdaBindablePropertyEnd()

      CdaPropertiesEnd()

      CdaWidgetDefEnd()
};


#endif // HTC_EditField_WIDGET_2D_H
