/* ***************************************************************************************
* FILE:          EditFieldWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  EditFieldWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "CanderaPlatform/Device/Common/Effects/TextBrushBlend.h"
#include "CanderaPlatform/Device/Common/Effects/BitmapBrushBlend.h"
#include "Widgets/2D/BaseWidget2D.h"
#include "EditFieldWidget2D.h"
#include "../Speller/SpellerConstants.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_EDITFIELD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/EditFieldWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(EditFieldWidget2D);

using namespace Candera;
using namespace Courier;


/******************************************************************************
*     Description : Constructor
******************************************************************************/
EditFieldWidget2D::EditFieldWidget2D():
   m_enWidgetMode(FreeTextMode),
   m_enEFState(enEFNormal),

   m_u16BgWidth(0),
   m_u8BgHeight(0),
   m_u8TextAreaMarginX(0),
   m_u8TextAreaMarginY(0),

   m_pBackgroundNode(NULL),
   m_pCursorNode(NULL),
   m_pEntryTextNode(NULL),
   m_pMatchTextNode(NULL),

   m_u16TextAreaWidth(0),
   m_u8TextAreaHeight(0),
   m_v2EntryTextSize(),
   m_v2MatchTextSize(),

   m_u8CurStrLength(EF_MIN_TEXT_LENGTH),
   m_u8MaxTextLength(EF_MAX_TEXT_LENGTH),

   m_sMaskChar(),
   m_sInstructionText(),

   m_oDefaultEnabledTextColor(),
   m_oDefaultDisabledTextColor(),

   m_oEntryTextFont(),
   m_oEnabledEntryTextColor(),
   m_oDisabledEntryTextColor(),

   m_oMatchTextFont(),
   m_oEnabledMatchTextColor(),
   m_oDisabledMatchTextColor(),

   m_oInstructionTextFont(),
   m_oEnabledInstructionTextColor(),
   m_oDisabledInstructionTextColor(),

   m_sCursorAnimation(),
   m_oCursorAnimProp(),
   m_u32BlinkInterval(BLINK_INTERVAL),
   m_u16NewCursorPosition(0),

   m_sEditFieldInfo(),
   m_sEntryTextReceived(),
   m_sSuggestionText(),
   m_sMaskedText(),

   m_sDisplayedText(),

   m_oEntryTextWidget(),
   m_oMatchTextWidget(),

   m_oCursorPosition(EF_MIN_INDEX),

   m_u8VisibleStartIndex(EF_MIN_INDEX),
   m_u8VisibleCharCount(EF_MIN_TEXT_LENGTH),

   oLayoutingOption(),
   oShapingOption(),
#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 2)))
   oMeasuringOption(),
#else
   oMeasuringOption(TextRendering::MeasuringOptions::DefaultFinalAdvance,
                    TextRendering::MeasuringOptions::DefaultTransversalSize),
#endif
   m_sKeyReceived(),
   m_vEntryTextNodePos(),
   m_vCursorDefaultPos(),

   m_bEditFieldEnabled(true),
   m_bStateChanged(false),

   m_bUpdateBgImgWidget(false),
   m_bUpdateEntryTextWidget(false),
   m_bUpdateMatchTextWidget(false),

   m_bPropertyChanged(false),
   m_bInInstructionMode(true),
   m_bUpdateInstructionText(false),
   m_bUpdateSuggestionText(false),
   m_bInvalidateRequired(false),
   m_TextStyle(NULL),
   m_uHighlightStartIndex(EF_MIN_INDEX),
   m_uHighlightCount(EF_MIN_TEXT_LENGTH),
   m_bUpdateAdditionalTextWidget(false),
   m_bHighlightDataChanged(false),
   m_uLeftVisibleStartIndex(EF_MIN_INDEX),
   m_uLeftVisibleCharCount(EF_MIN_TEXT_LENGTH),
   m_uHighlightVisibleStartIndex(EF_MIN_INDEX),
   m_uHighlightVisibleCharCount(EF_MIN_TEXT_LENGTH),
   m_uEndVisibleStartIndex(EF_MIN_INDEX),
   m_uEndVisibleCharCount(EF_MIN_TEXT_LENGTH),
   m_pEndTextNode(NULL),
   m_pHighlightTextNode(NULL),
   m_oHighlightEnabledTextColor(),
   m_oHighlightDisabledTextColor(),
   m_oHighlightTextWidget(),
   m_oEndTextWidget(),
   m_vHighlightTextNodePos(),
   m_vEndTextNodePos(),
   m_sDisplayedLeftText(),
   m_sDisplayedHighightText(),
   m_sDisplayedEndText(),
   m_bMultilineLayouting(false),
   m_uLineSpacing(0),
   m_uNumberOfLines(1),
   m_uCurrentLine(1),
   m_u16TextRectHeight(0),
   m_uLineHeightWithSpacing(0),
   m_uMaxNumberOfLines(1),
   m_rectTopEachLine(),
   m_charCountEachLine(),
   m_bShiftRestricted(false),
   m_u32TruncationTextDisplayedWidth(0),
   m_mixedCharHandlingRequired(false),
   m_cultureCurrent()

//   m_sElipsisText()
{
   SetTouchable(true);
   memset(m_pEditFieldText, 0, (EF_MAX_TEXT_LENGTH * MAX_BYTE_FOR_A_CHARACTER) + 1);
   memset(m_pLeftText, 0, (EF_MAX_TEXT_LENGTH * MAX_BYTE_FOR_A_CHARACTER) + 1);
   memset(m_pHighlightText, 0, (EF_MAX_TEXT_LENGTH * MAX_BYTE_FOR_A_CHARACTER) + 1);
   memset(m_pEndText, 0, (EF_MAX_TEXT_LENGTH * MAX_BYTE_FOR_A_CHARACTER) + 1);
   m_oEntryTextWidget.AddListener(this);
}


/******************************************************************************
*     Description : Destructor
******************************************************************************/
EditFieldWidget2D::~EditFieldWidget2D()
{
   m_pBackgroundNode = NULL;
   m_pCursorNode = NULL;
   m_pEntryTextNode = NULL;
   m_pMatchTextNode = NULL;
   m_pHighlightTextNode = NULL;
   m_pEndTextNode = NULL;
}


/****************************************************************************
*     Description : Is called during every render cycle. (Overridden)
****************************************************************************/
void EditFieldWidget2D::Update()
{
   if (m_bUpdateBgImgWidget)
   {
      m_bUpdateBgImgWidget = false;
   }
   if (m_bUpdateEntryTextWidget)
   {
      m_oEntryTextWidget.Update();
      m_bUpdateEntryTextWidget = false;
   }
   if (m_bUpdateAdditionalTextWidget)
   {
      m_oHighlightTextWidget.Update();
      m_oEndTextWidget.Update();
      m_bUpdateAdditionalTextWidget = false;
   }
   if (m_bUpdateMatchTextWidget)
   {
      m_oMatchTextWidget.Update();
      m_bUpdateMatchTextWidget = false;
   }
   if (m_bStateChanged || m_bPropertyChanged)
   {
      vUpdateStateProperties();
   }
   if (m_bHighlightDataChanged)
   {
      m_sKeyReceived = "";
      vInsertChar();
      //Set the variable m_bHighlightDataChanged to false after vInsertChar() as used inside the function called from vInsertChar()
      //if (m_bHighlightDataChanged){//--m_oCursorPosition;}
      bPostEditFieldUpdMsg();
      m_bHighlightDataChanged = false;
   }
   if ((m_bUpdateInstructionText  || (!m_bInInstructionMode && (EF_MIN_TEXT_LENGTH == m_u8CurStrLength))))
   {
      m_oEntryTextWidget.SetTextColor(
         (enEFDisabled != m_enEFState) ? m_oEnabledInstructionTextColor : m_oDisabledInstructionTextColor);
      //kanji Req
      m_oHighlightTextWidget.SetTextColor((enEFDisabled != m_enEFState) ? m_oEnabledInstructionTextColor : m_oDisabledInstructionTextColor);
      m_oEndTextWidget.SetTextColor((enEFDisabled != m_enEFState) ? m_oEnabledInstructionTextColor : m_oDisabledInstructionTextColor);

      m_oEntryTextWidget.SetText(m_sInstructionText);
      _TODO("Editfield with Style")
      ///      m_oEntryTextWidget.SetFont(m_oInstructionTextFont); -> Style fr83hi
      m_oEntryTextWidget.Update();
      m_bInInstructionMode = true;
      m_bUpdateInstructionText = false;
   }
   if (m_bInInstructionMode && (EF_MIN_TEXT_LENGTH != m_u8CurStrLength))
   {
      m_oEntryTextWidget.SetTextColor(
         (enEFDisabled != m_enEFState) ? m_oEnabledEntryTextColor : m_oDisabledEntryTextColor);

      //kanji Req
      m_oHighlightTextWidget.SetTextColor((enEFDisabled != m_enEFState) ? m_oHighlightEnabledTextColor : m_oHighlightDisabledTextColor);
      m_oEndTextWidget.SetTextColor((enEFDisabled != m_enEFState) ? m_oEnabledEntryTextColor : m_oDisabledEntryTextColor);

      ///      m_oEntryTextWidget.SetFont(m_oEntryTextFont); -> Style fr83hi
      m_oEntryTextWidget.Update();
      m_bInInstructionMode = false;
   }
   if (m_bInvalidateRequired)
   {
      m_bInvalidateRequired = false;
      //temp fix till Sesa Label comes
      InvalidateLayout();
      Invalidate();
   }
}


/****************************************************************************
*  Called when Localization/Culture has changed during runtime
****************************************************************************/
void EditFieldWidget2D::CultureChanged()
{
   CulturePtr cultureCurrent = Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture();
   if (m_cultureCurrent != cultureCurrent)
   {
      m_cultureCurrent = cultureCurrent;
      onLanguageChange();
   }
}


void EditFieldWidget2D::onLanguageChange()
{
   vUpdateTextNodePosDim();
   vUpdateCursorPos();

   m_bInvalidateRequired = true;
   triggerUpdate();
}


void EditFieldWidget2D::OnLayoutingInfoChanged(TextLayoutingData& data)
{
   if (true == cobGetMultiLineLayouting())
   {
      m_charCountEachLine = data.GetCharCountEachLine();
      m_rectTopEachLine = data.GetRectTopEachLine();
      m_uNumberOfLines = static_cast<Candera::UInt32>(m_charCountEachLine.size());
      vProcessCursorPosition();
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D::OnLayoutingInfoChanged m_uNumberOfLines = %d ", m_uNumberOfLines));
   }
   else {} //do nothing
}


void EditFieldWidget2D::vProcessCursorPosition()
{
   Candera::String contentOfLine = "";
   std::vector<Candera::UInt32>::iterator it = m_charCountEachLine.begin();
   Candera::UInt32 totalChars = 0;
   Candera::UInt32 currentLine = 1;
   Candera::UInt32 offset = 0;
   if (m_oCursorPosition.GetCursorIndex() > 0)
   {
      while (it != m_charCountEachLine.end())
      {
         totalChars += *it;
         if (totalChars >= m_oCursorPosition.GetCursorIndex())
         {
            totalChars -= *it;
            offset = m_oCursorPosition.GetCursorIndex() - totalChars;
            if ((offset == 0)/* && (currentLine > 1)*/)
            {
               currentLine--;
            }
            break;
         }
         it++;
         currentLine++;
      }
   }
   m_uCurrentLine = currentLine;

   Candera::Int32 fontHeight = 0;
   if (currentLine <= m_rectTopEachLine.size())
   {
      fontHeight += m_rectTopEachLine[currentLine - 1];
   }

   Candera::TChar* pVisibleText = CANDERA_NEW_ARRAY(Char, (m_u8CurStrLength * MAX_BYTE_FOR_A_CHARACTER) + 1);
   if (NULL != pVisibleText)
   {
      //initialize the temp string to empty string
      StringPlatform::Copy(pVisibleText, "");
      if ((m_charCountEachLine.size() > 0) && (m_uCurrentLine > 1) && (m_uCurrentLine < m_charCountEachLine.size()))
      {
         m_u8VisibleCharCount = m_charCountEachLine[m_uCurrentLine - 1];
      }
      else if ((m_uCurrentLine == 1) || (m_uCurrentLine == 0))
      {
         m_u8VisibleCharCount = m_oCursorPosition.GetCursorIndex();
      }
      offset = (offset > 0) ? offset : m_u8VisibleCharCount;
      uCopyUTF8Data(pVisibleText, m_u8CurStrLength, 0, m_sMaskedText, totalChars, offset);
      contentOfLine = pVisibleText;

      if (NULL != m_pCursorNode)
      {
         Candera::UInt32 u16offset = cou16ComputeWidth(contentOfLine, 0, offset);
         m_pCursorNode->SetPosition(m_vEntryTextNodePos + Candera::Vector2(static_cast<Candera::Float>(u16offset), static_cast<Candera::Float>(fontHeight)));
      }
      m_bInvalidateRequired = true;
      triggerUpdate();
      FEATSTD_SAFE_DELETE_ARRAY(pVisibleText);
   }
}


/****************************************************************************
*     Description : Initializes the widget so that all referred resource
*                   can be resolved (Overridden)
*     Parameters  : assetFactory Asset provider to resolve all referred resources
****************************************************************************/
void EditFieldWidget2D::InitWidget()
{
   Base::InitWidget();
   m_cultureCurrent = Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture();
   if (NULL != GetAssetProvider())
   {
#ifdef USE_TEXTEFFECT_FOR_EDITFIELD
      m_oEntryTextWidget.Init(GetAssetProvider());
      m_oMatchTextWidget.Init(GetAssetProvider());
#else
      m_oEntryTextWidget.InitWidget();
      m_oMatchTextWidget.InitWidget();
#endif
      m_enEFState = m_bEditFieldEnabled ? enEFNormal : enEFDisabled;
      //SetRepeatCount to 0 for continuosly Repeating Animation
      m_oCursorAnimProp.SetRepeatCount(REPEAT_CONST);
      m_bUpdateBgImgWidget = false;
      vConfgEntryTxtWidget();
      vConfgMatchTxtWidget();
      vUpdateCursorPos();
      vUpdateCursorImage();
      m_u32TruncationTextDisplayedWidth = cou8GetTruncationTextDisplayedWidth();
   }
}


/****************************************************************************
* Req ID      : NA
* Function    : OnParentViewRenderingEnabled
* Called by   : HMI Framework whenever rendering of the parent view is
*               enabled or disabled
* Description : Function for the EditField  to be ready with the data required
*               to be displayed
* Parameters  : enable if the rendering is enabled or disabled
* Return      : void
****************************************************************************/
void EditFieldWidget2D::OnParentViewRenderingEnabled(bool enable)
{
   if ((true == enable) &&
         (NULL != GetAssetProvider()))
   {
      if (NULL == GetNode())
      {
         return;
      }
   }
   else if (false == enable)
   {
      PropertyModified("NewCursorPosition");
   }
}


/****************************************************************************
*     Description : Is called when a message shall be distributed through the view
*                   tree and its views and widgets (Overridden)
*     Parameters  : Message object to be processed.
*     Return      : true if the message is consumed
****************************************************************************/
bool EditFieldWidget2D::OnMessage(const Message& msg)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%10s] - OnMessage called with [%s] message",
                       GetLegacyName(), msg.GetName()));

   bool bIsMsgConsumed = Base::OnMessage(msg);

   switch (msg.GetId())
   {
      case TouchMsg::ID:
      {
         const TouchMsg* pTouchMsg = message_cast<const TouchMsg*>(&msg);
         if ((NULL != pTouchMsg) && m_bEditFieldEnabled)
         {
            TouchMsgState::Enum enTouchMsgState = pTouchMsg ->GetState();

            //Change state to selected on touch down
            if (TouchMsgState::Down == enTouchMsgState)
            {
               ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s] - Touch Down received.", GetLegacyName()));
               m_enEFState = enEFSelected;
               m_bStateChanged = true;
            }
         }
         bIsMsgConsumed = true;
         break;
      }
      case ::hmibase::widget::editfield::EditFieldHightlightReqMsg::ID:
      {
         const ::hmibase::widget::editfield::EditFieldHightlightReqMsg* pEditFieldHightlightReqMsg = message_cast<const ::hmibase::widget::editfield::EditFieldHightlightReqMsg*>(&msg);
         if (NULL != pEditFieldHightlightReqMsg && m_bEditFieldEnabled)
         {
            vProcessEditFieldHightlightReqMsg(*pEditFieldHightlightReqMsg);
         }
         bIsMsgConsumed = true;
         break;
      }
      default:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s].Message not consumed.", GetLegacyName()));
         break;
   }
   return bIsMsgConsumed;
}


Candera::UInt32 EditFieldWidget2D::calculateOffeSetOnTouch(Candera::UInt32 newIndexOffset, bool isRightToLeft)
{
   // let assume that the string is AB@#$%^C in Input buffer, wher @#$%^ is Arabic
   //Output buffer will show the same as => AB^%$#@C
   //current cursor pos is after between A and B => A|B^%$#@C
   //user has touched after third character from beginning of output buffer, so new cursor is expected to be in output buffer as => AB^|%$#@C

   //concept: read the number of Arabic characters before and after the probable new cursor position
   //create two string for the same
   FeatStd::Internal::Utf8TextIterator iter_DisplayedText_BeforeCur(m_sDisplayedText.GetCString());
   FeatStd::Internal::Utf8TextIterator iter_DisplayedText_AFterCur(m_sDisplayedText.GetCString());

   Candera::Int32 virtualCurPos = static_cast<Candera::Int32>(newIndexOffset);
   Candera::Int32 index = 0;
   Candera::UInt32 charCount_BeforeCur = 0;
   Candera::UInt32 charCount_AfterCur = 0;
   TChar achCurrChar[4];
   //make the created string reach to the probale new cursor position
   while (index != virtualCurPos)
   {
      if (!iter_DisplayedText_BeforeCur.EndOfString())
      {
         iter_DisplayedText_BeforeCur.Advance();
      }
      if (!iter_DisplayedText_AFterCur.EndOfString())
      {
         iter_DisplayedText_AFterCur.Advance();
      }
      index++;
   }

   //count all the English characters AFTER the probable cursor position
   while (!(iter_DisplayedText_AFterCur.EndOfString()))
   {
      FeatStd::Internal::TextEncoding::Copy(achCurrChar, 4, iter_DisplayedText_AFterCur.Pos(), 1/*read only one UTF-8 character*/);
      if (isRightToLeft != IsArabicChar(static_cast<wchar_t>(FeatStd::Internal::TextEncoding::Ucs4(achCurrChar))))
      {
         charCount_AfterCur++;
      }
      else
      {
         break;
      }
      if (!iter_DisplayedText_AFterCur.EndOfString())
      {
         iter_DisplayedText_AFterCur.Advance();
      }
   }

   index--;

   //create one more temporary string which will have one character (the last one) chopped off from the iter_DisplayedText_BeforeCur
   while (index >= 0)
   {
      Candera::Int32 index_DisplayedText_Temp = 0;
      FeatStd::Internal::Utf8TextIterator iter_DisplayedText_Temp(m_sDisplayedText.GetCString());

      while (index != index_DisplayedText_Temp)
      {
         if (!iter_DisplayedText_Temp.EndOfString())
         {
            iter_DisplayedText_Temp.Advance();
         }
         index_DisplayedText_Temp++;
      }
      iter_DisplayedText_BeforeCur = iter_DisplayedText_Temp;

      //count all the English characters BEFORE the probable cursor position
      FeatStd::Internal::TextEncoding::Copy(achCurrChar, 4, iter_DisplayedText_BeforeCur.Pos(), 1/*read only one UTF-8 character*/);
      if (isRightToLeft != IsArabicChar(static_cast<wchar_t>(FeatStd::Internal::TextEncoding::Ucs4(achCurrChar))))
      {
         charCount_BeforeCur++;
         index--;
      }
      else
      {
         break;
      }
   }
   //given the number of Arabic characters are known before and after the new probable cursor position, calculate the new offset
   newIndexOffset += charCount_AfterCur - charCount_BeforeCur;
   return newIndexOffset;
}


/****************************************************************************
*     Description : vProcessOnTouchForMultiLine
*     Parameters  : Touch coordinates
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vProcessOnTouchForMultiLine(Candera::Vector2 point)
{
   //Obtain offset of point of touch from start of EditFieldTextArea
   Candera::Float flOffsetX = (point.GetX()) - ((m_pEntryTextNode->GetWorldPosition()).GetX());
   Candera::Float flOffsetY = (point.GetY()) - ((m_pEntryTextNode->GetWorldPosition()).GetY());
   Candera::Int32 touchedPoint = 0;
   Candera::Float flWidth1 = 0;
   bool enable = true;
   Candera::UInt32 numberOfLines = 1;
   std::vector<Candera::Int32>::iterator it = m_rectTopEachLine.begin();
   // TODO: Improve this whole block
   while ((flOffsetY > 0) && enable && it != m_rectTopEachLine.end())
   {
      touchedPoint = *it;
      if (static_cast<Candera::Float>(touchedPoint) < flOffsetY)
      {
         numberOfLines++;
      }
      else
      {
         numberOfLines--;
         if (numberOfLines > 0 && (numberOfLines < m_rectTopEachLine.size()))
         {
            touchedPoint = m_rectTopEachLine[numberOfLines - 1];
         }
         enable = false;
      }
      ++it;
   }

   if ((numberOfLines <= m_uNumberOfLines) && (numberOfLines <= couGetMaximumNumberOfLines()) /*&& (numberOfLines <= m_uCurrentLine)*/)
   {
      std::vector<Candera::UInt32>::iterator it = m_charCountEachLine.begin();
      Candera::UInt32 count = 1;
      Candera::UInt32 index = 0;
      Candera::UInt32 charCount = 0;
      Candera::String contentOfLine = "";
      while (it != m_charCountEachLine.end())
      {
         if (count == numberOfLines)
         {
            charCount = *it;
            break;
         }
         index += *it;
         ++it;
         count++;
      }
      Candera::TChar* pVisibleText = CANDERA_NEW_ARRAY(Char, (m_u8CurStrLength * MAX_BYTE_FOR_A_CHARACTER) + 1);
      if (NULL != pVisibleText)
      {
         //initialize the temp string to empty string
         StringPlatform::Copy(pVisibleText, "");
         charCount = (charCount > 0) ? charCount : m_u8VisibleCharCount;
         uCopyUTF8Data(pVisibleText, m_u8CurStrLength, 0, m_sMaskedText, index, charCount);
         contentOfLine = pVisibleText;

         Candera::UInt32 u8NewIndexOffset = EF_MIN_INDEX;
         UInt32 displayedTextLength = u8UTFLength(contentOfLine);
         while ((u8NewIndexOffset < displayedTextLength) && (flOffsetX >= flWidth1))
         {
            u8NewIndexOffset++;
            flWidth1 = static_cast<Candera::Float>(cou16ComputeWidth(contentOfLine, 0, u8NewIndexOffset));
         }
         /*To Compute exact Cursor position in the current line, based whether touch
         happened before the half width of the touched char*/
         if (u8NewIndexOffset > EF_MIN_INDEX)
         {
            Candera::Float flWidth2 = static_cast<Candera::Float>(cou16ComputeWidth(contentOfLine, 0, (u8NewIndexOffset - 1)));

            if (flOffsetX < ((flWidth1 + flWidth2) / CONST_FOR_AVG_OF_WIDTHS))
            {
               u8NewIndexOffset--;
            }
         }
         if (NULL != m_pCursorNode)
         {
            Candera::UInt32 u16offset = cou16ComputeWidth(contentOfLine, 0, u8NewIndexOffset);
            m_pCursorNode->SetPosition(m_vEntryTextNodePos + Candera::Vector2(static_cast<Candera::Float>(u16offset), static_cast<Candera::Float>(touchedPoint)));
            //index added as it will take to the line in which currently cursor is present
            m_oCursorPosition = index + u8NewIndexOffset;
         }
         m_uCurrentLine = numberOfLines;
         m_bInvalidateRequired = true;
         triggerUpdate();
         FEATSTD_SAFE_DELETE_ARRAY(pVisibleText);
      }
   }
}


/****************************************************************************
*     Description : Overloaded function used when the widget should perform
*                    a custom hit test on a Touch message  (Overridden)
*     Parameters  : Camera reference and Touch coordinates
*     Return      : true if touch intersects the node else false
****************************************************************************/
bool EditFieldWidget2D::OnTouch(const Candera::Camera2D& camera2D, const Candera::Vector2& point)
{
   bool bRtnValue = false;
   if (IsVisible() && (GetNode() != NULL) && (GetNode()->IsEffectiveRenderingEnabled()))
   {
      bool isRightToLeft = false;
      bRtnValue = IsPickIntersectingNode(camera2D, GetNode(), point);
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s] - OnTouch called.", GetLegacyName()));
      Candera::Float flOffset = 0;

      //Cursor positioning on touch in FreeTextMode
      if (bRtnValue && GetTouchable() && m_bEditFieldEnabled && m_pEntryTextNode && (FreeTextMode == m_enWidgetMode))
      {
         if (false == cobGetMultiLineLayouting())
         {
            //Obtain offset of point of touch from start of EditFieldTextArea
            if (!m_cultureCurrent.PointsToNull() && m_cultureCurrent->GetTextDirection() == Candera::Globalization::RightToLeft)
            {
               isRightToLeft = true;
               Candera::Rectangle rectangle;
               m_pEntryTextNode->GetComputedBoundingRectangle(rectangle);
               //Candera::Int off = (cou16GetBgWidth() - cou8GetTextAreaMarginX() + (m_pEntryTextNode->GetWorldPosition()).GetX());
               flOffset = static_cast<Candera::Float>(rectangle.GetWidth() - (point.GetX() - (static_cast<Candera::Float>(cou16GetBgWidth()) + m_pEntryTextNode->GetWorldPosition().GetX() - (2 * static_cast<Candera::Float>(cou8GetTextAreaMarginX())) - rectangle.GetWidth())));
            }
            else
            {
               flOffset = (point.GetX()) - ((m_pEntryTextNode->GetWorldPosition()).GetX());
            }
            /*Compute Cursor index based on touch
            if touch happens on the char, Cursor is positioned to nearest cursor position from touch
            i.e if touch happened somewere before the half the width of touched char,
            cursor is positioned before that char and vice-versa
            */
            Candera::Float flWidth1 = 0;
            //index offset from VisibleStartIndex for point of touch
            Candera::UInt32 u8NewIndexOffset = EF_MIN_INDEX;
            /*Calculate the NewIndexOffset, which is the index for cursor
            position to the left of touch char
            */
            UInt32 displayedTextLength = u8UTFLength(m_sDisplayedText);
            while ((u8NewIndexOffset < displayedTextLength) && (flOffset >= flWidth1))
            {
               u8NewIndexOffset++;
               flWidth1 = static_cast<Candera::Float>(cou16ComputeWidth(m_sDisplayedText, 0, u8NewIndexOffset));
            }
            /*To Compute exact Cursor position, based whether touch
            happened before the half width of the touched char*/
            if (u8NewIndexOffset > EF_MIN_INDEX)
            {
               Candera::Float flWidth2 = static_cast<Candera::Float>(cou16ComputeWidth(m_sDisplayedText, 0, (u8NewIndexOffset - 1)));

               if (flOffset < ((flWidth1 + flWidth2) / CONST_FOR_AVG_OF_WIDTHS))
               {
                  u8NewIndexOffset--;
               }
            }
            //Special handling in case of arabic charachters are entered
            u8NewIndexOffset = calculateOffeSetOnTouch(u8NewIndexOffset, isRightToLeft);
            //set new cursor index and update cursor position
            if (u8NewIndexOffset > m_oCursorPosition.GetDisplayedCursorIndex())
            {
               m_oCursorPosition.ShiftRight(u8NewIndexOffset - m_oCursorPosition.GetDisplayedCursorIndex());
            }
            else
            {
               m_oCursorPosition.ShiftLeft(m_oCursorPosition.GetDisplayedCursorIndex() - u8NewIndexOffset);
            }
            m_mixedCharHandlingRequired = true;
            vUpdateCursorPos();
         }
         else
         {
            vProcessOnTouchForMultiLine(point);
         }
         bPostEditFieldUpdMsg();
      }
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%10s] - OnTouch returning with value[%s].",
                          GetLegacyName(), (bRtnValue ? "TRUE" : "FALSE")));
   }
   return bRtnValue;
}


/****************************************************************************
*     Description : Is triggered when the widget gets the focus (Overridden)
****************************************************************************/
void EditFieldWidget2D::OnFocus()
{
   if (m_bEditFieldEnabled)
   {
      //if Focus gained by event other than touch say rotary
      if (enEFSelected != m_enEFState)
      {
         m_enEFState = enEFFocused;
      }
      m_bStateChanged = true;
   }
#ifdef xxxhtcFocusHandlingxxx
   if (copGetFocusBitmapNode())
   {
      copGetFocusBitmapNode()->SetRenderingEnabled(true);
      m_bInvalidateRequired = true;
      vCheckAndInvalidate();
      //Invalidate();
   }
#endif
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s] - gains the focus.", GetLegacyName()));
}


/****************************************************************************
*     Description : Is triggered when the widget loses the focus (Overridden)
****************************************************************************/
void EditFieldWidget2D::OnLostFocus()
{
   if (m_bEditFieldEnabled)
   {
      //OnFocusLost Set widget state to Normal state if Enabled
      m_enEFState = enEFNormal;
      m_bStateChanged = true;
   }
#ifdef xxxhtcFocusHandlingxxx
   if (copGetFocusBitmapNode())
   {
      copGetFocusBitmapNode()->SetRenderingEnabled(false);
      m_bInvalidateRequired = true;
      vCheckAndInvalidate();
      //Invalidate();
   }
#endif
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s] - losses focus.", GetLegacyName()));
}


//Getters and Setters for EditFieldWidget2D

/****************************************************************************
*     Function    : coenGetMode
*     Description : Interface for accesing Widget Mode
*     Parameters  : none
*     Return      : Candera::enWidgetMode
****************************************************************************/
Candera::enWidgetMode EditFieldWidget2D::coenGetMode() const
{
   return m_enWidgetMode;
}


/****************************************************************************
*     Function    : pSetMode
*     Description : Interface for Setting Widget Mode
*     Parameters  : Candera::enWidgetMode
*     Return      : none
****************************************************************************/
void EditFieldWidget2D::vSetMode(Candera::enWidgetMode enMode)
{
   m_enWidgetMode = enMode;
   m_bPropertyChanged = true;
}


/****************************************************************************
*     Function    : cou16GetBgWidth
*     Description : Interface for accesing BackGround width
*     Parameters  : none
*     Return      : Candera::UInt16
****************************************************************************/
Candera::UInt32 EditFieldWidget2D::cou16GetBgWidth() const
{
   return m_u16BgWidth;
}


/****************************************************************************
*     Function    : vSetBgWidth
*     Description : Interface for setting BackGround width
*     Parameters  : Candera::UInt16
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetBgWidth(Candera::UInt32 u16BgWidth)
{
   m_u16BgWidth = u16BgWidth;

   //Offset to left and right of Text from Background
   Candera::UInt32 u8Offset = CONST_FOR_MARGIN * m_u8TextAreaMarginX;
   if (m_u16BgWidth > u8Offset)
   {
      m_u16TextAreaWidth = m_u16BgWidth - u8Offset;
      m_v2EntryTextSize.SetX(static_cast<Candera::Float>(m_u16BgWidth - u8Offset));
   }
   else
   {
      m_u16TextAreaWidth = 0;
      m_v2EntryTextSize.SetX(0.0f);
   }
   vUpdateTextNodePosDim();
   m_bPropertyChanged = true;
}


/****************************************************************************
*     Function    : cou8GetBgHeight
*     Description : Interface for accesing BackGround Height
*     Parameters  : none
*     Return      : Candera::UInt8
****************************************************************************/
Candera::UInt32 EditFieldWidget2D::cou8GetBgHeight() const
{
   return m_u8BgHeight;
}


/****************************************************************************
*     Function    : vSetBgHeight
*     Description : Interface for setting BackGround Height
*     Parameters  : Candera::UInt8
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetBgHeight(Candera::UInt32 u8BgHeight)
{
   m_u8BgHeight = u8BgHeight;

   //Offset to top and buttom of Text from Background
   Candera::UInt32 u8Offset = CONST_FOR_MARGIN * m_u8TextAreaMarginY;
   if (m_u8BgHeight > u8Offset)
   {
      m_u8TextAreaHeight = m_u8BgHeight - u8Offset;
      m_v2EntryTextSize.SetY(static_cast<Candera::Float>(m_u8BgHeight - u8Offset));
   }
   else
   {
      m_u8TextAreaHeight = 0;
      m_v2EntryTextSize.SetY(0.0f);
   }
   vUpdateTextNodePosDim();
   m_bPropertyChanged = true;
}


/****************************************************************************
*     Function    : cou8GetTextAreaMarginX
*     Description : Interface for accesing Text area MarginX
*     Parameters  : none
*     Return      : Candera::UInt8
****************************************************************************/
Candera::UInt32 EditFieldWidget2D::cou8GetTextAreaMarginX() const
{
   return m_u8TextAreaMarginX;
}


/****************************************************************************
*     Function    : vSetTextAreaMarginX
*     Description : Interface for setting Text area MarginX
*     Parameters  : Candera::UInt8
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetTextAreaMarginX(Candera::UInt32 u8TextAreaMarginX)
{
   m_u8TextAreaMarginX = u8TextAreaMarginX;

   //Offset to left and right of Text from Background
   Candera::UInt32 u8Offset = static_cast<UInt32>(CONST_FOR_MARGIN * m_u8TextAreaMarginX);
   if (m_u16BgWidth > u8Offset)
   {
      m_u16TextAreaWidth = m_u16BgWidth - u8Offset;
      m_v2EntryTextSize.SetX(static_cast<Candera::Float>(m_u16BgWidth - u8Offset));
   }
   else
   {
      m_u16TextAreaWidth = 0;
      m_v2EntryTextSize.SetX(0.0f);
   }
   vUpdateTextNodePosDim();
   m_bPropertyChanged = true;
}


/****************************************************************************
*     Function    : cou8GetTextAreaMarginY
*     Description : Interface for accesing Text area MarginY
*     Parameters  : none
*     Return      : Candera::UInt8
****************************************************************************/
Candera::UInt32 EditFieldWidget2D::cou8GetTextAreaMarginY() const
{
   return m_u8TextAreaMarginY;
}


/****************************************************************************
*     Function    : vSetTextAreaMarginY
*     Description : Interface for setting Text area MarginY
*     Parameters  : Candera::UInt8
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetTextAreaMarginY(Candera::UInt32 u8TextAreaMarginY)
{
   m_u8TextAreaMarginY = u8TextAreaMarginY;

   //Offset to top and buttom of Text from Background
   Candera::UInt32 u8Offset = static_cast<UInt32>(CONST_FOR_MARGIN * m_u8TextAreaMarginY);
   if (m_u8BgHeight > u8Offset)
   {
      m_u8TextAreaHeight = m_u8BgHeight - u8Offset;
      m_v2EntryTextSize.SetY(static_cast<Candera::Float>(m_u8BgHeight - u8Offset));
   }
   else
   {
      m_u8TextAreaHeight = 0;
      m_v2EntryTextSize.SetY(0.0f);
   }
   vUpdateTextNodePosDim();
   m_bPropertyChanged = true;
}


/****************************************************************************
*     Function    : cou8GetMaxTextLength
*     Description : Interface for Accesing MaxTextLength
*     Parameters  : void
*     Return      : Candera::UInt8
****************************************************************************/
Candera::UInt32 EditFieldWidget2D::cou8GetMaxTextLength() const
{
   return m_u8MaxTextLength;
}


/****************************************************************************
*     Function    : SetTextStyle
*     Description : Interface for Setting Text Style
*     Parameters  : StyleType
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::SetTextStyle(const StyleType& style)
{
   m_TextStyle = style;
   if (false == m_TextStyle.PointsToNull())
   {
      m_oEntryTextWidget.SetStyle(m_TextStyle);
      m_oMatchTextWidget.SetStyle(m_TextStyle);
      m_oHighlightTextWidget.SetStyle(m_TextStyle);
      m_oEndTextWidget.SetStyle(m_TextStyle);
      oShapingOption = TextRendering::ShapingOptions(m_TextStyle);
      m_bPropertyChanged = true;
      m_bInvalidateRequired = true;
   }
}


/****************************************************************************
*     Function    : vSetMaxTextLength
*     Description : Interface for setting MaxTextLength
*     Parameters  : Candera::UInt8
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetMaxTextLength(Candera::UInt32 u8MaxTextLength)
{
   m_u8MaxTextLength = u8MaxTextLength;
}


/****************************************************************************
*     Function    : cosGetMaskChar
*     Description : Interface for getting mask char
*     Parameters  : Candera::String
*     Return      : void
****************************************************************************/
Candera::String EditFieldWidget2D::cosGetMaskChar() const
{
   return m_sMaskChar;
}


/****************************************************************************
*     Function    : vSetMaskChar
*     Description : Interface for setting mask char
*     Parameters  : Candera::String
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetMaskChar(Candera::String sMaskChar)
{
   m_sMaskChar = sMaskChar;
}


/****************************************************************************
*     Function    : cosGetInstructionText
*     Description : Interface for getting Instruction text string
*     Parameters  : void
*     Return      : Candera::String
****************************************************************************/
Candera::String EditFieldWidget2D::cosGetInstructionText() const
{
   return m_sInstructionText;
}


/****************************************************************************
*     Function    : vSetInstructionText
*     Description : Interface for setting Instruction text string
*     Parameters  : Candera::String
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetInstructionText(Candera::String sInstructionText)
{
   m_sInstructionText = sInstructionText;
   m_bUpdateInstructionText = true;
}


/****************************************************************************
*     Function    : copGetBgNode
*     Description : Interface for accesing Bg Node
*     Parameters  : None
*     Return      : Candera::Node2D*
****************************************************************************/
Candera::Node2D* EditFieldWidget2D::getBackgroundNode() const
{
   return m_pBackgroundNode;
}


/****************************************************************************
*     Description : Interface for setting Bg Node
*     Parameters  : Candera::Node2D*
****************************************************************************/
void EditFieldWidget2D::setBackgroundNode(Candera::Node2D* pBgNode)
{
   m_pBackgroundNode = pBgNode;
   m_bUpdateBgImgWidget = true;
}


/****************************************************************************
*     Function    : copGetCursorNode
*     Description : Interface for accesing Cursor Node
*     Parameters  : None
*     Return      : Candera::Node2D*
****************************************************************************/
Candera::Node2D* EditFieldWidget2D::getCursorNode() const
{
   return m_pCursorNode;
}


/****************************************************************************
*     Function    : vSetCursorNode
*     Description : Interface for setting Cursor Node
*     Parameters  : Candera::Node2D*
*     Return      : None
****************************************************************************/
void EditFieldWidget2D::setCursorNode(Candera::Node2D* pCursorNode)
{
   m_pCursorNode = pCursorNode;
   vUpdateCursorPos();
   vUpdateCursorImage();
}


/****************************************************************************
*     Function    : copGetEntryTextNode
*     Description : Interface for accesing EntryText Node
*     Parameters  : None
*     Return      : Candera::Node2D*
****************************************************************************/
Candera::Node2D* EditFieldWidget2D::copGetEntryTextNode() const
{
   return m_pEntryTextNode;
}


/****************************************************************************
*     Function    : vSetEntryTextNode
*     Description : Interface for setting EntryText Node
*     Parameters  : Candera::Node2D*
*     Return      : None
****************************************************************************/
void EditFieldWidget2D::vSetEntryTextNode(Candera::Node2D* pEntryTextNode)
{
   m_pEntryTextNode = pEntryTextNode;
   m_oEntryTextWidget.SetNode(m_pEntryTextNode);
   m_bUpdateEntryTextWidget = true;
}


/****************************************************************************
*     Function    : copGetMatchTextNode
*     Description : Interface for accesing MatchText Node
*     Parameters  : None
*     Return      : Candera::Node2D*
****************************************************************************/
Candera::Node2D* EditFieldWidget2D::copGetMatchTextNode() const
{
   return m_pMatchTextNode;
}


/****************************************************************************
*     Function    : vSetMatchTextNode
*     Description : Interface for setting MatchText Node
*     Parameters  : Candera::Node2D*
*     Return      : None
****************************************************************************/
void EditFieldWidget2D::vSetMatchTextNode(Candera::Node2D* pMatchTextNode)
{
   m_pMatchTextNode =  pMatchTextNode;
   m_oMatchTextWidget.SetNode(pMatchTextNode);
   m_bUpdateMatchTextWidget = true;
}


/****************************************************************************
*     Function    : copGetHighlightTextNode
*     Description : Interface for accesing HighLight Text Node
*     Parameters  : None
*     Return      : Candera::Node2D*
****************************************************************************/
Candera::Node2D* EditFieldWidget2D::copGetHighlightTextNode() const
{
   return m_pHighlightTextNode;
}


/****************************************************************************
*     Function    : vSetHighlightTextNode
*     Description : Interface for setting HighlightText Node
*     Parameters  : Candera::Node2D*
*     Return      : None
****************************************************************************/
void EditFieldWidget2D::vSetHighlightTextNode(Candera::Node2D* pHighlightTextNode)
{
   m_pHighlightTextNode = pHighlightTextNode;
   m_oHighlightTextWidget.SetNode(pHighlightTextNode);
   m_bInvalidateRequired = true;
}


/****************************************************************************
*     Function    : copGetEndTextNode
*     Description : Interface for accesing End Text Node
*     Parameters  : None
*     Return      : Candera::Node2D*
****************************************************************************/
Candera::Node2D* EditFieldWidget2D::copGetEndTextNode() const
{
   return m_pEndTextNode;
}


/****************************************************************************
*     Function    : vSetEndTextNode
*     Description : Interface for setting EndText Node
*     Parameters  : Candera::Node2D*
*     Return      : None
****************************************************************************/
void EditFieldWidget2D::vSetEndTextNode(Candera::Node2D* pEndTextNode)
{
   m_pEndTextNode = pEndTextNode;
   m_oEndTextWidget.SetNode(pEndTextNode);
   m_bInvalidateRequired = true;
}


/****************************************************************************
*     Function    : cou8GetHighlightStartIndex
*     Description : Interface for setting Highlight Start Index
*     Parameters  : None
*     Return      : Candera::UInt8
****************************************************************************/
Candera::UInt32 EditFieldWidget2D::cou8GetHighlightStartIndex() const
{
   return m_uHighlightStartIndex;
}


/****************************************************************************
*     Function    : vSetHighlightStartIndex
*     Description : Interface for setting HighLightStartIndex
*     Parameters  : Candera::UInt8 highlightStartIndex
*     Return      : None
****************************************************************************/
void EditFieldWidget2D::vSetHighlightStartIndex(Candera::UInt32 highlightStartIndex)
{
   m_bHighlightDataChanged = true;
   m_uHighlightStartIndex = highlightStartIndex;
   m_bInvalidateRequired = true;
   triggerUpdate();
}


/****************************************************************************
*     Function    : cou8GetHighlightCount
*     Description : Interface for setting Highlight Start Index
*     Parameters  : None
*     Return      : Candera::UInt8
****************************************************************************/
Candera::UInt32 EditFieldWidget2D::cou8GetHighlightCount() const
{
   return m_uHighlightCount;
}


/****************************************************************************
*     Function    : vSetHighlightCount
*     Description : Interface for setting HighLightStartIndex
*     Parameters  : Candera::Unt8
*     Return      : None
****************************************************************************/
void EditFieldWidget2D::vSetHighlightCount(Candera::UInt32 u8HighlightCount)
{
   m_bHighlightDataChanged = true;
   m_uHighlightCount = u8HighlightCount;
   m_bInvalidateRequired = true;
   triggerUpdate();
}


void EditFieldWidget2D::SetNormalBgBitmap(const BitmapType& bitmap)
{
   _bitmapNormalBg = bitmap;
   if (enEFNormal == m_enEFState)
   {
      m_bPropertyChanged = true;
   }
   // OnChanged((::FeatStd::UInt32)BitmapPropertyId);
   if (0 != GetParentView())
   {
      PropertyModified("NormalBgBitmap");  // Todo: check complete property handling
   }
}


void EditFieldWidget2D::SetDisabledBgBitmap(const BitmapType& bitmap)
{
   _bitmapDisabledBg = bitmap;
   if (enEFDisabled == m_enEFState)
   {
      m_bPropertyChanged = true;
   }
   // OnChanged((::FeatStd::UInt32)BitmapPropertyId);
   if (0 != GetParentView())
   {
      PropertyModified("DisabledBgBitmap"); // Todo: check complete property handling
   }
}


void EditFieldWidget2D::SetSelectedBgBitmap(const BitmapType& bitmap)
{
   _bitmapSelectedBg = bitmap;
   if (enEFSelected == m_enEFState)
   {
      m_bPropertyChanged = true;
   }
   // OnChanged((::FeatStd::UInt32)BitmapPropertyId);
   if (0 != GetParentView())
   {
      PropertyModified("SelectedBgBitmap"); // Todo: check complete property handling
   }
}


void EditFieldWidget2D::SetCursorBgBitmap(const BitmapType& bitmap)
{
   _bitmapCursorBg = bitmap;
   vUpdateCursorImage();

   /*
   if (enEFCursor == m_enEFState)
      {
         m_bPropertyChanged = true;
      }
   */
   // OnChanged((::FeatStd::UInt32)BitmapPropertyId);
   if (0 != GetParentView())
   {
      PropertyModified("CursorBgBitmap"); // Todo: check complete property handling
   }
}


/****************************************************************************
*     Function    : cooGetDefaultEnabledTextColor
*     Description : Interface for Accesing Default text Color in Enabled State
*     Parameters  : void
*     Return      : Candera::Color
****************************************************************************/
Candera::Color EditFieldWidget2D::cooGetDefaultEnabledTextColor() const
{
   return m_oDefaultEnabledTextColor;
}


/****************************************************************************
*     Function    : vSetEnabledEntryTextColor
*     Description : Interface for setting Default text Color in Enabled State
*     Parameters  : Candera::Color
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetDefaultEnabledTextColor(Candera::Color oDefaultEnabledTextColor)
{
   m_oDefaultEnabledTextColor = oDefaultEnabledTextColor;
}


/****************************************************************************
*     Function    : cooGetDefaultDisabledTextColor
*     Description : Interface for Accesing Default text Color in Disabled State
*     Parameters  : void
*     Return      : Candera::Color
****************************************************************************/
Candera::Color EditFieldWidget2D::cooGetDefaultDisabledTextColor() const
{
   return m_oDefaultDisabledTextColor;
}


/****************************************************************************
*     Function    : vSetDefaultDisabledTextColor
*     Description : Interface for setting Default text Color in Disabled State
*     Parameters  : Candera::Color
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetDefaultDisabledTextColor(Candera::Color oDefaultDisabledTextColor)
{
   m_oDefaultDisabledTextColor = oDefaultDisabledTextColor;
}


//Kanji Req Start

/****************************************************************************
*     Function    : cooGetHighlightEnabledTextColor
*     Description : Interface for Accesing HighLight text Color in Enabled State
*     Parameters  : void
*     Return      : Candera::Color
****************************************************************************/
Candera::Color EditFieldWidget2D::cooGetHighlightEnabledTextColor() const
{
   return m_oHighlightEnabledTextColor;
}


/****************************************************************************
*     Function    : vSetHighlightEnabledTextColor
*     Description : Interface for setting Highlight text Color in Enabled State
*     Parameters  : Candera::Color
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetHighlightEnabledTextColor(Candera::Color oHighlightEnabledTextColor)
{
   m_oHighlightEnabledTextColor = oHighlightEnabledTextColor;
}


/****************************************************************************
*     Function    : cooGetHighlightDisabledTextColor
*     Description : Interface for Accesing Highlight text Color in Disabled State
*     Parameters  : void
*     Return      : Candera::Color
****************************************************************************/
Candera::Color EditFieldWidget2D::cooGetHighlightDisabledTextColor() const
{
   return m_oHighlightDisabledTextColor;
}


/****************************************************************************
*     Function    : vSetHighlightDisabledTextColor
*     Description : Interface for setting Highlight text Color in Disabled State
*     Parameters  : Candera::Color
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetHighlightDisabledTextColor(Candera::Color oHighlightDisabledTextColor)
{
   m_oHighlightDisabledTextColor = oHighlightDisabledTextColor;
}


//Kanji Req End


/****************************************************************************
*     Function    : cooGetEntryTextFont
*     Description : Interface for Accesing Entry text font
*     Parameters  : void
*     Return      : Candera::TextRendering::Font
****************************************************************************/
Candera::TextRendering::Font EditFieldWidget2D::cooGetEntryTextFont() const
{
   return m_oEntryTextFont;
}


/****************************************************************************
*     Function    : vSetEntryTextFont
*     Description : Interface for setting Entry text font
*     Parameters  : Candera::TextRendering::Font
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetEntryTextFont(const Candera::TextRendering::Font& oEntryTextFont)
{
   m_oEntryTextFont = oEntryTextFont;
   if (EF_MIN_TEXT_LENGTH != m_u8CurStrLength)
   {
      m_bPropertyChanged = true;
   }
}


/****************************************************************************
*     Function    : cooGetEnabledEntryTextColor
*     Description : Interface for Accesing Entry text Color in Enabled State
*     Parameters  : void
*     Return      : Candera::Color
****************************************************************************/
Candera::Color EditFieldWidget2D::cooGetEnabledEntryTextColor() const
{
   return m_oEnabledEntryTextColor;
}


/****************************************************************************
*     Function    : vSetEnabledEntryTextColor
*     Description : Interface for setting Entry text Color in Enabled State
*     Parameters  : Candera::Color
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetEnabledEntryTextColor(Candera::Color oEnabledEntryTextColor)
{
   m_oEnabledEntryTextColor = oEnabledEntryTextColor;
   if ((EF_MIN_TEXT_LENGTH != m_u8CurStrLength) && (enEFDisabled != m_enEFState))
   {
      m_bPropertyChanged = true;
   }
}


/****************************************************************************
*     Function    : cooGetDisabledEntryTextColor
*     Description : Interface for Accesing Entry text Color in Disabled State
*     Parameters  : void
*     Return      : Candera::Color
****************************************************************************/
Candera::Color EditFieldWidget2D::cooGetDisabledEntryTextColor() const
{
   return m_oDisabledEntryTextColor;
}


/****************************************************************************
*     Function    : vSetDisabledEntryTextColor
*     Description : Interface for setting Entry text Color in Disabled State
*     Parameters  : Candera::Color
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetDisabledEntryTextColor(Candera::Color oDisabledEntryTextColor)
{
   m_oDisabledEntryTextColor = oDisabledEntryTextColor;
   if ((EF_MIN_TEXT_LENGTH != m_u8CurStrLength) && (enEFDisabled == m_enEFState))
   {
      m_bPropertyChanged = true;
   }
}


/****************************************************************************
*     Function    : cooGetMatchTextFont
*     Description : Interface for Accesing Match text font
*     Parameters  : void
*     Return      : Candera::TextRendering::Font
****************************************************************************/
Candera::TextRendering::Font EditFieldWidget2D::cooGetMatchTextFont() const
{
   return m_oMatchTextFont;
}


/****************************************************************************
*     Function    : vSetMatchTextFont
*     Description : Interface for setting Match text font
*     Parameters  : Candera::TextRendering::Font
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetMatchTextFont(const Candera::TextRendering::Font& oMatchTextFont)
{
   m_oMatchTextFont = oMatchTextFont;
   if (MatchTextMode == m_enWidgetMode)
   {
      m_bPropertyChanged = true;
   }
}


/****************************************************************************
*     Function    : cooGetEnabledMatchTextColor
*     Description : Interface for Accesing Match text Color in Enabled State
*     Parameters  : void
*     Return      : Candera::Color
****************************************************************************/
Candera::Color EditFieldWidget2D::cooGetEnabledMatchTextColor() const
{
   return m_oEnabledMatchTextColor;
}


/****************************************************************************
*     Function    : vSetEnabledMatchTextColor
*     Description : Interface for setting Match text Color in Enabled State
*     Parameters  : Candera::Color
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetEnabledMatchTextColor(Candera::Color oEnabledMatchTextColor)
{
   m_oEnabledMatchTextColor = oEnabledMatchTextColor;
   if ((MatchTextMode == m_enWidgetMode) && (enEFDisabled != m_enEFState))
   {
      m_bPropertyChanged = true;
   }
}


/****************************************************************************
*     Function    : cooGetDisabledMatchTextColor
*     Description : Interface for Accesing Match text Color in Disabled State
*     Parameters  : void
*     Return      : Candera::Color
****************************************************************************/
Candera::Color EditFieldWidget2D::cooGetDisabledMatchTextColor() const
{
   return m_oDisabledMatchTextColor;
}


/****************************************************************************
*     Function    : vSetDisabledMatchTextColor
*     Description : Interface for setting Match text Color in Disabled State
*     Parameters  : Candera::Color
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetDisabledMatchTextColor(Candera::Color oDisabledMatchTextColor)
{
   m_oDisabledMatchTextColor = oDisabledMatchTextColor;
   if ((MatchTextMode == m_enWidgetMode) && (enEFDisabled == m_enEFState))
   {
      m_bPropertyChanged = true;
   }
}


/****************************************************************************
*     Function    : cooGetInstructionTextFont
*     Description : Interface for Accesing Instruction text font
*     Parameters  : void
*     Return      : Candera::TextRendering::Font
****************************************************************************/
Candera::TextRendering::Font EditFieldWidget2D::cooGetInstructionTextFont() const
{
   return m_oInstructionTextFont;
}


/****************************************************************************
*     Function    : vSetInstructionTextFont
*     Description : Interface for setting Instruction text font
*     Parameters  : Candera::TextRendering::Font
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetInstructionTextFont(const Candera::TextRendering::Font& oInstructionTextFont)
{
   m_oInstructionTextFont = oInstructionTextFont;
   if (EF_MIN_TEXT_LENGTH == m_u8CurStrLength)
   {
      m_bUpdateInstructionText = true;
   }
}


/****************************************************************************
*     Function    : cooGetEnabledInstructionTextColor
*     Description : Interface for Accesing Match text Color in Enabled State
*     Parameters  : void
*     Return      : Candera::Color
****************************************************************************/
Candera::Color EditFieldWidget2D::cooGetEnabledInstructionTextColor() const
{
   return m_oEnabledInstructionTextColor;
}


/****************************************************************************
*     Function    : vSetEnabledInstructionTextColor
*     Description : Interface for setting Instruction text Color in Enabled State
*     Parameters  : Candera::Color
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetEnabledInstructionTextColor(Candera::Color oEnabledInstructionTextColor)
{
   m_oEnabledInstructionTextColor = oEnabledInstructionTextColor;
   if ((EF_MIN_TEXT_LENGTH == m_u8CurStrLength) && (enEFDisabled != m_enEFState))
   {
      m_bUpdateInstructionText = true;
   }
}


/****************************************************************************
*     Function    : cooGetDisabledInstructionTextColor
*     Description : Interface for Accesing Instruction text Color in Disabled State
*     Parameters  : void
*     Return      : Candera::Color
****************************************************************************/
Candera::Color EditFieldWidget2D::cooGetDisabledInstructionTextColor() const
{
   return m_oDisabledInstructionTextColor;
}


/****************************************************************************
*     Function    : vSetDisabledInstructionTextColor
*     Description : Interface for setting Instruction text Color in Disabled State
*     Parameters  : Candera::Color
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetDisabledInstructionTextColor(Candera::Color oDisabledInstructionTextColor)
{
   m_oDisabledInstructionTextColor = oDisabledInstructionTextColor;
   if ((EF_MIN_TEXT_LENGTH == m_u8CurStrLength) && (enEFDisabled == m_enEFState))
   {
      m_bUpdateInstructionText = true;
   }
}


/****************************************************************************
*     Function    : cosGetCursorAnimation
*     Description : Interface for getting Cursor animation path
*     Parameters  : void
*     Return      : Candera::String
****************************************************************************/
Candera::String EditFieldWidget2D::cosGetCursorAnimation() const
{
   return m_sCursorAnimation;
}


/****************************************************************************
*     Function    : vSetCursorAnimation
*     Description : Interface for setting Cursor animation path
*     Parameters  : Candera::String
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetCursorAnimation(Candera::String sCursorAnimation)
{
   m_sCursorAnimation = sCursorAnimation;
   m_bPropertyChanged = true;
}


/****************************************************************************
*     Function    : cou32GetBlinkInterval
*     Description : Interface for Accesing BlinkInterval
*     Parameters  : void
*     Return      : Candera::UInt32
****************************************************************************/
Candera::UInt32 EditFieldWidget2D::cou32GetBlinkInterval() const
{
   return m_u32BlinkInterval;
}


/****************************************************************************
*     Function    : vSetBlinkInterval
*     Description : Interface for setting BlinkInterval
*     Parameters  : Candera::UInt32
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetBlinkInterval(Candera::UInt32 u32BlinkInterval)
{
   if (0 != u32BlinkInterval)
   {
      m_u32BlinkInterval = u32BlinkInterval;
      //Compute Speed factor w.r.t default speed of 1 blink/1000msec
      Candera::Float flSpeed = static_cast<Candera::Float>(BLINK_INTERVAL) / static_cast<Candera::Float>(m_u32BlinkInterval);
      m_oCursorAnimProp.SetSpeedFactor(flSpeed);
   }
   //else if u32BlinkInterval is 0 use default value
   else
   {
      m_u32BlinkInterval = BLINK_INTERVAL;
   }
   m_bPropertyChanged = true;
}


/****************************************************************************
*     Function    : cosGetEditFieldInfo
*     Description : Interface for Accesing EditFieldInfo
*     Parameters  : Candera::String
*     Return      : void
****************************************************************************/
Candera::String EditFieldWidget2D::cosGetEditFieldInfo() const
{
   return m_sEditFieldInfo;
}


/****************************************************************************
*     Function    : vSetEditFieldInfo
*     Description : Interface for setting EditFieldInfo
*     Parameters  : Candera::String
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetEditFieldInfo(Candera::String sEditFieldInfo)
{
   m_sEditFieldInfo = sEditFieldInfo;
}


/****************************************************************************
*     Function    : cosGetNewCursorPosition
*     Description : Interface for Accesing CursorPosition (FreeextMode)
*     Parameters  : Candera::UInt16
*     Return      : void
****************************************************************************/
Candera::UInt32 EditFieldWidget2D::cou16GetNewCursorPosition() const
{
   return m_u16NewCursorPosition;
}


/****************************************************************************
*     Function    : cossetNewCursorPosition
*     Description : Interface for Accesing CursorPosition (FreeextMode)
*     Parameters  : Candera::UInt16
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetNewCursorPosition(Candera::UInt32 u16NewCursorPos)
{
   m_u16NewCursorPosition = u16NewCursorPos;
   if (0 != GetParentView())
   {
      m_oCursorPosition.SetCursorIndex(static_cast<Candera::UInt32>(m_u16NewCursorPosition));
      m_oCursorPosition.SetDisplayedCursorIndex(static_cast<Candera::UInt32>(m_u16NewCursorPosition));
      Candera::UInt32 u16offset = cou16ComputeWidth(m_sDisplayedText, 0, static_cast<Candera::UInt32>(m_u16NewCursorPosition));
      if (NULL != m_pCursorNode)
      {
         m_pCursorNode->SetPosition(m_vEntryTextNodePos + Candera::Vector2(static_cast<Candera::Float>(u16offset), 0.0f));
      }
      m_bInvalidateRequired = true;
      triggerUpdate();
   }
}


/****************************************************************************
*     Function    : cosGetEntryTextReceived
*     Description : Interface for Accesing EntryTextReceived (MatchTextMode)
*     Parameters  : Candera::String
*     Return      : void
****************************************************************************/
Candera::String EditFieldWidget2D::cosGetEntryTextReceived() const
{
   return m_sEntryTextReceived;
}


/****************************************************************************
*     Function    : vSetEntryTextReceived
*     Description : Interface for setting EntryTextReceived (MatchTextMode)
*     Parameters  : Candera::String
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetEntryTextReceived(Candera::String sEntryTextReceived)
{
   m_sEntryTextReceived = sEntryTextReceived;
   Candera::UInt32 u8Length;
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_sEntryTextReceived);
   u8Length = u8UTFLength(m_sEntryTextReceived.GetCString());
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%10s] - In vSetEntryTextReceived \
                     with EntryTextReceived as [%s].", GetLegacyName(), sEntryTextReceived.GetCString()));
   SECURE_FEATSTD_STRING_ACCESS_END();
   //Candera::UInt8 u8Length = (Candera::UInt8)StringPlatform::Length(m_sEntryTextReceived.GetCString());
   //If received text length is <= maximum allowable, then only process the string
   if (m_pEntryTextNode && (MatchTextMode == m_enWidgetMode) && (u8Length <= m_u8MaxTextLength))
   {
      vUpdateEntryText();
      m_bUpdateSuggestionText = true;
   }
}


/****************************************************************************
*     Function    : cosGetSuggestionText
*     Description : Interface for Accesing SuggestionText (MatchTextMode)
*     Parameters  : Candera::String
*     Return      : void
****************************************************************************/
Candera::String EditFieldWidget2D::cosGetSuggestionText() const
{
   return m_sSuggestionText;
}


/****************************************************************************
*     Function    : vSetSuggestionText
*     Description : Interface for setting SuggestionText (MatchTextMode)
*     Parameters  : Candera::String
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetSuggestionText(Candera::String sSuggestionText)
{
   m_sSuggestionText = sSuggestionText;

   SECURE_FEATSTD_STRING_ACCESS_BEGIN(sSuggestionText);
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%10s] - In vSetSuggestionText \
                     with SuggestionText as [%s].", GetLegacyName(), sSuggestionText.GetCString()));
   SECURE_FEATSTD_STRING_ACCESS_END();
   //If length of EntryText Displayed < maximim allowable then only process the string
   if ((m_bUpdateSuggestionText) &&
         (m_pMatchTextNode) &&
         (MatchTextMode == m_enWidgetMode) &&
         (m_u8CurStrLength < m_u8MaxTextLength)
      )
   {
      vUpdateMatchText();
      m_bUpdateSuggestionText = false;
   }
}


/****************************************************************************
*     Function    : cosGetKeyReceived
*     Description : Interface for Accesing Key received
*     Parameters  : Candera::UInt16
*     Return      : void
****************************************************************************/
Candera::String EditFieldWidget2D::cosGetKeyReceived() const
{
   return m_sKeyReceived;
}


/****************************************************************************
*     Function    : vSetKeyReceived
*     Description : Interface for setting Key received
*     Parameters  : Candera::UInt16
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetKeyReceived(Candera::String sReturnCharacter)
{
   m_sKeyReceived = sReturnCharacter;

   SECURE_FEATSTD_STRING_ACCESS_BEGIN(sReturnCharacter);
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%10s] - In vSetKeyReceived \
                     with KeyReceived as [%s].", GetLegacyName(), sReturnCharacter.GetCString()));
   SECURE_FEATSTD_STRING_ACCESS_END();
   /* if EditField is in enabled state,
   and received string is non-empty
   and EditField is in FreeTextMode */
   if ((m_bEditFieldEnabled) &&
         (false == sReturnCharacter.IsEmpty()) &&
         (FreeTextMode == m_enWidgetMode)
      )
   {
      if (sReturnCharacter == DELETE_CHAR)
      {
         vDeleteChar();
      }
      else if (sReturnCharacter == DELETEALL_CHAR)
      {
         vDeleteAll();
      }
      else if (sReturnCharacter == SHIFT_CURSOR_LEFT)
      {
         m_mixedCharHandlingRequired = true;
         vShiftCursorLeft();
      }
      else if (sReturnCharacter == SHIFT_CURSOR_RIGHT)
      {
         m_mixedCharHandlingRequired = true;
         vShiftCursorRight();
      }
      else
      {
         // insert received string at cursor position character by character
         UInt32 u8Length;
         UInt32 u8Counter = 0;
         SECURE_FEATSTD_STRING_ACCESS_BEGIN(sReturnCharacter);
         u8Length = u8UTFLength(sReturnCharacter.GetCString());
         SECURE_FEATSTD_STRING_ACCESS_END();
         //u8Length =  (Candera::UInt8)StringPlatform::Length(sReturnCharacter.GetCString());
         if ((1 == u8Length) && (m_u8CurStrLength < m_u8MaxTextLength))
         {
            if ((false == cobRestrictShiftOperations())
                  || (true == cobRestrictShiftOperations() && (m_oCursorPosition.GetCursorIndex() < m_uHighlightStartIndex + m_uHighlightCount)))
            {
               m_u8CurStrLength++;
               vInsertChar();
            }
         }
         /* Insert char one by one,
         while Current String length is less than max text length allowed,
         and all the characters in the received string have been proccesed */
         else
         {
            if (true == cobGetMultiLineLayouting())
            {
               m_bUpdateInstructionText = false;
               m_sEditFieldInfo = sReturnCharacter;
               m_u8CurStrLength = u8Length;
               m_oCursorPosition.SetCursorIndex(m_u8CurStrLength - 1);
               m_oCursorPosition.SetDisplayedCursorIndex(m_u8CurStrLength - 1);
               vCreateMaskedText();
               vUpdateIndexOnInsertMultiLine();
               bPostEditFieldUpdMsg();
               PropertyModified("EditFieldInfo");
               return;
            }
            while ((m_u8CurStrLength < m_u8MaxTextLength) && (u8Length != u8Counter))
            {
               /* Create a variable to store char to be inserted,
               and allocate max possible bytes it can take */
               Candera::TChar* pCharBeingProcesed = CANDERA_NEW_ARRAY(Char, MAX_BYTE_FOR_A_CHARACTER);

               if (NULL != pCharBeingProcesed)
               {
                  //initialize the temp string to empty string
                  StringPlatform::Copy(pCharBeingProcesed, "");
                  m_u8CurStrLength++;
                  /* Copy a char from string received into pCharBeingProcesed for inserting,
                  and perform insertion operation */
                  uCopyUTF8Data(pCharBeingProcesed, m_u8CurStrLength, 0, sReturnCharacter, u8Counter++, 1);
                  m_sKeyReceived = pCharBeingProcesed;
                  vInsertChar();
                  //Release memory allocated from heap
                  FEATSTD_SAFE_DELETE_ARRAY(pCharBeingProcesed);
               }
               else
               {
                  //memory could not be allocated for char
                  ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], vSetKeyReceived() \
                                         - Memory could not be allocated for character !", GetLegacyName()));
                  break;
               }
            }
         }
         bPostEditFieldUpdMsg();
         PropertyModified("EditFieldInfo");
      }
   }
}


/****************************************************************************
*     Function    : cosGetTruncationText
*     Description : Interface for Accesing the truncation text
*     Parameters  : void
*     Return      : Candera::String
****************************************************************************/
Candera::String EditFieldWidget2D::cosGetTruncationText() const
{
   return m_sElipsisText;
}


/****************************************************************************
*     Function    : cosSetTruncationText
*     Description : Interface for setting the truncation text
*     Parameters  : Candera::String
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetTruncationText(Candera::String sTruncationText)
{
   if (sTruncationText != m_sElipsisText)
   {
      m_sElipsisText = sTruncationText;
      //  vUpdateVisibleText();

      PropertyModified("TruncationText");
   }
}


/****************************************************************************
*     Function    : SetEnableStatus
*     Description : Function for setting Enable status for the Widget
*     Parameters  : const bool
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::OnChanged(FeatStd::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   if ((propertyId == EnabledPropertyId) && (m_bEditFieldEnabled != IsEnabled()))
   {
      m_bEditFieldEnabled = IsEnabled();
      m_enEFState = (m_bEditFieldEnabled ? enEFNormal : enEFDisabled);
      m_bStateChanged = true;
   }
}


/****************************************************************************
*     Function    : cobMatchMode
*     Description : Function for performing visibility test for MatchTextMode
*     Parameters  : void
*     Return      : void
****************************************************************************/
bool EditFieldWidget2D::cobMatchMode() const
{
   if (MatchTextMode == m_enWidgetMode)
   {
      return true;
   }
   else
   {
      return false;
   }
}


//***************************************************************************
//TODO: the datatype 'Ucs2Char' should be used instead of 'wchar_t'
bool EditFieldWidget2D::IsArabicChar(wchar_t chr) const
{
   return (chr >= 0x0600 && chr <= 0x06ff) || (chr >= 0x0750 && chr <= 0x077f);
}


Candera::UInt32 EditFieldWidget2D::calculateCharCount(Candera::TChar* pCurrChar, Candera::TChar* pPrevChar, bool isRightToLeft)
{
   Candera::UInt32 index = m_oCursorPosition.GetDisplayedCursorIndex();
   Candera::UInt32 u8CharCount = 0;

   if (isRightToLeft != (IsArabicChar(static_cast<wchar_t>(FeatStd::Internal::TextEncoding::Ucs4(pCurrChar)))) &&
         isRightToLeft != (IsArabicChar(static_cast<wchar_t>(FeatStd::Internal::TextEncoding::Ucs4(pPrevChar)))))
   {
      Candera::UInt32 u8TempIndex = index;
      Candera::TChar* pParsedChar = NULL;
      pParsedChar = CANDERA_NEW_ARRAY(Char, MAX_BYTE_FOR_A_CHARACTER + 1);
      if (0 == *pParsedChar)
      {
         FEATSTD_SAFE_DELETE_ARRAY(pParsedChar);
         return 0;
      }

      while (u8TempIndex <= m_sDisplayedText.GetCharCount())
      {
         //initialize the temp string to empty string
         StringPlatform::Copy(pParsedChar, "");
         uCopyUTF8Data(pParsedChar, MAX_BYTE_FOR_A_CHARACTER, 0, m_sDisplayedText, u8TempIndex, 1);

         //parse each char from thereon and check isRightToLeft is true
         //if English, then increment the counter
         //isRightToLeft is false
         //if Arabic, then increment the counter
         if ((0 != (*pParsedChar)) && (isRightToLeft != (IsArabicChar(static_cast<wchar_t>(FeatStd::Internal::TextEncoding::Ucs4(pParsedChar))))))
         {
            u8CharCount++;
         }
         else
         {
            //else break the loop
            break;
         }
         u8TempIndex++;
      }

      FEATSTD_SAFE_DELETE_ARRAY(pParsedChar);
   }
   return u8CharCount;
}


Candera::UInt32 EditFieldWidget2D::calculateCursorOffset(bool isRightToLeft, Candera::UInt32 currCharPos_Actual)
{
   Candera::UInt32 offset = 0;
   Candera::UInt32 charCount = 0;
   Candera::UInt32 index = m_oCursorPosition.GetDisplayedCursorIndex();
   Candera::TChar* pCurrChar = NULL;
   pCurrChar = CANDERA_NEW_ARRAY(Char, MAX_BYTE_FOR_A_CHARACTER + 1);
   if (NULL != pCurrChar)
   {
      //initialize the temp string to empty string
      StringPlatform::Copy(pCurrChar, "");
      uCopyUTF8Data(pCurrChar, MAX_BYTE_FOR_A_CHARACTER, 0, m_sDisplayedText, index, 1);
   }

   Candera::TChar* pPrevChar = NULL;
   pPrevChar = CANDERA_NEW_ARRAY(Char, MAX_BYTE_FOR_A_CHARACTER + 1);
   if (NULL != pPrevChar)
   {
      //initialize the temp string to empty string
      StringPlatform::Copy(pPrevChar, "");
      uCopyUTF8Data(pPrevChar, MAX_BYTE_FOR_A_CHARACTER, 0, m_sDisplayedText, index - 1, 1);
   }
   if ((NULL == pCurrChar) || (NULL == pPrevChar))
   {
      FEATSTD_SAFE_DELETE_ARRAY(pCurrChar);
      FEATSTD_SAFE_DELETE_ARRAY(pPrevChar);
      return offset;
   }
   charCount = calculateCharCount(pCurrChar, pPrevChar, isRightToLeft);

   Candera::UInt32 cursorIndex = m_oCursorPosition.GetDisplayedCursorIndex();
   Candera::UInt32 currCharWidthVal = cou16ComputeWidth(m_sDisplayedText, cursorIndex, charCount);
   offset = cou16ComputeWidth(m_sDisplayedText, 0, currCharPos_Actual) + currCharWidthVal;

   FEATSTD_SAFE_DELETE_ARRAY(pCurrChar);
   FEATSTD_SAFE_DELETE_ARRAY(pPrevChar);

   m_mixedCharHandlingRequired = false;
   return offset;
}


/****************************************************************************
*     Function    : vUpdateCursorPos
*     Description : Function for Updating Cursor Position
*     Parameters  : void
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vUpdateCursorPos()
{
   if (NULL != m_pCursorNode)
   {
      Candera::UInt32 u8CurrCharPos_Virtual = 0;
      Candera::UInt32 u8CurrCharPos_Actual = 0;
      bool isRightToLeft = false;
      SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_sDisplayedText);
      FeatStd::Internal::Utf8TextIterator iter_DisplayedText = m_sDisplayedText.GetCString();

      Candera::UInt32 index = m_oCursorPosition.GetDisplayedCursorIndex();
      if (!m_cultureCurrent.PointsToNull() && m_cultureCurrent->GetTextDirection() == Candera::Globalization::RightToLeft)
      {
         isRightToLeft = true;
      }
      bool isLastCharArabic = false;
      while (index > 0)
      {
         TChar achCurrChar[4];
         FeatStd::Internal::TextEncoding::Copy(achCurrChar, 4, iter_DisplayedText.Pos(), 1/*read only one UTF-8 character*/);

         u8CurrCharPos_Virtual++;
         if (isRightToLeft == IsArabicChar(static_cast<wchar_t>(FeatStd::Internal::TextEncoding::Ucs4(achCurrChar))))
         {
            u8CurrCharPos_Actual = u8CurrCharPos_Virtual;
            isLastCharArabic = true;
         }
         else
         {
            if (isLastCharArabic && (FeatStd::Internal::TextEncoding::Ucs4(achCurrChar) == 0x20))
            {
               u8CurrCharPos_Actual = u8CurrCharPos_Virtual;
            }
            else
            {
               isLastCharArabic = false;
            }
         }

         if (!iter_DisplayedText.EndOfString())
         {
            iter_DisplayedText.Advance();
         }
         index--;
      }
      SECURE_FEATSTD_STRING_ACCESS_END();

      m_u16NewCursorPosition = u8CurrCharPos_Actual;
      //Compute width of char starting from visible start index till character at index m_u8CursorIndex-1
      //u16offset = cou16ComputeWidth(m_sDisplayedText, 0, m_oCursorPosition.GetDisplayedCursorIndex());
      Candera::UInt32 u32offset = 0;
      if (true == m_mixedCharHandlingRequired && (m_oCursorPosition.GetDisplayedCursorIndex() > 0))
      {
         u32offset = calculateCursorOffset(isRightToLeft, u8CurrCharPos_Actual);
      }
      else
      {
         u32offset = cou16ComputeWidth(m_sDisplayedText, 0, u8CurrCharPos_Actual);
      }

      if (isRightToLeft)
      {
         m_pCursorNode->SetPosition(static_cast<Candera::Float>((cou16GetBgWidth() - cou8GetTextAreaMarginX()) - u32offset), static_cast<Candera::Float>(m_u8TextAreaMarginY));
      }
      else
      {
         m_pCursorNode->SetPosition(m_vEntryTextNodePos + Candera::Vector2(static_cast<Candera::Float>(u32offset), static_cast<Candera::Float>(0)));
      }
      m_bInvalidateRequired = true;
      //invalidate_IfWakeUpRenderMechanismIsEnabled();
      triggerUpdate();
   }
}


/****************************************************************************
*     Function    : vUpdateVisibleTextMultiLine
*     Description : Function for updating visible text for multi line text
*     Parameters  : void
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vUpdateVisibleTextMultiLine()
{
   Candera::TChar* pVisibleText = CANDERA_NEW_ARRAY(Char, (m_u8CurStrLength * MAX_BYTE_FOR_A_CHARACTER) + 1);
   if (NULL != pVisibleText)
   {
      //initialize the temp string to empty string
      StringPlatform::Copy(pVisibleText, "");
      uCopyUTF8Data(pVisibleText, m_u8CurStrLength, 0, m_sMaskedText, 0, m_u8CurStrLength);
      m_sDisplayedText = pVisibleText;

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: m_sDisplayedText = [%s] in vUpdateVisibleTextMultiLine", m_sDisplayedText.GetCString()));
      m_oEntryTextWidget.SetText(m_sDisplayedText);
      m_oEntryTextWidget.Update();

      m_bInvalidateRequired = true;

      triggerUpdate();
      FEATSTD_SAFE_DELETE_ARRAY(pVisibleText);
   }
}


/****************************************************************************
*     Function    : vUpdateVisibleText
*     Description : Function for updating visible text
*     Parameters  : void
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vUpdateVisibleText()
{
   Candera::TChar* pVisibleText = CANDERA_NEW_ARRAY(Char, (m_u8CurStrLength * MAX_BYTE_FOR_A_CHARACTER) + 1);
   if (NULL != pVisibleText)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], vUpdateVisibleText() - pVisibleText not NULL !", GetLegacyName()));

      //initialize the temp string to empty string
      StringPlatform::Copy(pVisibleText, "");
      uCopyUTF8Data(pVisibleText, m_u8CurStrLength, 0, m_sMaskedText, m_u8VisibleStartIndex, m_u8VisibleCharCount);

      vAddElipsisText(pVisibleText);

      if (((NULL == m_pHighlightTextNode) || (NULL == m_pEndTextNode)) || (u8UTFLength(m_sEditFieldInfo.GetCString()) <= m_uHighlightStartIndex) || (m_uHighlightCount <= 0)) //TODO : Improve this condition
      {
         m_oEntryTextWidget.SetText(m_sDisplayedText);
         m_oEntryTextWidget.Update();
         vResetHighlightAndEndText();
      }
      else
      {
         m_oEntryTextWidget.SetText("");
         m_oEntryTextWidget.Update();
      }
      //m_oEntryTextWidget.Invalidate();
      if ((NULL != m_pHighlightTextNode) && (NULL != m_pEndTextNode))
      {
         if ((m_uHighlightCount > 0) /*&& (u8UTFLength(m_sEditFieldInfo.GetCString()) > m_uHighlightStartIndex)*/)
         {
            //m_oHighlightTextWidget.SetText(m_pLeftText);
            //m_oHighlightTextWidget.Update();
            vProcessVisibleTextAndCharCount();
            vUpdateHighlightAndEndText();
         }
         else
         {
            //TODO :: Improve this condition
            vResetHighlightAndEndText();
         }
      }
      //SUZUKI - 13572 - Start
      m_bInvalidateRequired = true;
      // vCheckAndInvalidate();
      //SUZUKI - 13572 - End
      FEATSTD_SAFE_DELETE_ARRAY(pVisibleText);
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], vUpdateVisibleText() - visible text updated successfully!", GetLegacyName()));
   }
}


/****************************************************************************
*     Function    : vCreateMaskedText
*     Description : Function for creating text string if a mask character is specified
*     Parameters  : void
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vCreateMaskedText()
{
   //If no mask char specified, no masking
   if (m_sMaskChar.IsEmpty())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], vCreateMaskedText() - No masking!", GetLegacyName()));
      SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_sEditFieldInfo);
      m_sMaskedText = m_sEditFieldInfo.GetCString();
      SECURE_FEATSTD_STRING_ACCESS_END();
   }
   //If mask char specified, perform masking
   else
   {
      Candera::TChar* pTempStr = CANDERA_NEW_ARRAY(Char, (m_u8CurStrLength + 1));
      if (NULL != pTempStr)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], vCreateMaskedText() - Masking!", GetLegacyName()));
         //initialize the temp string to empty string
         StringPlatform::Copy(pTempStr, "");
         Candera::TChar cMaskChar;
         SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_sMaskChar);
         cMaskChar = (m_sMaskChar.GetCString())[0];
         SECURE_FEATSTD_STRING_ACCESS_END();
         Candera::UInt32 u8Itr;
         for (u8Itr = 0 ; u8Itr < m_u8CurStrLength ; u8Itr++)
         {
            pTempStr[u8Itr] = cMaskChar;
         }
         pTempStr[m_u8CurStrLength] = '\0';
         m_sMaskedText = pTempStr;

         FEATSTD_SAFE_DELETE_ARRAY(pTempStr);
      }
   }
}


/****************************************************************************
*     Function    : vInsertChar
*     Description : Function for Inserting a character (FreeTextMode)
*     Parameters  : void
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vInsertChar()
{
   //Candera::TChar* pEditFieldText = new Char [(m_u8CurStrLength * MAX_BYTE_FOR_A_CHARACTER) + 1];		//This has been removed for NCG3D-72296
   memset(m_pEditFieldText, 0, (EF_MAX_TEXT_LENGTH * MAX_BYTE_FOR_A_CHARACTER) + 1);

   // todo : fr83hi : check next statement
   m_bUpdateInstructionText = false;//added for use-case required by suzuki china variant

   //if (m_pEditFieldText != NULL)
   //{
   //initialize the temp string to empty string
   StringPlatform::Copy(m_pEditFieldText, "");
//      //create a iterator for the string text content
//      FeatStd::Internal::Utf8TextIterator iter_EditFieldText(pEditFieldText);
//      FeatStd::Internal::Utf8TextIterator iter_EditFieldInfo(m_sEditFieldInfo.GetCString());

   // Copy Characters before the cursor index to temparary string(pEditFieldText)
   uCopyUTF8Data(m_pEditFieldText, m_u8CurStrLength, 0, m_sEditFieldInfo, 0, m_oCursorPosition.GetCursorIndex());
   // Copy the Entered Key Chracter to the End of the temparary string(pEditFieldText)
   uCopyUTF8Data(m_pEditFieldText, m_u8CurStrLength, m_oCursorPosition.GetCursorIndex(), m_sKeyReceived, 0, 1);

   uCopyUTF8Data(m_pEditFieldText, m_u8CurStrLength, m_oCursorPosition.GetCursorIndex() + 1 ,
                 m_sEditFieldInfo, m_oCursorPosition.GetCursorIndex(), static_cast<UInt32>(m_sEditFieldInfo.GetCodePointCount() - m_oCursorPosition.GetCursorIndex()));

   m_sEditFieldInfo = m_pEditFieldText;
   //PropertyModified("EditFieldInfo");

   if (false == cobGetMultiLineLayouting())
   {
      //Process for Kanji
      if ((NULL != m_pHighlightTextNode) && (NULL != m_pEndTextNode))
      {
         if ((u8UTFLength(m_sEditFieldInfo.GetCString()) > m_uHighlightStartIndex) /*&& (m_uHighlightStartIndex >= 0) */ && (m_uHighlightCount > 0))
         {
            // THis block required if highlight index and count needs to be changed if the insertion of chars is done in between the left or highlight text
            /*//if Entered character is in Left, Middle or End Part of total text
            if (m_oCursorPosition.GetCursorIndex() < u8UTFLength(m_sEditFieldInfo.GetCString()))
            {
            if ((m_oCursorPosition.GetCursorIndex() < m_uHighlightStartIndex)
            && (m_oCursorPosition.GetCursorIndex() < (u8UTFLength(m_pLeftText))))
            {
            //Increase highlight index by one so index is new one to take input char
            m_uHighlightStartIndex++;
            }
            else if ((m_oCursorPosition.GetCursorIndex() > m_uHighlightStartIndex)
            && (m_oCursorPosition.GetCursorIndex() < (m_uHighlightStartIndex + m_uHighlightCount))
            && (m_oCursorPosition.GetCursorIndex() < (u8UTFLength(m_pHighlightText) + u8UTFLength(m_pLeftText))))
            {
            //Increase highlight count by one so total hightlight charaters can be increased
            if (u8UTFLength(m_pHighlightText) >= m_uHighlightCount)      // This is added if insertion of character in middle of highlight text is done before total highlight count is completed/typed
            {
            m_uHighlightCount++;
            }
            //TODO:: Need to confirm below requirement
            // What if highlight index and count is more than 0 before even single character is typed by key
            // or even first time if count is set to some value while inserting characters and count should not be increased
            }
            else{}// Todo : check above requirement and update this condition
            }*/
            Candera::UInt32 totalStringLen = u8UTFLength(m_sEditFieldInfo.GetCString());  // TODO :: Check as same to m_u8CurStrLength
            Candera::UInt32 endTextLen = totalStringLen - (m_uHighlightStartIndex + m_uHighlightCount);
            memset(m_pLeftText, 0, (EF_MAX_TEXT_LENGTH * MAX_BYTE_FOR_A_CHARACTER) + 1);
            memset(m_pHighlightText, 0, (EF_MAX_TEXT_LENGTH * MAX_BYTE_FOR_A_CHARACTER) + 1);
            memset(m_pEndText, 0, (EF_MAX_TEXT_LENGTH * MAX_BYTE_FOR_A_CHARACTER) + 1);

            //Prepare Left Text
            uCopyUTF8Data(m_pLeftText, m_uHighlightStartIndex, 0, m_sEditFieldInfo, 0, m_uHighlightStartIndex);

            //Prepare Highlight Text
            uCopyUTF8Data(m_pHighlightText, m_uHighlightCount, 0, m_sEditFieldInfo, m_uHighlightStartIndex, m_uHighlightCount);

            //Prepare End Text
            uCopyUTF8Data(m_pEndText, endTextLen, 0, m_sEditFieldInfo, m_uHighlightStartIndex + m_uHighlightCount, endTextLen);
         }
         else
         {
            // TODO : check to improve this condition
            //Required if data binding highlight text and count updated before entry of any text and hightlightIndex in greater than 0
            // Need to check requirement how can it be incorporated
            memset(m_pHighlightText, 0, (EF_MAX_TEXT_LENGTH * MAX_BYTE_FOR_A_CHARACTER) + 1);
            memset(m_pEndText, 0, (EF_MAX_TEXT_LENGTH * MAX_BYTE_FOR_A_CHARACTER) + 1);
            memset(m_pLeftText, 0, (EF_MAX_TEXT_LENGTH * MAX_BYTE_FOR_A_CHARACTER) + 1);
            strcpy(m_pLeftText, m_pEditFieldText);
         }
      }
      vCreateMaskedText();
      vUpdateIndexOnInsert();
   }
   else
   {
      vCreateMaskedText();
      vUpdateIndexOnInsertMultiLine();
   }
   //This has been removed for NCG3D-72296
   // FEATSTD_SAFE_DELETE_ARRAY(pEditFieldText);
   // }
   // else
   // {
   // ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], vInsertChar() UNSUCCESSFULL - pEditFieldText is NULL !", GetLegacyName()));
   // m_u8CurStrLength--;
   // }
   //TODO
   if (EF_MIN_TEXT_LENGTH == m_u8CurStrLength)
   {
      m_bUpdateInstructionText = true;
   }
}


/****************************************************************************
*     Function    : vDeleteChar
*     Description : Function for Deleting a Character (FreeTextMode)
*     Parameters  : void
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vDeleteChar()
{
   // Validating if there are any characters before the cursor to be deleted
   if (EF_MIN_INDEX != m_oCursorPosition.GetCursorIndex())
   {
      m_u8CurStrLength--;
      Candera::TChar* pEditFieldText = CANDERA_NEW_ARRAY(Char, (m_u8CurStrLength * MAX_BYTE_FOR_A_CHARACTER) + 1);

      if (NULL != pEditFieldText)
      {
         //initialize the temp string to empty string
         StringPlatform::Copy(pEditFieldText, "");
         uCopyUTF8Data(pEditFieldText, m_u8CurStrLength, 0, m_sEditFieldInfo, 0, m_oCursorPosition.GetCursorIndex() - 1);

         uCopyUTF8Data(pEditFieldText, m_u8CurStrLength, m_oCursorPosition.GetCursorIndex() - 1,
                       m_sEditFieldInfo, m_oCursorPosition.GetCursorIndex(), static_cast<UInt32>(m_sMaskedText.GetCodePointCount() - m_oCursorPosition.GetCursorIndex()));

         m_sEditFieldInfo = pEditFieldText;

         //bPostEditFieldUpdMsg();
         //PropertyModified("EditFieldInfo");

         //Kanji Req
         vUpdateAllTextOnDeleteChar();

         vCreateMaskedText();
         if (false == cobGetMultiLineLayouting())
         {
            vUpdateIndexOnDelete();
         }
         else
         {
            vUpdateIndexOnDeleteMultiLine();
         }

         bPostEditFieldUpdMsg();
         PropertyModified("EditFieldInfo");

         FEATSTD_SAFE_DELETE_ARRAY(pEditFieldText);
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], Deletion of Char in vDeleteChar() SUCCESSFULL!", GetLegacyName()));
      }
      else
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], vDeleteChar() UNSUCCESSFULL - pEditFieldText is NULL !", GetLegacyName()));
         m_u8CurStrLength++;
      }
      if (EF_MIN_TEXT_LENGTH == m_u8CurStrLength)
      {
         m_bUpdateInstructionText = true;
      }
   }
}


/****************************************************************************
*     Function    : vDeleteAll
*     Description : Function for Deleting Entire text string (FreeTextMode)
*     Parameters  : void
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vDeleteAll()
{
   //if Current string length is zero no need to perform this operation
   if (EF_MIN_TEXT_LENGTH != m_u8CurStrLength)
   {
      m_u8CurStrLength = EF_MIN_TEXT_LENGTH;
      m_oCursorPosition = CursorPosition(EF_MIN_INDEX);
      m_u8VisibleStartIndex = EF_MIN_INDEX;
      m_u8VisibleCharCount = EF_MIN_TEXT_LENGTH;

      m_bUpdateInstructionText = true;
      m_sEditFieldInfo = NULL;

      bPostEditFieldUpdMsg();
      PropertyModified("EditFieldInfo");

      m_sMaskedText = NULL;

      //Kanji Req :: TODO check Requirement, it highlight info needs to be cleared on delete all
      /*m_uHighlightStartIndex = 0;
      m_uHighlightCount = 0;*/

      //multiLine Edifield variables
      m_uCurrentLine = 1;
      m_uNumberOfLines = 1;
      m_u16TextRectHeight = 0;
      m_charCountEachLine.clear();
      vProcessCursorPosition();
      /*Update Index need not be called as all the indexes are known.
      Since sMaskedText will be empty no need to Call vCreateMaskedText()
      Update visibleText and CursorPosition
      */
      vUpdateVisibleText();
      vUpdateCursorPos();
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], Deletion of Entire String in vDeleteAll() SUCCESSFULL!", GetLegacyName()));
   }
}


/****************************************************************************
*     Function    : vShortHoldDelete
*     Description : Function for Deleting all characters to left of Cursor
*     Parameters  : void
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vShortHoldDelete()
{
   //If Cursor at beginning of EditField text String
   if (EF_MIN_INDEX != m_oCursorPosition.GetCursorIndex())
   {
      //Otherwise Cursor not at the beginning of EditField text String
      //New String length = Old str length - nos of Char(visible & invisible) to left of cursor
      m_u8CurStrLength -= m_oCursorPosition.GetCursorIndex();

      Candera::TChar* pEditFieldText = CANDERA_NEW_ARRAY(Char, ((m_u8CurStrLength * MAX_BYTE_FOR_A_CHARACTER) + 1));
      if (NULL != pEditFieldText)
      {
         //initialize the temp string to empty string
         StringPlatform::Copy(pEditFieldText, "");
         SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_sEditFieldInfo);
         uCopyUTF8Data(pEditFieldText, m_u8CurStrLength, 0, m_sEditFieldInfo.GetCString(), m_oCursorPosition.GetCursorIndex(), m_u8CurStrLength);
         //StringPlatform::Copy((Char*)pEditFieldText, (m_sEditFieldInfo.GetCString() + m_u8CursorIndex));
         SECURE_FEATSTD_STRING_ACCESS_END();
         //pEditFieldText[m_u8CurStrLength] = '\0';
         m_sEditFieldInfo = pEditFieldText;

         bPostEditFieldUpdMsg();
         PropertyModified("EditFieldInfo");

         vCreateMaskedText();

         m_oCursorPosition = CursorPosition(EF_MIN_INDEX);
         //First character will be the visible character
         m_u8VisibleStartIndex = EF_MIN_INDEX;
         //Compute no. of characters visible starting from index 0
         m_u8VisibleCharCount = m_u8CurStrLength;
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], vShortHoldDelete() before while loop !", GetLegacyName()));
         while (m_u8VisibleCharCount > EF_MIN_TEXT_LENGTH
                && (cou16ComputeWidth(m_sMaskedText, m_u8VisibleStartIndex, m_u8VisibleCharCount) >= m_u16TextAreaWidth))
         {
            m_u8VisibleCharCount--;
         }
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], vShortHoldDelete() after while loop !", GetLegacyName()));
         if (false == cobGetMultiLineLayouting())
         {
            vUpdateVisibleText();
            vUpdateCursorPos();
         }
         else
         {
            vUpdateVisibleTextMultiLine();
         }

         FEATSTD_SAFE_DELETE_ARRAY(pEditFieldText);
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], Deletion of string to left of Cursor successfull in vShortHoldDelete() !", GetLegacyName()));
      }
      else
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], vShortHoldDelete() - pEditFieldText is NULL !", GetLegacyName()));
         m_u8CurStrLength += m_oCursorPosition.GetCursorIndex();
      }
      if (EF_MIN_TEXT_LENGTH == m_u8CurStrLength)
      {
         m_bUpdateInstructionText = true;
      }
   }
}


void EditFieldWidget2D::vUpdateIndexOnInsertMultiLine()
{
   if (m_uNumberOfLines == 1)
   {
      m_u8VisibleStartIndex = EF_MIN_INDEX;
      m_u8VisibleCharCount = m_u8CurStrLength;
   }
   else
   {
      m_u8VisibleCharCount++;
   }
   Candera::TChar* pVisibleText = CANDERA_NEW_ARRAY(Char, (m_u8CurStrLength * MAX_BYTE_FOR_A_CHARACTER) + 1);
   if (NULL != pVisibleText)
   {
      StringPlatform::Copy(pVisibleText, "");
      uCopyUTF8Data(pVisibleText, m_u8CurStrLength, 0, m_sMaskedText, 0, m_u8CurStrLength);
      Candera::UInt32 textHeight = cou16ComputeHeight(pVisibleText, 0, m_u8CurStrLength);

      //Initially setting height for 1st line
      if (m_u8CurStrLength == 1)
      {
         m_u16TextRectHeight = textHeight;
      }

      if ((textHeight < m_u8TextAreaHeight) && (m_uNumberOfLines <= couGetMaximumNumberOfLines()))
      {
         //Using height to detemine line change or word wrap
         // IF height changes means new line has to be added
         if (textHeight > m_u16TextRectHeight)
         {
            m_uNumberOfLines++;
            if (m_uNumberOfLines > couGetMaximumNumberOfLines())
            {
               uCopyUTF8Data(m_pEditFieldText, m_u8CurStrLength, 0, m_sEditFieldInfo, 0, m_oCursorPosition.GetCursorIndex());

               uCopyUTF8Data(m_pEditFieldText, m_u8CurStrLength, m_oCursorPosition.GetCursorIndex(),
                             m_sEditFieldInfo, m_oCursorPosition.GetCursorIndex() + 1, static_cast<UInt32>(m_sEditFieldInfo.GetCodePointCount() - m_oCursorPosition.GetCursorIndex()));
               ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: vUpdateIndexOnInsertMultiLine() && lines are more than max -> m_pEditFieldText = %s ", m_pEditFieldText));

               m_sEditFieldInfo = m_pEditFieldText;
               vCreateMaskedText();
               --m_u8CurStrLength;
               --m_u8VisibleCharCount;
               --m_uNumberOfLines;
               return;
            }
            m_uCurrentLine++;
            m_u16TextRectHeight = textHeight;
            // No Need to store the as m_charCountEachLine will update when listener will update the data.
            //m_charCountEachLine.push_back((m_u8VisibleCharCount - 1));
         }
         else {} // do nothing
         ++m_oCursorPosition;
      }
      else
      {
         uCopyUTF8Data(m_pEditFieldText, m_u8CurStrLength, 0, m_sEditFieldInfo, 0, m_oCursorPosition.GetCursorIndex());

         uCopyUTF8Data(m_pEditFieldText, m_u8CurStrLength, m_oCursorPosition.GetCursorIndex(),
                       m_sEditFieldInfo, m_oCursorPosition.GetCursorIndex() + 1, static_cast<UInt32>(m_sEditFieldInfo.GetCodePointCount() - m_oCursorPosition.GetCursorIndex()));
         m_sEditFieldInfo = m_pEditFieldText;
         vCreateMaskedText();
         --m_u8CurStrLength;
         --m_u8VisibleCharCount;
      }
   }
   FEATSTD_SAFE_DELETE_ARRAY(pVisibleText);

   //Update VisibleText and CursorPosition will be updated when OnLayoutingInfoChanged() is called when OnMeasure() is called in TextWidget2DV2
   vUpdateVisibleTextMultiLine();
}


/****************************************************************************
*     Function    : vUpdateIndexOnInsert
*     Description : Function for Updating Indexes
*     Parameters  : void
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vUpdateIndexOnInsert()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], vUpdateIndexOnInsert() - Entered!", GetLegacyName()));
   /* UseCase : If width of EditField String is < EditFields TextArea
   i.e [abc|    ] => [abcd|   ]
   */
   if (cou16ComputeWidth(m_sMaskedText, EF_MIN_INDEX , m_u8CurStrLength) < m_u16TextAreaWidth)
   {
      m_u8VisibleStartIndex = EF_MIN_INDEX;
      m_u8VisibleCharCount = m_u8CurStrLength;
   }
   // UseCases : If width of EditField String is >= tha EditFields TextArea

   /* UseCase : If Cursor at the end of visible text
   i.e [abcdef|] => a[bcdefg|]
   */
   else if (m_oCursorPosition.GetCursorIndex() == (m_u8VisibleStartIndex + m_u8VisibleCharCount))
   {
      m_u8VisibleCharCount++;
      //Scroll text to left if previous visibleText + new char doesn`t fit in EditField TextArea
      while ((m_u8VisibleCharCount > EF_MIN_TEXT_LENGTH)
             && ((m_u32TruncationTextDisplayedWidth + cou16ComputeWidth(m_sMaskedText, m_u8VisibleStartIndex, m_u8VisibleCharCount)) >= m_u16TextAreaWidth))
      {
         m_u8VisibleStartIndex++;
         m_u8VisibleCharCount--;
      }
   }
   /*If Cursor is not at the end of visible text
   UseCase : if width of new char along with chars from visiblestartindex till cursor index doesn`t fit TextArea width
   i.e a[ibcde|i]z => ai[bcdeW|]iz
   */
   else if ((m_u32TruncationTextDisplayedWidth + cou16ComputeWidth(m_sMaskedText, m_u8VisibleStartIndex,
             (m_oCursorPosition.GetCursorIndex() + 1 - m_u8VisibleStartIndex))) >= m_u16TextAreaWidth)
   {
      /*Since the newly added char should be visible,
      Assume newly added char to be at visibleStartIndex and zero chars fit the TextAreaWidth
      & then compute recursively nos. of char that fit TextArea and scroll text to right if required
      */
      m_u8VisibleStartIndex = m_oCursorPosition.GetCursorIndex() + 1 ;
      m_u8VisibleCharCount = EF_MIN_TEXT_LENGTH;
      while ((m_u8VisibleStartIndex > EF_MIN_INDEX) &&
             (m_u8VisibleStartIndex > EF_MIN_INDEX) &&
             ((m_u32TruncationTextDisplayedWidth + cou16ComputeWidth(m_sMaskedText, (m_u8VisibleStartIndex - 1),
                   (m_u8VisibleCharCount + 1))) < m_u16TextAreaWidth)
            )
      {
         m_u8VisibleStartIndex--;
         m_u8VisibleCharCount++;
      }
   }
   /*UseCase : if width of new char along with chars from visiblestartindex till cursor index fits TextArea width
   i.e a[bc|de]z => a[bcW|d]ez
   */
   else
   {
      m_u8VisibleCharCount++;
      //Fit as much chars possible, scroll right if required
      while ((m_u8VisibleCharCount > EF_MIN_TEXT_LENGTH)
             && (m_u32TruncationTextDisplayedWidth + cou16ComputeWidth(m_sMaskedText, m_u8VisibleStartIndex, m_u8VisibleCharCount)) >= m_u16TextAreaWidth)
      {
         m_u8VisibleCharCount--;
      }
   }

   ++m_oCursorPosition;
   if (m_bHighlightDataChanged)
   {
      --m_oCursorPosition;
   }
   //Update VisibleText and CursorPosition
   vUpdateVisibleText();
   vUpdateCursorPos();
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], vUpdateIndexOnInsert() - Index Updated!", GetLegacyName()));
}


/****************************************************************************
*     Function    : vUpdateIndexOnDeleteMultiLine
*     Description : Function for Updating Indexes on Delete operation in Multiline
*     Parameters  : void
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vUpdateIndexOnDeleteMultiLine()
{
   --m_oCursorPosition;
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D:  Deletion of Char in vUpdateIndexOnDeleteMultiLine() m_oCursorPosition = %d", m_oCursorPosition.GetCursorIndex()));
   vUpdateVisibleTextMultiLine();
}


/****************************************************************************
*     Function    : vUpdateIndexOnDelete
*     Description : Function for Updating Indexes on Delete operation
*     Parameters  : void
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vUpdateIndexOnDelete()
{
   /* UseCase : If Cursor at beginning of visible text
   & there are invisible chars to left of cursor
   i.e abc[|def]xyz => ab[|def]xyz
   */
   if ((m_u8VisibleStartIndex == m_oCursorPosition.GetCursorIndex()) && (m_u8VisibleStartIndex > EF_MIN_INDEX))
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], vUpdateIndexOnDelete() - \
                         Cursor at beginning of visibleText & not at beginning of EditFieldString!", GetLegacyName()));
      m_u8VisibleStartIndex--;
      --m_oCursorPosition;

      while ((m_u8VisibleStartIndex > EF_MIN_INDEX) &&
             (cou16ComputeWidth(m_sMaskedText, m_u8VisibleStartIndex - 1, (m_u8VisibleCharCount + 1))) < m_u16TextAreaWidth
            )
      {
         m_u8VisibleStartIndex--;
         m_u8VisibleCharCount++;
      }
   }
   //If Cursor not at the beginning of visible text
   else if (m_u8VisibleCharCount > EF_MIN_TEXT_LENGTH)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], vUpdateIndexOnDelete() - Cursor not at beginning of visibleText", GetLegacyName()));
      m_u8VisibleCharCount--;
      /*Use Case :If there are invisible chars to right of EditField
      i.e xyz[acb|d]xyz => xyz[ac|dx]yz
      */
      if ((m_u8VisibleStartIndex + m_u8VisibleCharCount) < m_u8CurStrLength)
      {
         //try to fit as much chars from VisibleStartIndex as possible
         m_u8VisibleCharCount = m_u8CurStrLength - m_u8VisibleStartIndex;
         while ((m_u8VisibleCharCount > EF_MIN_TEXT_LENGTH) &&
                (cou16ComputeWidth(m_sMaskedText, m_u8VisibleStartIndex, m_u8VisibleCharCount)) >= m_u16TextAreaWidth)
         {
            m_u8VisibleCharCount--;
         }
      }
      /*Use Case : If Cursor at the end of visible text
      & there are no more invisible chars to right of cursor
      i.e xyz[acbd|] => xy[zacb|]
      */
      if ((m_u8VisibleStartIndex + m_u8VisibleCharCount) == m_u8CurStrLength)
      {
         //Shift text right until and unless it fits
         while ((m_u8VisibleStartIndex > EF_MIN_INDEX) &&
                (cou16ComputeWidth(m_sMaskedText, m_u8VisibleStartIndex - 1, (m_u8VisibleCharCount + 1))) < m_u16TextAreaWidth
               )
         {
            m_u8VisibleStartIndex--;
            m_u8VisibleCharCount++;
         }
      }
      --m_oCursorPosition;
   }
   // Update VisibleText and CursorPosition
   vUpdateVisibleText();
   vUpdateCursorPos();
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], vUpdateIndexOnDelete() - Index Updated!", GetLegacyName()));
}


/****************************************************************************
*     Function    : vShiftCursorLeft
*     Description : Function for shifting cursor on left event for FreeTextMode
*     Parameters  : void
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vShiftCursorLeft()
{
   if (true == cobGetMultiLineLayouting())
   {
      if (m_oCursorPosition.GetCursorIndex() > 0)
      {
         --m_oCursorPosition;
      }
      vProcessCursorPosition();
      bPostEditFieldUpdMsg();
      return;
   }

   //UseCase: If their are chars(visible & invisible) to the left of cursor
   if (m_oCursorPosition.GetCursorIndex() > EF_MIN_INDEX)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], vShiftCursorLeft() - (m_u8CursorIndex > 0)!", GetLegacyName()));
      // UseCase: If Cursor is in between the visible text i.e [abcd|ef]
      if (m_oCursorPosition.GetCursorIndex() > m_u8VisibleStartIndex)
      {
         if ((false == cobRestrictShiftOperations())
               || ((true == cobRestrictShiftOperations()) && (m_oCursorPosition.GetCursorIndex() > m_uHighlightStartIndex) && (m_oCursorPosition.GetCursorIndex() <= m_uHighlightStartIndex + m_uHighlightCount)))
         {
            if (m_oCursorPosition.GetDisplayedCursorIndex() > 0)
            {
               --m_oCursorPosition;
            }
            else if (m_oCursorPosition.GetCursorIndex() > 0)
            {
               m_oCursorPosition.SetCursorIndex(m_oCursorPosition.GetCursorIndex() - 1);
            }
         }
      }
      // UseCase: If Cursor is at beginning of visible text & their are invisible chars to left of EditField i.e abc[|defgh]
      else if (m_oCursorPosition.GetCursorIndex() == m_u8VisibleStartIndex)
      {
         while (m_u8VisibleStartIndex > EF_MIN_INDEX
                && (cou16ComputeWidth(m_sMaskedText, m_u8VisibleStartIndex, m_u8VisibleCharCount) <= m_u16TextAreaWidth))
         {
            m_u8VisibleStartIndex--;
            m_u8VisibleCharCount++;
         }

         SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_sMaskedText)
         while (m_u8VisibleCharCount < StringPlatform::TLength(m_sMaskedText.GetCString())
                && (m_u32TruncationTextDisplayedWidth + cou16ComputeWidth(m_sMaskedText, m_u8VisibleStartIndex, m_u8VisibleCharCount)) <= m_u16TextAreaWidth)
         {
            m_u8VisibleCharCount++;
         }
         SECURE_FEATSTD_STRING_ACCESS_END();

         while ((m_u8VisibleCharCount > EF_MIN_TEXT_LENGTH)
                && ((m_u32TruncationTextDisplayedWidth + cou16ComputeWidth(m_sMaskedText, m_u8VisibleStartIndex, m_u8VisibleCharCount)) >= m_u16TextAreaWidth))
         {
            m_u8VisibleCharCount--;
         }
         if ((false == cobRestrictShiftOperations())
               || ((true == cobRestrictShiftOperations()) && (m_oCursorPosition.GetCursorIndex() > m_uHighlightStartIndex) && (m_oCursorPosition.GetCursorIndex() <= m_uHighlightStartIndex + m_uHighlightCount)))
         {
            if (m_oCursorPosition.GetDisplayedCursorIndex() > 0)
            {
               --m_oCursorPosition;
            }
            else if (m_oCursorPosition.GetCursorIndex() > 0)
            {
               m_oCursorPosition.SetCursorIndex(m_oCursorPosition.GetCursorIndex() - 1);
            }
         }
         vUpdateVisibleText();
      }
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], vShiftCursorLeft() - Index updated!", GetLegacyName()));
      vUpdateCursorPos();
      bPostEditFieldUpdMsg();
   }
}


/****************************************************************************
*     Function    : vShiftCursorRight
*     Description : Function for shifting cursor on right event for FreeTextMode
*     Parameters  : void
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vShiftCursorRight()
{
   if (true == cobGetMultiLineLayouting())
   {
      if (/*(m_oCursorPosition.GetCursorIndex() >= 0) &&*/ (m_oCursorPosition.GetCursorIndex() < m_u8CurStrLength))
      {
         ++m_oCursorPosition;
      }
      vProcessCursorPosition();
      bPostEditFieldUpdMsg();
      return;
   }
   //UseCase: If their are chars(visible & invisible) to the right of cursor
   if (m_oCursorPosition.GetCursorIndex() < m_u8CurStrLength)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], vShiftCursorLeft() - visible/invisible char to right of cursor!", GetLegacyName()));
      // UseCase: If Cursor is in between the visible text i.e [abcd|ef]
      UInt32 displayedTextLength = u8UTFLength(m_sDisplayedText);
      if (m_oCursorPosition.GetDisplayedCursorIndex() < displayedTextLength)
      {
         if ((false == cobRestrictShiftOperations())
               || ((true == cobRestrictShiftOperations()) && (m_oCursorPosition.GetCursorIndex() >= m_uHighlightStartIndex) && m_oCursorPosition.GetCursorIndex() < m_uHighlightStartIndex + m_uHighlightCount))
         {
            ++m_oCursorPosition;
         }
      }
      // UseCase: If Cursor is at end of visible text & their are invisible chars to right of EditField i.e abc[defgh|]jhk
      else if (m_oCursorPosition.GetDisplayedCursorIndex() == displayedTextLength)
      {
         m_u8VisibleCharCount++;
         ++m_oCursorPosition;
         while ((m_u8VisibleCharCount > EF_MIN_TEXT_LENGTH)
                && (m_u32TruncationTextDisplayedWidth + cou16ComputeWidth(m_sMaskedText, m_u8VisibleStartIndex, m_u8VisibleCharCount)) >= m_u16TextAreaWidth)
         {
            m_u8VisibleStartIndex++;
            m_u8VisibleCharCount--;
         }
         //Since the text has Scrolled to left, update the visible text
         vUpdateVisibleText();
      }
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], vShiftCursorRight() - Index updated!", GetLegacyName()));
      vUpdateCursorPos();
      bPostEditFieldUpdMsg();
   }
}


/****************************************************************************
*     Function    : vUpdateEntryText
*     Description : Function for updating EntryText in MatchTextMode
*     Parameters  : void
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vUpdateEntryText()
{
   //User entered text is the EntryText
   m_sMaskedText = m_sEntryTextReceived;
   m_sEditFieldInfo = m_sEntryTextReceived;

   bPostEditFieldUpdMsg();
   PropertyModified("EditFieldInfo");

   m_u8VisibleStartIndex = EF_MIN_INDEX;
   m_u8VisibleCharCount = EF_MIN_TEXT_LENGTH;

   SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_sEditFieldInfo);
   m_u8CurStrLength = u8UTFLength(m_sEditFieldInfo.GetCString());
   //m_u8CurStrLength = (Candera::UInt8)StringPlatform::Length(m_sEditFieldInfo.GetCString());
   SECURE_FEATSTD_STRING_ACCESS_END();

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%19s] - In vUpdateEntryText \
                         with m_u8CurStrLength as [%d].", GetLegacyName(), m_u8CurStrLength));
   //Display Entry Text only if its not empty otherwise display Instruction text
   if (EF_MIN_TEXT_LENGTH != m_u8CurStrLength)
   {
      m_u8VisibleStartIndex = m_u8CurStrLength - 1;
      m_u8VisibleCharCount = 1;
      //Determine right end of entry text that can be displayed
      while (m_u8VisibleStartIndex &&
             (cou16ComputeWidth(m_sMaskedText, m_u8VisibleStartIndex - 1,
                                m_u8VisibleCharCount + 1) <= m_u16TextAreaWidth)
            )
      {
         m_u8VisibleStartIndex--;
         m_u8VisibleCharCount++;
      }
      //Set the cursor index for cursor positioning
      m_oCursorPosition.SetCursorIndex(m_u8VisibleStartIndex + m_u8VisibleCharCount);
      //Set the width of Entry text widget dynamicaly to accomodate place for match text also
      m_v2EntryTextSize.SetX(static_cast<Candera::Float>(cou16ComputeWidth(m_sMaskedText, m_u8VisibleStartIndex, m_u8VisibleCharCount)));
      m_oEntryTextWidget.TextWidget2D::SetMaximumSize(m_v2EntryTextSize);

      vUpdateVisibleText();
      vUpdateCursorPos();
      //SUZUKI - 13572 - Start
      m_bUpdateInstructionText = false;
      //SUZUKI - 13572 - End
   }
   else
   {
      m_oCursorPosition = CursorPosition(0);
      m_u8VisibleCharCount = 0;
      m_u8VisibleStartIndex = 0;
      m_v2EntryTextSize.SetX(static_cast<Candera::Float>(m_u16TextAreaWidth));
      vUpdateVisibleText();
      vUpdateTextNodePosDim();
      vUpdateCursorPos();
      m_bUpdateInstructionText = true;
   }

   //Set the width and position for MatchText based on EntryText
   if (NULL != m_pMatchTextNode)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s] - In vUpdateEntryText \
                         with m_pMatchTextNode Not NULL.", GetLegacyName()));
      m_pMatchTextNode->SetPosition(m_vEntryTextNodePos + Candera::Vector2(m_v2EntryTextSize.GetX(), 0.0f));
      m_v2MatchTextSize.SetX(static_cast<Candera::Float>(m_u16TextAreaWidth) - m_v2EntryTextSize.GetX());
      m_oMatchTextWidget.TextWidget2D::SetMaximumSize(m_v2MatchTextSize);
   }
   //To reflect changes on the display
   m_bInvalidateRequired = true;
//   invalidate_IfWakeUpRenderMechanismIsEnabled();
   triggerUpdate();
}


/****************************************************************************
*     Function    : vUpdateMatchText
*     Description : Function for updating MatchText in MatchTextMode
*     Parameters  : void
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vUpdateMatchText()
{
   Candera::UInt32 u8MatchTextLength;
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_sSuggestionText);
   u8MatchTextLength = u8UTFLength(m_sSuggestionText.GetCString());
   SECURE_FEATSTD_STRING_ACCESS_END();
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%19s] - In vUpdateMatchText \
                     with u8MatchTextLength as [%d].", GetLegacyName(), u8MatchTextLength));
   //Candera::UInt8 u8MatchTextLength = (Candera::UInt8)StringPlatform::Length(m_sSuggestionText.GetCString());
   //If EntryText is not empty and suggestion text length is greater than length of entry text
   //eg: EntryText:Banga , MatchText:Bangalore
   if (u8MatchTextLength >= m_u8CurStrLength)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s] - In vUpdateMatchText \
                         with (MIN_TEXT_LENGTH != m_u8CurStrLength) && \
                         (u8MatchTextLength >= m_u8CurStrLength).", GetLegacyName()));
      //Compute the length of Suggestion,eg: in above case it is 4
      Candera::UInt32 u8SuggestionLength = u8MatchTextLength - m_u8CurStrLength;
      Candera::TChar* pMatchTextStr = CANDERA_NEW_ARRAY(Char, (u8SuggestionLength * MAX_BYTE_FOR_A_CHARACTER + 1));
      if (NULL != pMatchTextStr)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s] - In vUpdateMatchText \
                         with pMatchTextStr not NULL.", GetLegacyName()));
         //initialize the temp string to empty string
         StringPlatform::Copy(pMatchTextStr, "");
         //Create the MatchText from the string received from lower layers
         //StringPlatform::Copy((Char*)pMatchTextStr, (m_sSuggestionText.GetCString() + m_u8CurStrLength));
         SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_sSuggestionText);
         uCopyUTF8Data(pMatchTextStr, u8MatchTextLength, 0, m_sSuggestionText.GetCString(), m_u8CurStrLength, u8SuggestionLength);
         SECURE_FEATSTD_STRING_ACCESS_END();

         pMatchTextStr[u8SuggestionLength] = '\0';
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%10s] - In vUpdateMatchText \
                         with pMatchTextStr [%s].", GetLegacyName(), pMatchTextStr));
         //Set MatchText
         m_oMatchTextWidget.SetText(pMatchTextStr);
         m_oMatchTextWidget.Update();
         //m_oMatchTextWidget.Invalidate();

         //SUZUKI - 13572 - Start
         m_bInvalidateRequired = true;
         //SUZUKI - 13572 - End
         //vCheckAndInvalidate();
         //free heap memory
         FEATSTD_SAFE_DELETE_ARRAY(pMatchTextStr);
      }
   }
}


/****************************************************************************
*     Function    : cou16ComputeHeight
*     Description : Function for computing height of substring of n chars out of MaskedText string from index 'u8Start'
*     Parameters  : Candera::UInt8 , Candera::UInt8
*     Return      : Candera::UInt16
****************************************************************************/
Candera::UInt32 EditFieldWidget2D::cou16ComputeHeight(Candera::String text, Candera::UInt32 u8Start, Candera::UInt32 u8nChars)
{
   Candera::UInt32 u16Height = 0;
   if (u8nChars > 0)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], cou16ComputeWidth() - NULL!=m_oEntryTextWidget && u8nChars>0)!", GetLegacyName()));
      UInt32 textLength;

      SECURE_FEATSTD_STRING_ACCESS_BEGIN(text);
      textLength = static_cast<UInt32>(StringPlatform::TLength(text.GetCString()));
      SECURE_FEATSTD_STRING_ACCESS_END();

      Candera::TChar* pTempStr = CANDERA_NEW_ARRAY(Char, (textLength * MAX_BYTE_FOR_A_CHARACTER) + 1);
      if (NULL != pTempStr)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], cou16ComputeWidth() - pTempStr not NULL!", GetLegacyName()));
         TextRendering::TextRenderer oRenderer;

         //initialize the temp string to empty string
         StringPlatform::Copy(pTempStr, "");
         //Obtain the string for which the width has to be computed
         uCopyUTF8Data(pTempStr, textLength, 0, text, u8Start, u8nChars);

         //SUZUKI - 13572 - Start
         //set font if not previously set.
         if (oShapingOption.GetStyle().PointsToNull())
         {
            Candera::MemoryManagement::SharedPointer<Candera::TextRendering::SharedStyle> pStyle = Candera::TextRendering::SharedStyle::Create();
            if (false == pStyle.PointsToNull())
            {
               pStyle->SetDefaultFont(m_oEntryTextFont);
            }
            m_oEntryTextWidget.SetStyle(pStyle);
            oShapingOption = TextRendering::ShapingOptions(pStyle);
            //vConfgEntryTxtWidget();
         }
         //SUZUKI - 13572 - End
         Candera::TextRendering::LayoutingOptions layoutingOptions;
         layoutingOptions.SetMultilineTextEnabled(true);
         layoutingOptions.SetLineSpacing(static_cast<UInt16>(couGetLineSpacing()));
         layoutingOptions.SetWordWrapEnabled(true);
         Candera::TextRendering::TextSize size(static_cast<Candera::Int16>(m_oEntryTextWidget.TextWidget2D::GetMaximumSize().GetX()), static_cast<Candera::Int16>(m_oEntryTextWidget.TextWidget2D::GetMaximumSize().GetY()));
         layoutingOptions.SetSize(size);
         //Compute width the desired string
#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 2)))

#ifdef VARIANT_S_FTR_ENABLE_TEXTWIDGET2DV2
         //The CursorTextMeasureContext is used with TextWidget2DV2 as TextWidget2DV2 aligned to CursorTextMeasureContext
         Candera::TextRendering::CursorTextMeasureContext curContext(*(oShapingOption.GetStyle().GetPointerToSharedInstance()));
         if (!Candera::TextRendering::TextRenderer().Render(curContext, layoutingOptions, oShapingOption, Candera::TextRendering::TextProperties(pTempStr)))
#else
         Candera::TextRendering::GlyphTextMeasureContext context(0);
         if (!Candera::TextRendering::TextRenderer().Render(context, oLayoutingOption, oShapingOption, Candera::TextRendering::TextProperties(pTempStr)))
#endif
         {
            u16Height = 0;
         }
         else
         {
#ifdef VARIANT_S_FTR_ENABLE_TEXTWIDGET2DV2
            u16Height = curContext.GetTextRectangle().GetHeight();
#else
            u16width = context.GetTextRectangle().GetWidth();
#endif
         }
#else

         u16width = static_cast<Candera::UInt32>(oRenderer.GetBoundingRectangle(
               oLayoutingOption, oShapingOption, TextRendering::TextProperties(pTempStr), oMeasuringOption).GetWidth());
#endif

         FEATSTD_SAFE_DELETE_ARRAY(pTempStr);
      }
   }
   return u16Height;
}


/****************************************************************************
*     Function    : cou16ComputeWidth
*     Description : Function for computing width of substring of n chars out of MaskedText string from index 'u8Start'
*     Parameters  : Candera::UInt8 , Candera::UInt8
*     Return      : Candera::UInt16
****************************************************************************/
Candera::UInt32 EditFieldWidget2D::cou16ComputeWidth(Candera::String text, Candera::UInt32 u8Start, Candera::UInt32 u8nChars)
{
   Candera::UInt32 u16width = 0;
   if (u8nChars > 0)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], cou16ComputeWidth() - NULL!=m_oEntryTextWidget && u8nChars>0)!", GetLegacyName()));
      UInt32 textLength ;

      SECURE_FEATSTD_STRING_ACCESS_BEGIN(text);
      textLength = static_cast<UInt32>(StringPlatform::TLength(text.GetCString()));
      SECURE_FEATSTD_STRING_ACCESS_END();

      Candera::TChar* pTempStr = CANDERA_NEW_ARRAY(Char, (textLength * MAX_BYTE_FOR_A_CHARACTER) + 1);
      if (NULL != pTempStr)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s], cou16ComputeWidth() - pTempStr not NULL!", GetLegacyName()));

         //initialize the temp string to empty string
         StringPlatform::Copy(pTempStr, "");
         //Obtain the string for which the width has to be computed
         uCopyUTF8Data(pTempStr, textLength, 0, text, u8Start, u8nChars);

         //SUZUKI - 13572 - Start
         //set font if not previously set.
         if (oShapingOption.GetStyle().PointsToNull())
         {
            Candera::MemoryManagement::SharedPointer<Candera::TextRendering::SharedStyle> pStyle = Candera::TextRendering::SharedStyle::Create();
            if (false == pStyle.PointsToNull())
            {
               pStyle->SetDefaultFont(m_oEntryTextFont);
            }
            m_oEntryTextWidget.SetStyle(pStyle);
            //Kanji Req
            m_oHighlightTextWidget.SetStyle(pStyle);
            m_oEndTextWidget.SetStyle(pStyle);
            oShapingOption = TextRendering::ShapingOptions(pStyle);
            //vConfgEntryTxtWidget();
         }
         //SUZUKI - 13572 - End

         //Compute width the desired string
#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 2)))

#ifdef VARIANT_S_FTR_ENABLE_TEXTWIDGET2DV2
         //The CursorTextMeasureContext is used with TextWidget2DV2 as TextWidget2DV2 aligned to CursorTextMeasureContext
         Candera::TextRendering::CursorTextMeasureContext curContext(*(oShapingOption.GetStyle().GetPointerToSharedInstance()));
         if (!Candera::TextRendering::TextRenderer().Render(curContext, oLayoutingOption, oShapingOption, Candera::TextRendering::TextProperties(pTempStr)))
#else
         Candera::TextRendering::GlyphTextMeasureContext context(0);
         if (!Candera::TextRendering::TextRenderer().Render(context, oLayoutingOption, oShapingOption, Candera::TextRendering::TextProperties(pTempStr)))
#endif
         {
            u16width = 0;
         }
         else
         {
#ifdef VARIANT_S_FTR_ENABLE_TEXTWIDGET2DV2
            u16width = curContext.GetTextRectangle().GetWidth();
#else
            u16width = context.GetTextRectangle().GetWidth();
#endif
         }
#else
         TextRendering::TextRenderer oRenderer;
         u16width = static_cast<Candera::UInt32>(oRenderer.GetBoundingRectangle(
               oLayoutingOption, oShapingOption, TextRendering::TextProperties(pTempStr), oMeasuringOption).GetWidth());
#endif

         FEATSTD_SAFE_DELETE_ARRAY(pTempStr);
      }
   }
   return u16width;
}


/****************************************************************************
*     Function    : vConfgEntryTxtWidget
*     Description : Function for Configuring EntryText widget
*     Parameters  : void
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vConfgEntryTxtWidget()
{
   m_oEntryTextWidget.SetNode(m_pEntryTextNode);
   m_oEntryTextWidget.WidgetBase::SetName("EntryTextWidget");
   m_oEntryTextWidget.SetCultureDependentAlignment(true);

   if (cobGetMultiLineLayouting() == false)
   {
      m_oEntryTextWidget.TextWidget2D::SetVerticalAlignment(Candera::VCenter);
   }
   else
   {
      m_oEntryTextWidget.TextWidget2D::SetVerticalAlignment(Candera::VTop);
      m_oEntryTextWidget.TextWidget2D::SetMultiLineLayouting(true);
      m_oEntryTextWidget.TextWidget2D::SetWordWrap(true);
      m_oEntryTextWidget.TextWidget2D::SetLineSpacingInPixels(static_cast<Candera::Float>(couGetLineSpacing()));
      if ((NULL != m_pEntryTextNode) && (NULL != m_pEntryTextNode->GetLayouter()))
      {
         const Candera::Vector2 maxSize = static_cast<Candera::Vector2>(m_oEntryTextWidget.TextWidget2D::GetMaximumSize());
         m_pEntryTextNode->GetLayouter()->SetMaximumSize(*m_pEntryTextNode, maxSize);
      }
   }

   m_oEntryTextWidget.SetTruncationMethod(Hard);

   //Kanji Req
   if (NULL != m_pHighlightTextNode)
   {
      m_oHighlightTextWidget.SetNode(m_pHighlightTextNode);
      m_oHighlightTextWidget.WidgetBase::SetName("HighlightTextWidget");
      m_oHighlightTextWidget.TextWidget2D::SetVerticalAlignment(Candera::VCenter);
      m_oHighlightTextWidget.SetTruncationMethod(Hard);
      m_oHighlightTextWidget.SetCultureDependentAlignment(true);
   }

   if (NULL != m_pEndTextNode)
   {
      m_oEndTextWidget.SetNode(m_pEndTextNode);
      m_oEndTextWidget.WidgetBase::SetName("EndTextWidget");
      m_oEndTextWidget.TextWidget2D::SetVerticalAlignment(Candera::VCenter);
      m_oEndTextWidget.SetTruncationMethod(Hard);
      m_oEndTextWidget.SetCultureDependentAlignment(true);
   }

   if (true == m_TextStyle.PointsToNull())
   {
      Candera::MemoryManagement::SharedPointer<Candera::TextRendering::SharedStyle> pStyle = Candera::TextRendering::SharedStyle::Create();
      if (false == pStyle.PointsToNull())
      {
         pStyle->SetDefaultFont(m_oEntryTextFont);
      }
      _TODO("Editfield with Style")
      ///   m_oEntryTextWidget.SetFont(m_oInstructionTextFont);

      m_oEntryTextWidget.SetStyle(pStyle);
      //Kanji Req
      m_oHighlightTextWidget.SetStyle(pStyle);
      m_oEndTextWidget.SetStyle(pStyle);
      oShapingOption = TextRendering::ShapingOptions(m_oEntryTextWidget.GetStyle());
   }
   m_oEntryTextWidget.SetTextColor(m_bEditFieldEnabled ? m_oEnabledInstructionTextColor : m_oDisabledInstructionTextColor);
   //kanji Req
   m_oHighlightTextWidget.SetTextColor(m_bEditFieldEnabled ? m_oEnabledInstructionTextColor : m_oDisabledInstructionTextColor);
   m_oEndTextWidget.SetTextColor(m_bEditFieldEnabled ? m_oEnabledInstructionTextColor : m_oDisabledInstructionTextColor);
   m_oEntryTextWidget.TextWidget2D::SetDirection(hmibase::widget::text::enDirection::Implicit);
   vUpdateTextNodePosDim();
   m_bUpdateEntryTextWidget = true;
   m_bUpdateAdditionalTextWidget = true;
}


/****************************************************************************
*     Function    : vConfgMatchTxtWidget
*     Description : Function for Configuring MatcText Widget
*     Parameters  : void
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vConfgMatchTxtWidget()
{
   m_oMatchTextWidget.SetNode(m_pMatchTextNode);
   m_oMatchTextWidget.WidgetBase::SetName("MatchTextWidget");

   m_oMatchTextWidget.TextWidget2D::SetVerticalAlignment(Candera::VCenter);

   m_oMatchTextWidget.SetTruncationMethod(Soft);
   m_oMatchTextWidget.SetCultureDependentAlignment(true);

   if (true == m_TextStyle.PointsToNull())
   {
      Candera::MemoryManagement::SharedPointer<Candera::TextRendering::SharedStyle> pStyle = Candera::TextRendering::SharedStyle::Create();
      if (false == pStyle.PointsToNull())
      {
         pStyle->SetDefaultFont(m_oMatchTextFont);
      }
      _TODO("Editfield with Style")
///   m_oMatchTextWidget.SetFont(m_oMatchTextFont);
      m_oMatchTextWidget.SetStyle(pStyle);
   }
   m_oMatchTextWidget.SetTextColor(m_bEditFieldEnabled ? m_oEnabledMatchTextColor : m_oDisabledMatchTextColor);
   m_oMatchTextWidget.TextWidget2D::SetDirection(hmibase::widget::text::enDirection::Implicit);
   //do this outside entry and match config fnctions
   vUpdateTextNodePosDim();
   m_bUpdateMatchTextWidget =  true;
}


/****************************************************************************
*     Description : Function for setting Image
*     Parameters  : Candera::String
****************************************************************************/
void EditFieldWidget2D::setBgImage(const BitmapType& bitmap)
{
   if ((NULL != GetAssetProvider()) && (NULL != m_pBackgroundNode) && !bitmap.PointsToNull())
   {
      {
         Candera::UInt32 u32bmpWidth = bitmap->GetWidth();
         Candera::UInt32 u32bmpHeight = bitmap->GetHeight();
         //if Bitmap width and height not zero, and to avoid division by zero
         if ((0 != u32bmpWidth) && (0 != u32bmpHeight))
         {
            Candera::RenderNode* renderNode = Candera::Dynamic_Cast<Candera::RenderNode*>(m_pBackgroundNode);
            if ((renderNode != NULL) && (renderNode->GetEffect(0) != NULL))
            {
               Candera::BitmapBrush* bitmapBrush = Candera::Dynamic_Cast<Candera::BitmapBrush*>(renderNode->GetEffect(0)->GetBrushEffect2D());
               if (bitmapBrush != NULL)
               {
                  bitmapBrush->Unload();
                  bitmapBrush->Image().Set(bitmap);
                  bitmapBrush->Upload();
               }
            }
            Candera::Float flScalex = static_cast<Candera::Float>(m_u16BgWidth) / static_cast<Candera::Float>(u32bmpWidth);
            Candera::Float flScaley = static_cast<Candera::Float>(m_u8BgHeight) / static_cast<Candera::Float>(u32bmpHeight);
            m_pBackgroundNode->SetScale(flScalex, flScaley);
            m_bInvalidateRequired = true;
//            invalidate_IfWakeUpRenderMechanismIsEnabled();
            triggerUpdate();
         }
      }
   }
}


/****************************************************************************
*     Function    : vUpdateStateProperties
*     Description : Function for Updating state properties
*     Parameters  : void
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vUpdateStateProperties()
{
   if (enEFDisabled != m_enEFState)
   {
      m_oMatchTextWidget.SetTextColor(m_oEnabledMatchTextColor);
      if (EF_MIN_TEXT_LENGTH != m_u8CurStrLength)
      {
         m_oEntryTextWidget.SetTextColor(m_oEnabledEntryTextColor);
         //kanji Req
         m_oHighlightTextWidget.SetTextColor(m_oHighlightEnabledTextColor);
         m_oEndTextWidget.SetTextColor(m_oEnabledEntryTextColor);
      }
      else
      {
         m_bUpdateInstructionText = true;
      }
      switch (m_enEFState)
      {
         case enEFNormal:
         case enEFFocused:
         {
            setBgImage(_bitmapNormalBg);
            //Stop Cursor Animation to begin state if in Normal or Focused state

            Courier::Message* pCursorAnimStopMsg;
            SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_sCursorAnimation);

            pCursorAnimStopMsg = COURIER_MESSAGE_NEW(::Courier::AnimationReqMsg)(Courier::AnimationAction::ToBegin, Courier::ViewId(), Courier::CompositePath(), Courier::ItemId(m_sCursorAnimation.GetCString()), m_oCursorAnimProp);

            SECURE_FEATSTD_STRING_ACCESS_END();
            if (NULL != pCursorAnimStopMsg)
            {
               pCursorAnimStopMsg->Post();
            }
            break;
         }

         case enEFSelected:
         {
            setBgImage(_bitmapSelectedBg);
            //Start Cursor Animation on selection
            Courier::Message* pCursorAnimStartMsg;
            SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_sCursorAnimation);

            pCursorAnimStartMsg = COURIER_MESSAGE_NEW(::Courier::AnimationReqMsg)(Courier::AnimationAction::Start, Courier::ViewId(), Courier::CompositePath(), Courier::ItemId(m_sCursorAnimation.GetCString()), m_oCursorAnimProp);

            SECURE_FEATSTD_STRING_ACCESS_END();
            if (NULL != pCursorAnimStartMsg)
            {
               pCursorAnimStartMsg->Post();
            }

            break;
         }

         default:
         {
            break;
         }
      }
   }
   else
   {
      m_oMatchTextWidget.SetTextColor(m_oDisabledMatchTextColor);
      setBgImage(_bitmapDisabledBg);
      if (EF_MIN_TEXT_LENGTH != m_u8CurStrLength)
      {
         m_oEntryTextWidget.SetTextColor(m_oDisabledEntryTextColor);
         //kanji Req
         m_oHighlightTextWidget.SetTextColor(m_oHighlightDisabledTextColor);
         m_oEndTextWidget.SetTextColor(m_oDisabledEntryTextColor);
      }
      else
      {
         m_bUpdateInstructionText = true;
      }
   }
   m_bUpdateEntryTextWidget = true;
   m_bUpdateMatchTextWidget = true;
   m_bUpdateAdditionalTextWidget = true;
   m_bStateChanged = false;
   m_bPropertyChanged = false;
}


/****************************************************************************
*     Function    : vUpdateCursorImage
*     Description : Function for Updating Cursor Image
*     Parameters  : void
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vUpdateCursorImage()
{
   if (NULL != GetAssetProvider())
   {
      const BitmapType pBitmap = GetCursorBgBitmap();
      if (pBitmap == 0 || pBitmap.PointsToNull())
      {
         return;
      }
      Candera::RenderNode* renderNode = Candera::Dynamic_Cast<Candera::RenderNode*>(m_pCursorNode);
      if ((renderNode != NULL) && (renderNode->GetEffect(0) != NULL))
      {
         Candera::BitmapBrush* bitmapBrush = Candera::Dynamic_Cast<Candera::BitmapBrush*>(renderNode->GetEffect(0)->GetBrushEffect2D());
         if (bitmapBrush != NULL)
         {
            bitmapBrush->Unload();
            bitmapBrush->Image().Set(pBitmap);
            bitmapBrush->Upload();
         }
      }
      Candera::UInt32 u32bmpWidth = pBitmap->GetWidth();
      Candera::UInt32 u32bmpHeight = pBitmap->GetHeight();
      if (NULL != m_pCursorNode && (0 != u32bmpWidth) && (0 != u32bmpHeight))
      {
         Candera::Float flScalex = EF_CURSOR_WIDTH / static_cast<Candera::Float>(u32bmpWidth);
         Candera::Float flScaley = 1.0f;
         if (false == cobGetMultiLineLayouting())
         {
            flScaley = static_cast<Candera::Float>(m_u8TextAreaHeight) / static_cast<Candera::Float>(u32bmpHeight);
         }
         else
         {
            //TODO : check for height for cursor
            if (false == m_TextStyle.PointsToNull())
            {
               Int32 height = (GetTextStyle()->GetMetrics().ascender - GetTextStyle()->GetMetrics().descender);
               flScaley = static_cast<Candera::Float>(height) / static_cast<Candera::Float>(u32bmpHeight);
            }
            else
            {
               Int32 height = m_u8TextAreaHeight;
               if (cooGetEntryTextFont().IsValid())
               {
                  TextRendering::Metrics metrics;
                  cooGetEntryTextFont().GetMetrics(metrics);
                  height = (metrics.ascender - metrics.descender);
               }
               flScaley = static_cast<Candera::Float>(height) / static_cast<Candera::Float>(u32bmpHeight);
            }
         }
         m_pCursorNode->SetScale(flScalex, flScaley);
         m_bInvalidateRequired = true;
//         invalidate_IfWakeUpRenderMechanismIsEnabled();
         triggerUpdate();
      }
   }
}


/****************************************************************************
*     Function    : vUpdateTextNodePosDim
*     Description : Function for Updating text nodes position and dimensions
*     Parameters  : void
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vUpdateTextNodePosDim()
{
   m_oEntryTextWidget.TextWidget2D::SetMaximumSize(m_v2EntryTextSize);
   if ((NULL != m_pEntryTextNode) && (NULL != m_pEntryTextNode->GetLayouter()))
   {
      const Candera::Vector2 maxSize = static_cast<Candera::Vector2>(m_oEntryTextWidget.TextWidget2D::GetMaximumSize());
      m_pEntryTextNode->GetLayouter()->SetMaximumSize(*m_pEntryTextNode, maxSize);
   }
   //Kanji Req (Candera::Float)m_u8TextAreaHeight m_v2EntryTextSize
   Candera::Vector2 maxSize(static_cast<Candera::Float>(- 1), static_cast<Candera::Float>(m_u8TextAreaHeight));
   m_oHighlightTextWidget.TextWidget2D::SetMaximumSize(maxSize);
   m_oEndTextWidget.TextWidget2D::SetMaximumSize(maxSize);

   m_bUpdateEntryTextWidget =  true;
   m_bUpdateAdditionalTextWidget = true;
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "EditFieldWidget2D: [%s] - In vUpdateTextNodePosDim" , GetLegacyName()));
   if (NULL != m_pBackgroundNode)
   {
      m_vEntryTextNodePos.SetX(m_pBackgroundNode->GetPosition().GetX() + static_cast<Candera::Float>(m_u8TextAreaMarginX));
      m_vEntryTextNodePos.SetY(m_pBackgroundNode->GetPosition().GetY() + static_cast<Candera::Float>(m_u8TextAreaMarginY));

      //Kanji Req
      m_vHighlightTextNodePos.SetX(m_pBackgroundNode->GetPosition().GetX() + static_cast<Candera::Float>(m_u8TextAreaMarginX));
      m_vHighlightTextNodePos.SetY(m_pBackgroundNode->GetPosition().GetY() + static_cast<Candera::Float>(m_u8TextAreaMarginY));
      m_vEndTextNodePos.SetX(m_pBackgroundNode->GetPosition().GetX() + static_cast<Candera::Float>(m_u8TextAreaMarginX));
      m_vEndTextNodePos.SetY(m_pBackgroundNode->GetPosition().GetY() + static_cast<Candera::Float>(m_u8TextAreaMarginY));
   }
   else
   {
      m_vEntryTextNodePos.SetX(static_cast<Candera::Float>(m_u8TextAreaMarginX));
      m_vEntryTextNodePos.SetY(static_cast<Candera::Float>(m_u8TextAreaMarginY));

      //Kanji Req
      m_vHighlightTextNodePos.SetX(static_cast<Candera::Float>(m_u8TextAreaMarginX));
      m_vHighlightTextNodePos.SetY(static_cast<Candera::Float>(m_u8TextAreaMarginY));
      m_vEndTextNodePos.SetX(static_cast<Candera::Float>(m_u8TextAreaMarginX));
      m_vEndTextNodePos.SetY(static_cast<Candera::Float>(m_u8TextAreaMarginY));
   }

   if (NULL != m_pEntryTextNode)
   {
      m_pEntryTextNode->SetPosition(m_vEntryTextNodePos);
      //Kanji Req
      if (NULL != m_pHighlightTextNode)
      {
         m_pHighlightTextNode->SetPosition(m_vHighlightTextNodePos);
      }
      if (NULL != m_pEndTextNode)
      {
         m_pEndTextNode->SetPosition(m_vEndTextNodePos);
      }
      m_bInvalidateRequired = true;
//      invalidate_IfWakeUpRenderMechanismIsEnabled();
      triggerUpdate();
   }
   m_v2MatchTextSize.SetY(static_cast<Candera::Float>(m_u8TextAreaHeight));
   m_oMatchTextWidget.TextWidget2D::SetMaximumSize(m_v2MatchTextSize);
   m_bUpdateMatchTextWidget =  true;
}


/****************************************************************************
*     Function    : u8CalculateUTF8Bytes
*     Description : Calculate Number of Bytes occupied by a String with the index iStartIndex upto nCharCount (UTF8) characters
*     Parameters  : iStartIndex, uCharCount, string
*     Return      : Candera::UInt8 - Number of bytes the UTF8 characters occupies
****************************************************************************/
Candera::UInt32 EditFieldWidget2D::u8CalculateUTF8Bytes(Candera::UInt32 iStartIndex, Candera::UInt32 uCharCount, Candera::String string) const
{
   Candera::UInt32 nBytes = 0;
   // Validation
   if ((uCharCount > 0) && (uCharCount < EF_MAX_TEXT_LENGTH) && (iStartIndex < EF_MAX_TEXT_LENGTH))
   {
      Candera::UInt32 uStartIndex = iStartIndex;
      Candera::UInt32 iCharCount = uCharCount;
      SECURE_FEATSTD_STRING_ACCESS_BEGIN(string);
      FeatStd::Internal::Utf8TextIterator iter_Text(string.GetCString());

      // Calculate the number of bytes for a UTF character at index uStartIndex
      if (uStartIndex > 0)
      {
         while (uStartIndex > 0)
         {
            iter_Text.Advance(); // Move by one character till the start index is reached
            uStartIndex--;
         }
      }

      while (iCharCount > 0)
      {
         nBytes += static_cast<UInt32>(iter_Text.Advance());
         iCharCount--;
      }
      SECURE_FEATSTD_STRING_ACCESS_END();
   }
   return nBytes;
}


/****************************************************************************
*     Function    : uCopyUTF8Data
*     Description : Copy UTF8 Data
*     Parameters  : dstString, dstStringOffset, srcString, srcStringOffset,CharCountToCopy
*     Return      : Candera::UInt8 - Number of bytes copied
****************************************************************************/
Candera::UInt32 EditFieldWidget2D::uCopyUTF8Data(Char* dstString,
      Candera::UInt32 dstBufferLength,
      Candera::UInt32 dstStringOffset,
      Candera::String srcString,
      Candera::UInt32 srcStringOffset,
      Candera::UInt32 CharCountToCopy) const
{
   Candera::UInt32 uDstStringOffset = dstStringOffset;
   Candera::UInt32 bytesCopied = 0;
   Candera::UInt32 bytesBeforeStartIndex = 0;
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(srcString);
   FeatStd::Internal::Utf8TextIterator iter_SrcString(srcString.GetCString());

   if ((0 != CharCountToCopy) && (!srcString.IsEmpty()))
   {
      if (dstString != NULL)
      {
         bytesBeforeStartIndex = u8CalculateUTF8Bytes(0, uDstStringOffset, dstString);

         while (srcStringOffset > 0)
         {
            iter_SrcString.Advance();
            srcStringOffset--;
         }
         bytesCopied = static_cast<UInt32>(FeatStd::Internal::TextEncoding::Copy(dstString + bytesBeforeStartIndex,
                                           (dstBufferLength * MAX_BYTE_FOR_A_CHARACTER) + 1,
                                           iter_SrcString.Pos(),
                                           CharCountToCopy));

         Candera::UInt32 nbyte = u8CalculateUTF8Bytes(0, dstBufferLength, dstString);
         dstString[nbyte] = '\0';
      }
   }
   SECURE_FEATSTD_STRING_ACCESS_END();
   return bytesCopied;
}


/****************************************************************************
*     Function    : u8UTFLength
*     Description : Length of a utf8 string
*     Parameters  : Candera::String sText
*     Return      : UInt8 u8Length
****************************************************************************/
Candera::UInt32 EditFieldWidget2D::u8UTFLength(Candera::String sText) const
{
   UInt32 u8Length = 0;
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(sText);
   FeatStd::Internal::Utf8TextIterator iter_Text(sText.GetCString());

   while (*iter_Text)
   {
      u8Length++;
      iter_Text.Advance();
   }
   SECURE_FEATSTD_STRING_ACCESS_END();
   return u8Length;
}


void EditFieldWidget2D::vAddElipsisText(const Candera::TChar* text)
{
   if ((m_u8VisibleStartIndex != 0) && (m_sElipsisText != ""))
   {
      UInt32 visibleTextLength = static_cast<UInt32>(StringPlatform::TLength(text));
      UInt32 elipsisTextLength;
      SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_sElipsisText);
      elipsisTextLength = static_cast<UInt32>(StringPlatform::TLength(m_sElipsisText.GetCString()));
      SECURE_FEATSTD_STRING_ACCESS_END();

      TChar* textWithElipsis = CANDERA_NEW_ARRAY(TChar, (visibleTextLength + elipsisTextLength) * MAX_BYTE_FOR_A_CHARACTER + 1);
      UInt32 textStartIndex = 0;
      m_oCursorPosition.SetDisplayedCursorIndex((m_oCursorPosition.GetCursorIndex() - m_u8VisibleStartIndex) + elipsisTextLength + 1);

      UInt32 zero = 0;

      UInt32* srcOffset = 0;
      if (m_oCursorPosition.GetDisplayedCursorIndex() == elipsisTextLength + 1)
      {
         srcOffset = &zero;
      }
      else
      {
         srcOffset = &textStartIndex;
         --m_u8VisibleStartIndex;
      }

      UInt32 textWithElipsisWidth = 0;
      do
      {
         SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_sElipsisText);
         StringPlatform::Copy(textWithElipsis, m_sElipsisText.GetCString());
         SECURE_FEATSTD_STRING_ACCESS_END();
         StringPlatform::CopyPartial(textWithElipsis + elipsisTextLength, text + *srcOffset, visibleTextLength - textStartIndex);
         textWithElipsis[visibleTextLength + elipsisTextLength - textStartIndex] = 0;

         UInt32 textWithElipsisLength = static_cast<UInt32>(StringPlatform::TLength(textWithElipsis));
         textWithElipsisWidth = cou16ComputeWidth(textWithElipsis, 0, textWithElipsisLength);

         ++textStartIndex;
         m_oCursorPosition.SetDisplayedCursorIndex(m_oCursorPosition.GetDisplayedCursorIndex() - 1);
      }
      while (textWithElipsisWidth >= m_u16TextAreaWidth);

      m_u8VisibleStartIndex += *srcOffset;
      m_u8VisibleCharCount -= textStartIndex - 1;

      if (m_oCursorPosition.GetDisplayedCursorIndex() < elipsisTextLength)
      {
         m_oCursorPosition.SetDisplayedCursorIndex(elipsisTextLength);
      }

      m_sDisplayedText = textWithElipsis;
      CANDERA_DELETE_ARRAY(textWithElipsis);
   }
   else
   {
      m_sDisplayedText = text;
      m_oCursorPosition.SetDisplayedCursorIndex(m_oCursorPosition.GetCursorIndex() - m_u8VisibleStartIndex);
   }
}


FeatStd::UInt32 EditFieldWidget2D::cou8GetTruncationTextDisplayedWidth()
{
   UInt32 truncationTextDisplayedWidth = 0;

   if (m_u8VisibleStartIndex > 0)
   {
      truncationTextDisplayedWidth = cou16ComputeWidth(m_sElipsisText, 0, u8UTFLength(m_sElipsisText));
   }

   return truncationTextDisplayedWidth;
}


// To Reset the Highlight and End Text on special key like DeleteAll
void EditFieldWidget2D::vResetHighlightAndEndText()
{
   //Set text , then position and then call update() on text widgets
   m_oHighlightTextWidget.SetText("");
   m_oEndTextWidget.SetText("");
   if (NULL != m_pBackgroundNode)
   {
      m_vHighlightTextNodePos.SetX(m_pBackgroundNode->GetPosition().GetX() + static_cast<Candera::Float>(m_u8TextAreaMarginX));
      m_vHighlightTextNodePos.SetY(m_pBackgroundNode->GetPosition().GetY() + static_cast<Candera::Float>(m_u8TextAreaMarginY));

      m_vEndTextNodePos.SetX(m_pBackgroundNode->GetPosition().GetX() + static_cast<Candera::Float>(m_u8TextAreaMarginX));
      m_vEndTextNodePos.SetY(m_pBackgroundNode->GetPosition().GetY() + static_cast<Candera::Float>(m_u8TextAreaMarginY));
   }
   else
   {
      m_vHighlightTextNodePos.SetX(static_cast<Candera::Float>(m_u8TextAreaMarginX));
      m_vHighlightTextNodePos.SetY(static_cast<Candera::Float>(m_u8TextAreaMarginY));

      m_vEndTextNodePos.SetX(static_cast<Candera::Float>(m_u8TextAreaMarginX));
      m_vEndTextNodePos.SetY(static_cast<Candera::Float>(m_u8TextAreaMarginY));
   }
   if (NULL != m_pHighlightTextNode)
   {
      m_pHighlightTextNode->SetPosition(m_vHighlightTextNodePos);
   }
   if (NULL != m_pEndTextNode)
   {
      m_pEndTextNode->SetPosition(m_vEndTextNodePos);
   }
   m_oEndTextWidget.Update();
   m_oHighlightTextWidget.Update();
}


//TODO :: Improve name of this function
// To call from vUpdateVisibleText
void EditFieldWidget2D::vUpdateHighlightAndEndText()
{
   m_oEntryTextWidget.SetText(m_sDisplayedLeftText);
   m_oEntryTextWidget.Update();
   Candera::UInt32 LeftWidth = cou16ComputeWidth(m_sDisplayedLeftText, 0, u8UTFLength(m_sDisplayedLeftText));
   Candera::UInt32 HighlightWidth = cou16ComputeWidth(m_sDisplayedHighightText, 0, u8UTFLength(m_sDisplayedHighightText));

   // Use (m_pBackgroundNode->GetPosition().GetX() + m_u8TextAreaMarginX ) to always keep align to initial position, SetY() Not Required
   if (NULL != m_pBackgroundNode)
   {
      m_vHighlightTextNodePos.SetX(m_pBackgroundNode->GetPosition().GetX() + static_cast<Candera::Float>(m_u8TextAreaMarginX) + static_cast<Candera::Float>(LeftWidth));
      m_vEndTextNodePos.SetX(m_pBackgroundNode->GetPosition().GetX() + static_cast<Candera::Float>(m_u8TextAreaMarginX) + static_cast<Candera::Float>(LeftWidth) + static_cast<Candera::Float>(HighlightWidth));
   }
   else
   {
      m_vHighlightTextNodePos.SetX(static_cast<Candera::Float>(m_u8TextAreaMarginX + LeftWidth));
      m_vEndTextNodePos.SetX(static_cast<Candera::Float>(m_u8TextAreaMarginX + LeftWidth + HighlightWidth));
   }
   if (NULL != m_pHighlightTextNode)
   {
      m_pHighlightTextNode->SetPosition(m_vHighlightTextNodePos);
      m_oHighlightTextWidget.SetText(m_sDisplayedHighightText);
      m_oHighlightTextWidget.Update();
   }
   if (NULL != m_pEndTextNode)
   {
      m_pEndTextNode->SetPosition(m_vEndTextNodePos);
      m_oEndTextWidget.SetText(m_sDisplayedEndText);
      m_oEndTextWidget.Update();
   }
}


void EditFieldWidget2D::vProcessTextOnEditFieldAction(::hmibase::widget::editfield::enEditFieldTextAction::Enum action, Candera::String msgText)
{
   Candera::UInt32 sEditFieldInfoCharCount = u8UTFLength(m_sEditFieldInfo);
   Candera::UInt32 curHighlightCharCount = u8UTFLength(m_pHighlightText);
   Candera::UInt32 replaceCharCount = ((action == ::hmibase::widget::editfield::enEditFieldTextAction::Delete) ? 0 : u8UTFLength(msgText.GetCString()));
   Candera::Int32 diffCharCount = replaceCharCount - curHighlightCharCount;
   if (curHighlightCharCount > 0)
   {
      memset(m_pHighlightText, 0, (EF_MAX_TEXT_LENGTH * MAX_BYTE_FOR_A_CHARACTER) + 1);

      // Also update the m_sEditFieldInfo, CursorPosition etc
      Candera::UInt32 strLen = u8UTFLength(m_sEditFieldInfo.GetCString());
      Candera::UInt32 newStrLen = strLen + 1 + diffCharCount;
      TChar* changedEditFieldText = CANDERA_NEW_ARRAY(TChar, newStrLen);

      uCopyUTF8Data(changedEditFieldText, newStrLen, 0, m_sEditFieldInfo, 0, m_uHighlightStartIndex);
      uCopyUTF8Data(changedEditFieldText, newStrLen, m_uHighlightStartIndex, msgText.GetCString(), 0, replaceCharCount);
      Candera::UInt32 charsToCopy = sEditFieldInfoCharCount - (m_uHighlightStartIndex + m_uHighlightCount);
      uCopyUTF8Data(changedEditFieldText, newStrLen, m_uHighlightStartIndex + replaceCharCount, m_sEditFieldInfo, m_uHighlightStartIndex + curHighlightCharCount, charsToCopy);

      // Highlight Index and count reset to 0 as text is replaced
      m_uHighlightCount = EF_MIN_INDEX;
      m_uHighlightStartIndex = EF_MIN_INDEX;
      m_u8CurStrLength = EF_MIN_INDEX;
      m_oCursorPosition = EF_MIN_INDEX;

      m_sEditFieldInfo = "";
      // TODO : check if manually required to update data
      vSetKeyReceived(changedEditFieldText);
      FEATSTD_SAFE_DELETE_ARRAY(changedEditFieldText);
   }
}


void EditFieldWidget2D::vProcessEditFieldHightlightReqMsg(const ::hmibase::widget::editfield::EditFieldHightlightReqMsg& msg)
{
   switch (msg.GetAction())
   {
      case ::hmibase::widget::editfield::enEditFieldTextAction::Delete:
      {
         vProcessTextOnEditFieldAction(::hmibase::widget::editfield::enEditFieldTextAction::Delete, msg.GetText());
         break;
      }
      case ::hmibase::widget::editfield::enEditFieldTextAction::Replace:
      {
         vProcessTextOnEditFieldAction(::hmibase::widget::editfield::enEditFieldTextAction::Replace, msg.GetText());
         break;
      }
      case ::hmibase::widget::editfield::enEditFieldTextAction::Select:
      {
         m_uHighlightStartIndex = msg.GetIndex();
         m_uHighlightCount = msg.GetCount();

         m_bHighlightDataChanged = true;

         //reset m_sKeyReceived in case of highlight controlling.
         m_sKeyReceived = "";

         //Updating above variable will call vinsertchar() from update cycle and we can control cursorposition also line 2198
         vInsertChar();
         //if (m_bHighlightDataChanged){--m_oCursorPosition;}	//To be used if m_bHighlightDataChanged not used through vInsertChar call
         bPostEditFieldUpdMsg();
         m_bHighlightDataChanged = false;
         break;
      }
      default:
         break;
   }
}


void EditFieldWidget2D::vAddElipsisToText(const Candera::TChar* text, Candera::String& elipsesAdded)
{
   if ((m_u8VisibleStartIndex != 0) && (m_sElipsisText != ""))
   {
      UInt32 visibleTextLength = static_cast<UInt32>(StringPlatform::TLength(text));
      UInt32 elipsisTextLength;
      SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_sElipsisText);
      elipsisTextLength = static_cast<UInt32>(StringPlatform::TLength(m_sElipsisText.GetCString()));
      SECURE_FEATSTD_STRING_ACCESS_END();

      TChar* textWithElipsis = CANDERA_NEW_ARRAY(TChar, (visibleTextLength + elipsisTextLength) * MAX_BYTE_FOR_A_CHARACTER + 1);
      UInt32 textStartIndex = 0;

      SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_sElipsisText);
      StringPlatform::Copy(textWithElipsis, m_sElipsisText.GetCString());
      SECURE_FEATSTD_STRING_ACCESS_END();
      StringPlatform::CopyPartial(textWithElipsis + elipsisTextLength, text, visibleTextLength);
      textWithElipsis[visibleTextLength + elipsisTextLength - textStartIndex] = 0;

      elipsesAdded = textWithElipsis;
      CANDERA_DELETE_ARRAY(textWithElipsis);
   }
}


void EditFieldWidget2D::vUpdateAllText()
{
   m_sDisplayedLeftText = "";
   m_sDisplayedHighightText = "";
   m_sDisplayedEndText = "";
   Candera::TChar* pEditFieldText = CANDERA_NEW_ARRAY(Char, (m_u8CurStrLength * MAX_BYTE_FOR_A_CHARACTER) + 1);
   if (NULL != pEditFieldText)
   {
      StringPlatform::Copy(pEditFieldText, "");

      //Prepare Left Text
      uCopyUTF8Data(pEditFieldText, m_uLeftVisibleCharCount, 0, m_pLeftText, m_uLeftVisibleStartIndex, m_uLeftVisibleCharCount);
      m_sDisplayedLeftText = pEditFieldText;
      // We add the truncation text to only the left text widget. m_sDisplayedLeftText is updated after prepending the ...
      vAddElipsisToText(pEditFieldText, m_sDisplayedLeftText);

      //Prepare Highlight Text
      StringPlatform::Copy(pEditFieldText, "");
      uCopyUTF8Data(pEditFieldText, m_uHighlightVisibleCharCount, 0, m_pHighlightText, m_uHighlightVisibleStartIndex, m_uHighlightVisibleCharCount);
      m_sDisplayedHighightText = pEditFieldText;

      //Prepare End Text
      StringPlatform::Copy(pEditFieldText, "");
      uCopyUTF8Data(pEditFieldText, m_uEndVisibleCharCount, 0, m_pEndText, m_uEndVisibleStartIndex, m_uEndVisibleCharCount);
      m_sDisplayedEndText = pEditFieldText;
   }
   FEATSTD_SAFE_DELETE_ARRAY(pEditFieldText);
}


void EditFieldWidget2D::vProcessVisibleTextAndCharCount()
{
   Candera::UInt32 leftTextCharCount = u8UTFLength(m_pLeftText);
   Candera::UInt32 highlightTextCharCount = u8UTFLength(m_pHighlightText);
   Candera::UInt32 endTextCharCount = u8UTFLength(m_pEndText);
   // Left,HightLight and End Text Visible Index and Count: This information is used to pick the left,highlight and end text which is visible when text is more than the text area.
   // The current visible text starts within the end text range
   if (m_u8VisibleStartIndex > (leftTextCharCount + highlightTextCharCount))
   {
      m_uLeftVisibleStartIndex = EF_MIN_INDEX;
      m_uLeftVisibleCharCount = 0;
      m_uHighlightVisibleStartIndex = EF_MIN_INDEX;
      m_uHighlightVisibleCharCount = 0;
      //End Text Visible Index and Count

      m_uEndVisibleStartIndex = m_u8VisibleStartIndex - (leftTextCharCount + highlightTextCharCount);
      m_uEndVisibleCharCount = m_u8VisibleCharCount;
   }
   // The current visible text starts within the highlight text range
   else if (m_u8VisibleStartIndex > leftTextCharCount)
   {
      m_uLeftVisibleStartIndex = EF_MIN_INDEX;
      m_uLeftVisibleCharCount = 0;

      m_uHighlightVisibleStartIndex = m_u8VisibleStartIndex - leftTextCharCount;
      if (m_u8VisibleCharCount > (highlightTextCharCount - m_uHighlightVisibleStartIndex))
      {
         m_uHighlightVisibleCharCount = (highlightTextCharCount - m_uHighlightVisibleStartIndex);
         m_uEndVisibleStartIndex = EF_MIN_INDEX;
         m_uEndVisibleCharCount = m_u8VisibleCharCount - m_uHighlightVisibleCharCount;
      }
      else
      {
         m_uHighlightVisibleCharCount = m_u8VisibleCharCount;
         m_uEndVisibleStartIndex = EF_MIN_INDEX;
         m_uEndVisibleCharCount = 0;
      }
   }
   // The current visible text starts within the Left text range
   else
   {
      // The visible text consists of left and highlight text both
      m_uLeftVisibleStartIndex = m_u8VisibleStartIndex;
      if (m_u8VisibleCharCount > (leftTextCharCount - m_uLeftVisibleStartIndex))
      {
         m_uLeftVisibleCharCount = (leftTextCharCount) - m_u8VisibleStartIndex;
         m_uHighlightVisibleStartIndex = EF_MIN_INDEX;
         m_uEndVisibleStartIndex = EF_MIN_INDEX;
         if (highlightTextCharCount > 0)
         {
            if (highlightTextCharCount >= (m_u8VisibleCharCount - m_uLeftVisibleCharCount))
            {
               m_uHighlightVisibleCharCount = m_u8VisibleCharCount - m_uLeftVisibleCharCount;
               m_uEndVisibleCharCount = 0;
            }
            else
            {
               m_uHighlightVisibleCharCount = highlightTextCharCount;
               m_uEndVisibleCharCount = m_u8VisibleCharCount - (m_uLeftVisibleCharCount + m_uHighlightVisibleCharCount);
            }
         }
         else
         {
            m_uEndVisibleCharCount = m_u8VisibleCharCount - (m_uLeftVisibleCharCount + m_uHighlightVisibleCharCount);
         }
      }
      else
      {
         // Means only the Left text is visible in visible area
         m_uLeftVisibleCharCount = m_u8VisibleCharCount;

         m_uHighlightVisibleStartIndex = EF_MIN_INDEX;
         m_uHighlightVisibleCharCount = 0;
         m_uEndVisibleStartIndex = EF_MIN_INDEX;
         m_uEndVisibleCharCount = 0;
      }
   }
   // Update the left , highlight and end text with the above calculated values.
   // The text which is within the visible are is added to the left or highlight or end or mix of them depeding upon above calculated values
   vUpdateAllText();
}


void EditFieldWidget2D::vUpdateAllTextOnDeleteChar()
{
   Candera::TChar* pEditFieldText = CANDERA_NEW_ARRAY(Char, (m_u8CurStrLength * MAX_BYTE_FOR_A_CHARACTER) + 1);
   if (NULL != pEditFieldText)
   {
      if ((m_uHighlightCount > 0) && (m_u8CurStrLength > 0))
      {
         StringPlatform::Copy(pEditFieldText, "");
         Candera::UInt32 leftTextCharCount = u8UTFLength(m_pLeftText);
         Candera::UInt32 highlightTextCharCount = u8UTFLength(m_pHighlightText);
         Candera::UInt32 endTextCharCount = u8UTFLength(m_pEndText);
         Candera::UInt32 totalCharCount = endTextCharCount + highlightTextCharCount + leftTextCharCount;
         if ((m_oCursorPosition.GetCursorIndex()) > (leftTextCharCount + highlightTextCharCount))
         {
            StringPlatform::Copy(pEditFieldText, m_pEndText);
            memset(m_pEndText, 0, (EF_MAX_TEXT_LENGTH * MAX_BYTE_FOR_A_CHARACTER) + 1);
            if ((m_oCursorPosition.GetCursorIndex() < (leftTextCharCount + highlightTextCharCount + endTextCharCount)))
            {
               uCopyUTF8Data(m_pEndText, endTextCharCount, 0, pEditFieldText, 0, m_oCursorPosition.GetCursorIndex() - (leftTextCharCount + highlightTextCharCount) - 1);
               uCopyUTF8Data(m_pEndText, endTextCharCount, m_oCursorPosition.GetCursorIndex() - (leftTextCharCount + highlightTextCharCount) - 1, pEditFieldText, m_oCursorPosition.GetCursorIndex() - (leftTextCharCount + highlightTextCharCount), totalCharCount - (m_oCursorPosition.GetCursorIndex() - 1));
               //uCopyUTF8Data(m_pEndText, endTextCharCount, m_oCursorPosition.GetCursorIndex() - (leftTextCharCount + highlightTextCharCount) - 1, pEditFieldText, m_oCursorPosition.GetCursorIndex() - (leftTextCharCount + highlightTextCharCount), highlightTextCharCount - (m_oCursorPosition.GetCursorIndex() - (leftTextCharCount + highlightTextCharCount)));
            }
            else
            {
               uCopyUTF8Data(m_pEndText, endTextCharCount, 0, pEditFieldText, 0, endTextCharCount - 1);
            }
         }
         else if ((m_oCursorPosition.GetCursorIndex()) > leftTextCharCount)
         {
            StringPlatform::Copy(pEditFieldText, m_pHighlightText);
            memset(m_pHighlightText, 0, (EF_MAX_TEXT_LENGTH * MAX_BYTE_FOR_A_CHARACTER) + 1);
            if ((m_oCursorPosition.GetCursorIndex() > leftTextCharCount) && (m_oCursorPosition.GetCursorIndex() < (leftTextCharCount + highlightTextCharCount)))
            {
               uCopyUTF8Data(m_pHighlightText, highlightTextCharCount, 0, pEditFieldText, 0, m_oCursorPosition.GetCursorIndex() - leftTextCharCount - 1);
               uCopyUTF8Data(m_pHighlightText, highlightTextCharCount, m_oCursorPosition.GetCursorIndex() - leftTextCharCount - 1, pEditFieldText, m_oCursorPosition.GetCursorIndex() - leftTextCharCount, highlightTextCharCount - (m_oCursorPosition.GetCursorIndex() - leftTextCharCount));
            }
            else
            {
               uCopyUTF8Data(m_pHighlightText, highlightTextCharCount, 0, pEditFieldText, 0, highlightTextCharCount - 1);
            }
            if (endTextCharCount > 0)
            {
               StringPlatform::Copy(pEditFieldText, m_pEndText);
               memset(m_pEndText, 0, (EF_MAX_TEXT_LENGTH * MAX_BYTE_FOR_A_CHARACTER) + 1);
               uCopyUTF8Data(m_pHighlightText, highlightTextCharCount, highlightTextCharCount - 1, pEditFieldText, 0, 1);
               uCopyUTF8Data(m_pEndText, endTextCharCount - 1, 0, pEditFieldText, 1, endTextCharCount - 1);
            }
            // This line will be req if highlight index and count needs to be changed if the insertion of chars is done in between the left or highlight text
            //m_uHighlightCount--;
         }
         else
         {
            StringPlatform::Copy(pEditFieldText, m_pLeftText);
            memset(m_pLeftText, 0, (EF_MAX_TEXT_LENGTH * MAX_BYTE_FOR_A_CHARACTER) + 1);
            if ((m_oCursorPosition.GetCursorIndex() < leftTextCharCount))
            {
               uCopyUTF8Data(m_pLeftText, leftTextCharCount, 0, pEditFieldText, 0, m_oCursorPosition.GetCursorIndex() - 1);
               uCopyUTF8Data(m_pLeftText, leftTextCharCount, m_oCursorPosition.GetCursorIndex() - 1, pEditFieldText, m_oCursorPosition.GetCursorIndex(), leftTextCharCount - (m_oCursorPosition.GetCursorIndex() - 1));
            }
            else
            {
               uCopyUTF8Data(m_pLeftText, leftTextCharCount, 0, pEditFieldText, 0, leftTextCharCount - 1);
            }
            if (highlightTextCharCount > 0)
            {
               StringPlatform::Copy(pEditFieldText, m_pHighlightText);
               memset(m_pHighlightText, 0, (EF_MAX_TEXT_LENGTH * MAX_BYTE_FOR_A_CHARACTER) + 1);
               uCopyUTF8Data(m_pLeftText, leftTextCharCount, leftTextCharCount - 1, pEditFieldText, 0, 1);

               uCopyUTF8Data(m_pHighlightText, highlightTextCharCount, 0, pEditFieldText, 1, highlightTextCharCount - 1);
               uCopyUTF8Data(m_pHighlightText, highlightTextCharCount, highlightTextCharCount - 1, m_pEndText, 0, 1);

               StringPlatform::Copy(pEditFieldText, m_pEndText);
               memset(m_pEndText, 0, (EF_MAX_TEXT_LENGTH * MAX_BYTE_FOR_A_CHARACTER) + 1);
               uCopyUTF8Data(m_pEndText, endTextCharCount - 1, 0, pEditFieldText, 1, endTextCharCount - 1);
            }
            // This line will be req if highlight index and count needs to be changed if the insertion of chars is done in between the left or highlight text
            //m_uHighlightStartIndex--;
         }
      }
      FEATSTD_SAFE_DELETE_ARRAY(pEditFieldText);
   }
}


bool EditFieldWidget2D::bPostEditFieldUpdMsg()
{
   Courier::View* view = this->GetParentView();
   if (view != NULL)
   {
      hmibase::widget::editfield::EditFieldUpdMsg* pEditFieldUpdMsg = COURIER_MESSAGE_NEW(hmibase::widget::editfield::EditFieldUpdMsg)(
               ::Courier::ViewId(this->GetParentView()->GetId().CStr()),
               ::Courier::Identifier(this->GetLegacyName()),
               m_sEditFieldInfo, m_oCursorPosition.GetCursorIndex(), m_u8VisibleStartIndex, m_u8VisibleCharCount, m_uHighlightStartIndex, m_uHighlightCount, (NULL != GetNode()) ? (GetNode()->IsEffectiveRenderingEnabled()) : false);
      if (NULL != pEditFieldUpdMsg)
      {
         return pEditFieldUpdMsg->Post();
      }
   }
   return false;
}


/****************************************************************************
*     Function    : vSetMultiLineLayouting
*     Description : Enable MultiLine Layouting of TextWidget
*     Parameters  : bool
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetMultiLineLayouting(const bool multilinelayouting)
{
   m_bMultilineLayouting = multilinelayouting;
   vConfgEntryTxtWidget();
   vUpdateTextNodePosDim();
   vUpdateCursorImage();
}


/****************************************************************************
*     Function    : cobGetMultiLineLayouting
*     Description : Return MultiLine Layouting of TextWidget
*     Parameters  : void
*     Return      : bool
****************************************************************************/
bool EditFieldWidget2D::cobGetMultiLineLayouting() const
{
   return m_bMultilineLayouting;
}


/****************************************************************************
*     Function    : vSetMultiLineLayouting
*     Description : Enable Line Spacing for MultiLine Layouting of TextWidget
*     Parameters  : Candera::UInt16
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetLineSpacing(const Candera::UInt32 lineSpacing)
{
   m_uLineSpacing = lineSpacing;
}


/****************************************************************************
*     Function    : couGetLineSpacing
*     Description : Return Line Spacing for MultiLine Layouting of TextWidget
*     Parameters  : void
*     Return      : Candera::UInt16
****************************************************************************/
Candera::UInt32 EditFieldWidget2D::couGetLineSpacing() const
{
   return m_uLineSpacing;
}


/****************************************************************************
*     Function    : vSetMaximumNumberOfLines
*     Description : Set Maximum Number of Lines for MultiLine Layouting of TextWidget
*     Parameters  : Candera::UInt16
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetMaximumNumberOfLines(const Candera::UInt32 maximumNumberOfLines)
{
   m_uMaxNumberOfLines = maximumNumberOfLines;
}


/****************************************************************************
*     Function    : couGetMaximumNumberOfLines
*     Description : Return Maximum Number of Lines for MultiLine Layouting of TextWidget
*     Parameters  : void
*     Return      : Candera::UInt16
****************************************************************************/
Candera::UInt32 EditFieldWidget2D::couGetMaximumNumberOfLines() const
{
   return m_uMaxNumberOfLines;
}


/****************************************************************************
*     Function    : cobRestrictShiftOperations
*     Description : Interface to get the Restriction on the movement of cursor in highlight area only in EditField text
*     Parameters  : bool
*     Return      : void
****************************************************************************/
bool EditFieldWidget2D::cobRestrictShiftOperations() const
{
   return m_bShiftRestricted;
}


/****************************************************************************
*     Function    : vSetRestrictShiftOperations
*     Description : Interface for setting Restriction on the movement of cursor in highlight area only in EditField text
*     Parameters  : bool
*     Return      : void
****************************************************************************/
void EditFieldWidget2D::vSetRestrictShiftOperations(bool setRestriction)
{
   m_bShiftRestricted = setRestriction;
}
