/* ***************************************************************************************
* FILE:          DirectTextureConsumer2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  DirectTextureConsumer2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "CanderaPlatform/Device/Common/Effects/BitmapBrushBlend.h"
#include "Widgets/2D/DirectTextureConsumer/DirectTextureConsumer2D.h"

#ifndef VARIANT_S_FTR_ENABLE_GADGET_FILE_EXCHANGE
#ifdef VARIANT_S_FTR_ENABLE_USE_EGLKHR_EXTERNAL_TEXTURE_IMAGE
#include "CanderaPlatform/Device/Common/Base/ExternalTextureImage.h"
#include "CanderaPlatform/Device/Common/EGL/EglKhrExternalTextureImage.h"
#include <drm/drm_fourcc.h>
#else
#include "CanderaPlatform/Device/Common/Base/DirectTextureImage.h"
#endif
#endif

#include "CanderaPlatform/Device/Common/Base/TextureImageToImage2DAdaptor.h"
#include "Gadget/SyncBlockConsumer.h"
#include "FeatStd/Platform/PerfCounter.h"

#include "AppBase/ILM_Accessor.h"

#include "View/CGI/CgiExtensions/ViewScene2D.h"

#include "hmibase/gadget/videobuffer/VideoBufferType.h"
#include "View/CGI/CgiExtensions/ImageLoader.h"

// for debugging purpose
#include <inttypes.h>
#include <math.h>
#include <time.h>

using namespace FeatStd;
using namespace Candera;
using namespace hmibase::gadget;

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_DIRECTTEXTURE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/DirectTextureConsumer2D.cpp.trc.h"
#endif

// enable this define to see colored rectangle, where normally the direct texture image is shown.
// Might be usefull on LSim where drm buffer allocation does not work
//#define USE_DUMMY_IMAGE


CGI_WIDGET_RTTI_DEFINITION(DirectTextureConsumer2D)

bool DirectTextureConsumer2D::DebugUtil::s_initialized = false;
std::set<unsigned int> DirectTextureConsumer2D::DebugUtil::s_instanceIdsToDebug;

void DirectTextureConsumer2D::DebugUtil::Capture(unsigned int instanceId, hmibase::gadget::videobuffer::VideoBufferType* buffer)
{
   if (s_initialized == false)
   {
      s_initialized = true;
      // get instance ids to debug from environment
      char* tmp = getenv("WIDGET_DTC_DEBUG");// expect a comma separated list of ids

      if (tmp)
      {
         const char* delim = ",";
         char* token = strtok(tmp, delim);

         while (token)
         {
            int id2log;
            if (sscanf(token, "%d", &id2log) != 1)
            {
               // error
            }
            else
            {
               s_instanceIdsToDebug.insert(id2log);
            }
            token = strtok(NULL, delim);
         }
      }
   }

   if ((s_instanceIdsToDebug.find(instanceId) != s_instanceIdsToDebug.end()) && buffer)
   {
#if !defined(CGI_GPU_SIMULATION)
      char filePath[100];
      static int i = 0;
      struct timespec spec;
      long ms = 0;
      clock_gettime(CLOCK_MONOTONIC, &spec);

      ms = spec.tv_sec * 1000 + static_cast<long>(round(static_cast<double>(spec.tv_nsec) / 1000000.0));
      snprintf(filePath, 100, "/tmp/image_consumer_%04d_%02d__fd_%d__time_%ld.png", ++i, instanceId, buffer->getFd(), ms);

      if (ImageLoader::saveImage(buffer->getDataPtr(), buffer->getDataSize(), buffer->getWidth(), buffer->getHeight(), filePath))
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "drm buffer for instance %d written to %s", instanceId, filePath));
      }
#endif
   }
}


DirectTextureConsumer2D::DirectTextureConsumer2D() :
   Base(),
#if !defined(CGI_GPU_SIMULATION)
   _drmBuffer(0),
   _drmBufferUploaded(0),
#endif
   _touchedInstanceId(-1)
   , _gadgetContentStatus(UNDEFINED)
   , _lastProcessedFrameBuffer(0)
   , _isRegisteredCameraListenener(false)
   , _synchronizedSurface(0)
   , _isSlaveSurfaceIdchanged(false)
{
   SetTouchable(true);
   SetTap(true);
}


DirectTextureConsumer2D::~DirectTextureConsumer2D()
{
   if (IsTouchedInstanceValid())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: [%25s]. touch abort on widget destruction, Instance Id %d.", GetLegacyName(), _touchedInstanceId));
      Courier::Message* msg = COURIER_MESSAGE_NEW(hmibase::gadget::TouchAbortReqMsg)(_touchedInstanceId);

      if (msg)
      {
         msg->Post();
      }
      _touchedInstanceId = -1;
   }

#if !defined(CGI_GPU_SIMULATION)
   if (_drmBuffer)
   {
      if (_drmBuffer != _drmBufferUploaded)
      {
         delete _drmBuffer;
      }
      _drmBuffer = 0;
   }

   if (_drmBufferUploaded)
   {
      delete _drmBufferUploaded;
      _drmBufferUploaded = 0;
   }
#endif

   DeregisterCameraListener();
}


bool DirectTextureConsumer2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const DirectTextureConsumer2D* original = CLONEABLE_WIDGET_CAST<const DirectTextureConsumer2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }

      SetInstanceId(original->GetInstanceId());
      SetCamera(original->GetCamera());
      SetSlaveSurfaceId(original->GetSlaveSurfaceId());
      SetAlignSlaveSurfacePosition(original->GetAlignSlaveSurfacePosition());

      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: [%25s] cloned original %p this %p Instance Id %d", GetLegacyName() , GetOriginal(), this, GetInstanceId()));

      cloned = true;
   }
   return cloned;
}


void DirectTextureConsumer2D::InitWidget()
{
   RegisterCameraListener();
}


WidgetGestureConfig DirectTextureConsumer2D::getDefaultGestureConfig() const
{
   return WidgetGestureConfig(WidgetGestureConfig::Tap(true));
}


void DirectTextureConsumer2D::Update()
{
#if defined USE_DUMMY_IMAGE
   GetBitmapNode()->SetRenderingEnabled(true);
#endif
   if (IsEffectiveRenderingEnabled() == false)
   {
      return;
   }

   if (_isSlaveSurfaceIdchanged && !_isRegisteredCameraListenener)
   {
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: [%25s] RegisterCameraListener success %p this %p SlaveSurfaceId %d", GetLegacyName(), GetOriginal(), this, GetSlaveSurfaceId()));
      RegisterCameraListener();
      _isSlaveSurfaceIdchanged = false;
   }

   bool invalid = false;
   Base::Update();

#if !defined(CGI_GPU_SIMULATION)
   // get current framebuffer
   int frameBuffer = hmibase::gadget::SyncBlockConsumer::getInstance().exchangeFramebuffer((int)GetInstanceId());
#else
   int frameBuffer = -1;
#endif

   if (frameBuffer == _lastProcessedFrameBuffer)
   {
      //no news
      return;
   }

   _lastProcessedFrameBuffer = frameBuffer;

   if (frameBuffer > 0)
   {
      UpdateDRMBuffer(frameBuffer);
#if !defined(CGI_GPU_SIMULATION)
      if (_drmBuffer != 0)
      {
         // buffer is valid, so log in case of first time received
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: [%25s] FB attached to DRM. Handle %d, Instance Id %d", GetLegacyName() , frameBuffer, GetInstanceId()));

         DebugUtil::Capture(GetInstanceId(), _drmBuffer);

         invalid = true;
      }
      else
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Failed to attach drm buffer for handle %d", frameBuffer));
      }
#endif
   }
   else if (frameBuffer == -1)
   {
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: [%25s] NO_BUF received, Instance Id %d", GetLegacyName(), GetInstanceId()));

      UpdateNodes(NO_CONTENT_AVAILABLE);
#if !defined(CGI_GPU_SIMULATION)
      if (_drmBufferUploaded != 0)
      {
         invalid = true;
         _drmBuffer = 0;
      }
#endif
   }
   else
   {
      // unsupported value range
   }

#if defined USE_DUMMY_IMAGE
   UpdateNodes(CONTENT_AVAILABLE);
   invalid = true;
#endif

   if (invalid)
   {
      Candera::RenderNode* renderNode = dynamic_cast<Candera::RenderNode*>(GetBitmapNode());

      if (renderNode != 0)
      {
         BitmapBrushBlend* effect = Dynamic_Cast<BitmapBrushBlend*>(renderNode->GetEffect(0));
         if (effect != 0)
         {
            Candera::BitmapBrush* bitmapBrush = Candera::Dynamic_Cast<Candera::BitmapBrush*>(renderNode->GetEffect(0)->GetBrushEffect2D());

            if (bitmapBrush)
            {
               Candera::Image2D* oldBitmap = bitmapBrush->Image().Get();

               Candera::MemoryManagement::SharedPointer<Candera::Image2D> newBitmap;

#if !defined(CGI_GPU_SIMULATION)
               if (_drmBuffer != 0)
#endif
               {
                  newBitmap = GetBitmap();
               }

               //if both bitmaps are NULL or they point to same image=> do nothing
               if (oldBitmap != newBitmap.GetPointerToSharedInstance())
               {
                  bool invalidateLayout = false;
                  //if one of the bitmaps is NULL and the other one non null or if size is different=>Invalidate layout
                  //otherwise a simple invalidate is enough
                  if ((oldBitmap == NULL)
                        || (newBitmap.GetPointerToSharedInstance() == NULL)
                        || (oldBitmap->GetWidth() != newBitmap->GetWidth())
                        || (oldBitmap->GetHeight() != newBitmap->GetHeight()))
                  {
                     invalidateLayout = true;
                  }

                  bitmapBrush->Unload();

#if !defined(CGI_GPU_SIMULATION)
                  // now we can detach the previously allocated drm buffer
                  if (_drmBufferUploaded != NULL)
                  {
                     delete _drmBufferUploaded;
                     _drmBufferUploaded = NULL;
                  }

                  _drmBufferUploaded = _drmBuffer;
                  _drmBuffer = 0;
#endif

                  bitmapBrush->Image().Set(newBitmap);
                  if (!bitmapBrush->Upload())
                  {
                     ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "Direct texture image attaching failed."));
                  }
                  else
                  {
                     //ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: [%25s], InstanceId %d image uploaded, drmBuffer 0x%x.", GetLegacyName(), GetInstanceId(), _drmBufferUploaded));
                  }

                  if (invalidateLayout)
                  {
                     InvalidateLayout();
                  }
                  else
                  {
                     Invalidate();
                  }
               }
            }
         }
         else
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Attached effect to Node2D is not BitmapBrushBlend!"));
         }
      }
   }
}


Candera::MemoryManagement::SharedPointer<Candera::Image2D> DirectTextureConsumer2D::GetBitmap() const
{
#ifdef USE_DUMMY_IMAGE
   Candera::Bitmap::SharedPointer bitmap;
   Candera::UInt32 width = 64;
   Candera::UInt32 height = 64;

   size_t bufferSize = static_cast<size_t>(width * height * 4);
   FeatStd::UInt8* buffer = CANDERA_NEW_ARRAY(FeatStd::UInt8, bufferSize);
   if (buffer != NULL)
   {
      for (unsigned int i = 0; i < bufferSize; ++i)
      {
         buffer[i]   = ((GetInstanceId() + 10) * 5) % 255; // r
         buffer[++i] = ((GetInstanceId() + 5) * 10) % 255; // g
         buffer[++i] = ((GetInstanceId() + 20) * 15) % 255; // b
         buffer[++i] = 255; // a
      }

      bitmap = Candera::Bitmap::Create(static_cast<Candera::UInt16>(width),
                                       static_cast<Candera::UInt16>(height),
                                       //Candera::Bitmap::RgbaFormat,
                                       Candera::Bitmap::UnsignedByteType,
                                       Candera::Bitmap::PackAlignment1,
                                       buffer, Candera::Bitmap::Disposer::Dispose);
   }

   FeatStd::MemoryManagement::SharedPointer<Candera::BitmapImage2D> image = Candera::BitmapImage2D::Create();
   image->SetBitmap(bitmap);

#else

#ifdef VARIANT_S_FTR_ENABLE_GADGET_FILE_EXCHANGE

   FeatStd::MemoryManagement::SharedPointer<Candera::BitmapImage2D> image = Candera::BitmapImage2D::Create();
   if (!_currentFilePath.empty())
   {
      Candera::Bitmap::SharedPointer bitmap = ImageLoader::loadBitmapFile(_currentFilePath.c_str());

      if (!bitmap.PointsToNull())
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "gadget exchange buffer for instance %d loaded from %s", GetInstanceId(), _currentFilePath.c_str()));
      }
      image->SetBitmap(bitmap);
   }
#else
   TextureImageToImage2DAdaptor::SharedPointer image = TextureImageToImage2DAdaptor::Create();

#ifdef VARIANT_S_FTR_ENABLE_USE_EGLKHR_EXTERNAL_TEXTURE_IMAGE
   FeatStd::MemoryManagement::SharedPointer<Candera::EglKhrExternalTextureImage> _directTextureImage = Candera::EglKhrExternalTextureImage::Create();
#else
   Candera::MemoryManagement::SharedPointer<Candera::DirectTextureImage> _directTextureImage = DirectTextureImage::Create();
#endif

#if !defined(CGI_GPU_SIMULATION)
   if (_drmBuffer != 0)
   {
#ifdef VARIANT_S_FTR_ENABLE_USE_EGLKHR_EXTERNAL_TEXTURE_IMAGE

      EGLint defaultAttribs[13];
      defaultAttribs[0] = EGL_DMA_BUF_PLANE0_FD_EXT;
      defaultAttribs[1] = _drmBuffer->getFd(); // textureImage.GetPlane(index)->m_descriptor;
      defaultAttribs[2] = EGL_DMA_BUF_PLANE0_OFFSET_EXT;
      defaultAttribs[3] = 0; // textureImage.GetPlane(index)->m_offset; may be 0
      defaultAttribs[4] = EGL_DMA_BUF_PLANE0_PITCH_EXT;
      defaultAttribs[5] = _drmBuffer->getPitch(); // textureImage.GetPlane(index)->m_pitch;  start of next line. this could be width * bytesperpixel
      defaultAttribs[6] = EGL_LINUX_DRM_FOURCC_EXT;
      defaultAttribs[7] = DRM_FORMAT_ARGB8888; // textureImage.GetCustomFormat();
      defaultAttribs[8] = EGL_WIDTH;
      defaultAttribs[9] = _drmBuffer->getWidth();// textureImage.GetWidth();
      defaultAttribs[10] = EGL_HEIGHT;
      defaultAttribs[11] = _drmBuffer->getHeight(); // textureImage.GetHeight();
      defaultAttribs[12] = EGL_NONE; //Terminator

      _directTextureImage->SetRole(Candera::EglKhrExternalTextureImage::Consumer); //Can only be set once on purpose!!!!!!!
      _directTextureImage->SetAttributeList(defaultAttribs, sizeof(defaultAttribs) / sizeof(defaultAttribs[0])); // will be copied and stored
      _directTextureImage->SetExternalBufferType(EGL_LINUX_DMA_BUF_EXT); // buffer type
      _directTextureImage->SetExternalBuffer(static_cast<EGLClientBuffer>(0)); // actual buffer

      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer::Update EglKhrExternalTextureImage created, plane.descriptor %d, width %d, height %d",
                          _drmBuffer->getFd(),
                          _drmBuffer->getWidth(),
                          _drmBuffer->getHeight()));
#else
      _directTextureImage->SetWidth(_drmBuffer->getWidth());
      _directTextureImage->SetHeight(_drmBuffer->getHeight());
      _directTextureImage->SetFormat(DirectTextureImage::FormatRGBA);

      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Drm buffer[instanceId %d]: Data*: %p, Geometry %dx%d Size: %u", GetInstanceId(), _drmBuffer->getDataPtr(), _drmBuffer->getWidth(), _drmBuffer->getHeight(), _drmBuffer->getDataSize()));

      //Set logical address of allocated memory for direct texture image.
      void* logical[] = { _drmBuffer->getDataPtr(), NULL, NULL, NULL };
      Candera::UInt physical[4] = { ~static_cast<UInt>(0U), ~static_cast<UInt>(0U), ~static_cast<UInt>(0U), ~static_cast<UInt>(0U) };
      _directTextureImage->SetLogicalAddress(logical);
      _directTextureImage->SetPhysicalAddress(physical);
#endif

      if (!_directTextureImage->Upload())
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "DirectTextureImage Upload FAILED!"));
      }

      image->SetTextureImage(_directTextureImage, static_cast<int>(Candera::Bitmap::RgbaFormat));

      if (!image->Upload())
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "proxy image upload failed"));
      }
   }
#endif
#endif
#endif

   return image;
}


bool DirectTextureConsumer2D::OnTouch(const Camera2D& camera, const Vector2& touchedPoint)
{
   bool isNodeTouched = false;

   if ((GetBitmapNode() != NULL) && (GetBitmapNode()->IsEffectiveRenderingEnabled()))
   {
      isNodeTouched = IsPickIntersectingNode(camera, GetBitmapNode(), touchedPoint);
   }
   //ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: [%20s].Touch Message is received and nodes intesected? Yes:1 NO:0.[%d]", GetLegacyName(), isNodeTouched));

   return isNodeTouched;
}


bool DirectTextureConsumer2D::OnMessage(const Courier::Message& msg)
{
   bool isMsgConsumed = false;
   if (Base::OnMessage(msg))
   {
      return true;
   }

   switch (msg.GetId())
   {
      case hmibase::gadget::ConsumerMsg::ID:
      {
         if (IsEffectiveRenderingEnabled() == false)
         {
            // don't process and consume message
            break;
         }

         const hmibase::gadget::ConsumerMsg* Consmsg = Courier::message_cast<const hmibase::gadget::ConsumerMsg*>(&msg);
         if (NULL != Consmsg)
         {
            std::vector<Courier::Int32> ids = Consmsg->GetInstIds();
            std::vector<Courier::Int32> otherIds;

            bool updated = false;
            for (size_t i = 0; i < ids.size(); ++i)
            {
               if (ids[i] == static_cast<Courier::Int32>(GetInstanceId()))
               {
                  updated = true;
               }
               else
               {
                  otherIds.push_back(ids[i]);
               }
            }
            if (updated)
            {
               if (otherIds.size() > 0)
               {
                  // modify message content to give others the chance to get their ids as well
                  // finally AppViewHandler will receive a vector with ids not being consumed if any and can react accordingly
                  hmibase::gadget::ConsumerMsg* pMsg = const_cast<hmibase::gadget::ConsumerMsg*>(Consmsg);
                  pMsg->SetInstIds(otherIds);
               }
               else
               {
                  isMsgConsumed = true;
               }
               Invalidate();
            }
         }
      }
      break;

      //if any list enters into swiping state it will consume all touch messages
      //=>we have to clear touched/pressed flags if they are set
      case ListStatusUpdMsg::ID:
      {
         const ListStatusUpdMsg* listStatusUpdMsg = Courier::message_cast<const ListStatusUpdMsg*>(&msg);
         if (listStatusUpdMsg != NULL)
         {
            switch (listStatusUpdMsg->GetStatus())
            {
               case ListScrolling:
               case ListSwiping:
                  if (IsTouchedInstanceValid())
                  {
                     ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: [%25s]. ListSwiping received => request touch abort at provider, Instance Id %d.", GetLegacyName(), _touchedInstanceId));
                     Courier::Message* msg = COURIER_MESSAGE_NEW(hmibase::gadget::TouchAbortReqMsg)(_touchedInstanceId);

                     if (msg)
                     {
                        msg->Post();
                     }

                     _touchedInstanceId = -1;
                  }

                  break;

               default:
                  break;
            }
         }
      }
      break;

      case hmibase::gadget::PositionInfoReqMsg::ID:
      {
         const hmibase::gadget::PositionInfoReqMsg* message = Courier::message_cast<const hmibase::gadget::PositionInfoReqMsg*>(&msg);
         if (message != 0)
         {
            if ((GetInstanceId() == message->GetInstanceId()) && (GetBitmapNode() != 0) && IsEffectiveRenderingEnabled())
            {
               Courier::Message* response = COURIER_MESSAGE_NEW(hmibase::gadget::PositionInfoResMsg)(GetInstanceId(), true, GetBitmapNode()->GetWorldPosition());
               if (response)
               {
                  ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: [%25s] post PositionInfoResMsg with InstanceId %d, world position (%.2f, %.2f)", GetLegacyName(), GetInstanceId(), GetBitmapNode()->GetWorldPosition().GetX(), GetBitmapNode()->GetWorldPosition().GetY()));
                  response->Post();
               }
               isMsgConsumed = true;
            }
         }
      }
      break;

      case ::hmibase::gadget::SlaveSurfaceRepositionResMsg::ID:
      {
         const hmibase::gadget::SlaveSurfaceRepositionResMsg* message = Courier::message_cast<const hmibase::gadget::SlaveSurfaceRepositionResMsg*>(&msg);
         if (message)
         {
            if ((message->GetSlaveSurfaceId() == GetSlaveSurfaceId()) && (message->GetActivate() != GetAlignSlaveSurfacePosition()))
            {
               // if widget is part of a list, databinding will not be reflected until next clone happens. So we set the property here directly
               SetAlignSlaveSurfacePosition(message->GetActivate());
               ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: [%25s] AlignSurfaceProperty changed to %u, InstanceId %d, SlaveSurface %d", GetLegacyName(), GetAlignSlaveSurfacePosition(), GetInstanceId(), GetSlaveSurfaceId()));
               Invalidate();

               // don't consume by intent
            }
         }
      }
      break;

      default:
         break;
   }
   return isMsgConsumed;
}


bool DirectTextureConsumer2D::OnTapGesture(const hmibase::input::gesture::GestureEvent& gestureData)
{
   bool ret = false;

   switch (gestureData._gestureState)
   {
      case GestureEvent::ET_START:
      {
         if (!IsTouchedInstanceValid())
         {
            _touchedInstanceId = GetInstanceId();
            const Candera::Vector2 touchPoint((FeatStd::Float)gestureData._pt1.x, (FeatStd::Float)gestureData._pt1.y);
            Candera::Vector2 relPos = touchPoint - GetBitmapNode()->GetWorldPosition();
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: [%25s]. Tap gesture with state %d received. Absolute: %d %d, Relativ: %f %f", GetLegacyName(), ETG_CENUM(GestureState, gestureData._gestureState), gestureData._pt1.x, gestureData._pt1.y, relPos.GetX(), relPos.GetY()));

            ret = true;

            Courier::Message* msg = COURIER_MESSAGE_NEW(hmibase::gadget::TouchForwardReqMsg)(Courier::TouchMsgState::Down, (Courier::XYDIM)relPos.GetX(), (Courier::XYDIM)relPos.GetY(), FeatStd::Internal::PerfCounter::Now(), _touchedInstanceId);

            if (msg)
            {
               msg->Post();
            }
         }
      }
      break;
      case GestureEvent::ET_END:
      {
         if (IsTouchedInstanceValid())
         {
            const Candera::Vector2 touchPoint((FeatStd::Float)gestureData._pt1.x, (FeatStd::Float)gestureData._pt1.y);
            Candera::Vector2 relPos = touchPoint - GetBitmapNode()->GetWorldPosition();
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: [%25s]. Tap gesture with state %d received. Absolute: %d %d, Relativ: %f %f", GetLegacyName(), ETG_CENUM(GestureState, gestureData._gestureState), gestureData._pt1.x, gestureData._pt1.y, relPos.GetX(), relPos.GetY()));

            ret = true;

            Courier::Message* msg = COURIER_MESSAGE_NEW(hmibase::gadget::TouchForwardReqMsg)(Courier::TouchMsgState::Up, (Courier::XYDIM)relPos.GetX(), (Courier::XYDIM)relPos.GetY(), FeatStd::Internal::PerfCounter::Now(), _touchedInstanceId);

            if (msg)
            {
               msg->Post();
            }

            _touchedInstanceId = -1;
         }
      }
      break;
      case GestureEvent::ET_ABORT:
      {
         if (IsTouchedInstanceValid())
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: [%25s].Tap Gesture Abort received => request touch abort at provider, Instance Id %d.", GetLegacyName(), _touchedInstanceId));
            Courier::Message* msg = COURIER_MESSAGE_NEW(hmibase::gadget::TouchAbortReqMsg)(_touchedInstanceId);

            if (msg)
            {
               msg->Post();
            }
            _touchedInstanceId = -1;
         }
      }
      break;
      default:
         break;
   }

   return ret;
}


void DirectTextureConsumer2D::OnChanged(::FeatStd::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   if (propertyId == InstanceIdPropertyId)
   {
      Invalidate();
   }
   else if (propertyId == AlignSlaveSurfacePositionPropertyId)
   {
      if (GetAlignSlaveSurfacePosition()) // invalidate if slave surface position should be aligned
      {
         Invalidate();
      }
   }
   else if (propertyId == SlaveSurfaceIdPropertyId)
   {
      _isSlaveSurfaceIdchanged = true;
   }
}


void DirectTextureConsumer2D::RegisterCameraListener()
{
   // add camera listener
   if (GetSlaveSurfaceId() > 0)
   {
      if (GetCamera() && (GetCamera()->IsTypeOf(Candera::Camera2D::GetTypeId())))
      {
         DeregisterCameraListener();
         _isRegisteredCameraListenener = GetCamera()->AddCameraListener(this);

#if !defined(CGI_GPU_SIMULATION)
         ilmSurfaceProperties properties;
         if (hmibase::ILM_Accessor::getSurfaceProperties(GetSlaveSurfaceId(), &properties))
         {
            _slaveSurfacePosition.SetX(static_cast<Float>(properties.destX));
            _slaveSurfacePosition.SetY(static_cast<Float>(properties.destY));
         }
#endif
      }
   }
}


void DirectTextureConsumer2D::DeregisterCameraListener()
{
   // remove camera listener
   if (_isRegisteredCameraListenener)
   {
      if (GetCamera() && (GetCamera()->IsTypeOf(Candera::Camera2D::GetTypeId())))
      {
         GetCamera()->RemoveCameraListener(this);
         _isRegisteredCameraListenener = false;
      }
   }
}


void DirectTextureConsumer2D::OnParentViewLoad(bool load)
{
   if (load)
   {
      RegisterCameraListener();
   }
   else
   {
      if (IsTouchedInstanceValid())
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: [%25s]. touch abort OnParentViewLoad(false), Instance Id %d.", GetLegacyName(), _touchedInstanceId));
         Courier::Message* msg = COURIER_MESSAGE_NEW(hmibase::gadget::TouchAbortReqMsg)(_touchedInstanceId);

         if (msg)
         {
            msg->Post();
         }

         _touchedInstanceId = -1;
      }

      DeregisterCameraListener();
      _lastProcessedFrameBuffer = 0; // clear last processed fb id to be sure that image will be refreshed when reentering
   }
}


void DirectTextureConsumer2D::OnPreRender(Candera::Camera2D* /*camera*/)
{
}


void DirectTextureConsumer2D::OnPostRender(Candera::Camera2D* camera)
{
   if ((camera == GetCamera()) && (GetSlaveSurfaceId() > 0) && GetBitmapNode())
   {
      if (GetAlignSlaveSurfacePosition() && GetBitmapNode()->IsEffectiveRenderingEnabled())
      {
#if !defined(CGI_GPU_SIMULATION)
         ilmSurfaceProperties properties;
         if (hmibase::ILM_Accessor::getSurfaceProperties(GetSlaveSurfaceId(), &properties))
         {
            Candera::Vector2  slaveSurfaceCurrentPosition;
            slaveSurfaceCurrentPosition.SetX(static_cast<Float>(properties.destX));
            slaveSurfaceCurrentPosition.SetY(static_cast<Float>(properties.destY));

            if (slaveSurfaceCurrentPosition != GetBitmapNode()->GetWorldPosition())
            {
               float x = GetBitmapNode()->GetWorldPosition().GetX();
               float y = GetBitmapNode()->GetWorldPosition().GetY();
               ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D:ilm_surfaceSetPosition ; surfaceId %u, xPos %f, yPos %f", GetSlaveSurfaceId(), x, y));

               if (_synchronizedSurface == 0)
               {
                  Candera::Surface* surface = GetCamera()->GetRenderTarget();
                  if (0 != surface)
                  {
                     Candera::GraphicDeviceUnit* graphicDeviceUnit = surface->GetGraphicDeviceUnit();
                     if (0 != graphicDeviceUnit)
                     {
                        const Candera::MetaInfo::GraphicDeviceUnitMetaInfo* metaInfo = Candera::DevicePackageDescriptor::GetMetaInformation(graphicDeviceUnit->GetUnitType());
                        if (0 != metaInfo)
                        {
                           Candera::MetaInfo::GraphicDeviceUnitPropertyMetaInfo* propMetaInfo = metaInfo->LookupItem("SurfaceId");
                           if (0 != propMetaInfo)
                           {
                              FeatStd::Char buffer[32];
                              buffer[0] = '\0';
                              if (propMetaInfo->Get(graphicDeviceUnit, buffer, 32))
                              {
                                 ::sscanf(buffer, "%u", &_synchronizedSurface);
                                 ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: initialize _synchronizedSurface to %d", _synchronizedSurface));
                              }
                           }
                        }
                     }
                  }
               }

               if (_synchronizedSurface > 0)
               {
                  hmibase::view::RenderJobStrategy::RegisterListener(this);
                  if (hmibase::ILM_Accessor::setSynchronizedSurface(&_synchronizedSurface, 1))
                  {
                     ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: setsynchronizedSurface to %d ", _synchronizedSurface));
                  }
                  else
                  {
                     ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: setsynchronizedSurface to %d failed", _synchronizedSurface));
                  }
               }

               if (hmibase::ILM_Accessor::setSurfacePosition(GetSlaveSurfaceId(), static_cast<int>(x), static_cast<int>(y)))
               {
                  ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: [%25s]. Slave surface (id %u) repositioned, xPos %f, yPos %f", GetLegacyName(), GetSlaveSurfaceId(), x, y));
               }
               else
               {
                  ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: [%25s].SetSurfaceDestinationRectangle return error; surfaceId %u, xPos %f, yPos %f", GetLegacyName(), GetSlaveSurfaceId(), x, y));
               }
            }
         }
         else
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: [%25s].GetSurfaceProperties return error; surfaceId %u", GetLegacyName(), GetSlaveSurfaceId()));
         }
#else
         _slaveSurfacePosition = GetBitmapNode()->GetWorldPosition();
#endif
      }
   }
}


void DirectTextureConsumer2D::OnPostSwapBuffer(Courier::Gdu& /*gdu*/)
{
   if (_synchronizedSurface > 0)
   {
      if (hmibase::ILM_Accessor::removeSynchronizedSurface(&_synchronizedSurface, 1))
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: removeSynchronizedSurface to %d", _synchronizedSurface));
      }
      else
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: removeSynchronizedSurface to %d failed", _synchronizedSurface));
      }
   }
   hmibase::view::RenderJobStrategy::DeregisterListener(this);
}


void DirectTextureConsumer2D::UpdateDRMBuffer(int fb)
{
#if !defined(CGI_GPU_SIMULATION)
   if (fb != -1)
   {
      if (_drmBuffer && (_drmBuffer != _drmBufferUploaded))
      {
         delete _drmBuffer;
         _drmBuffer = NULL;
      }

      if (_drmBuffer == NULL)
      {
         BufferInfoType bi = hmibase::gadget::SyncBlockConsumer::getInstance().getBufferInfo(GetInstanceId());
         _drmBuffer = new hmibase::gadget::videobuffer::VideoBufferType(bi.getWidth(), bi.getHeight(), bi.getDepth(), bi.getBpp());
         _drmBuffer->attach(fb, bi.getSize(), bi.getPitch());

#ifdef VARIANT_S_FTR_ENABLE_GADGET_FILE_EXCHANGE
         std::stringstream ss;
         ss << "/tmp/gadget_" << GetInstanceId() << "_" << fb << ".png";
         _currentFilePath = ss.str();
#endif
      }
      UpdateNodes(CONTENT_AVAILABLE);
   }
   else
   {
      UpdateNodes(NO_CONTENT_AVAILABLE);
   }
#else
   PARAM_UNUSED(fb);
#endif
}


void DirectTextureConsumer2D::UpdateNodes(enGadgetStatus status)
{
   if (_gadgetContentStatus != status)
   {
      _gadgetContentStatus = status;
      hmibase::gadget::gadgetContentStatus consumerContentStatus = (_gadgetContentStatus == CONTENT_AVAILABLE) ? hmibase::gadget::AVAILABLE : hmibase::gadget::UNAVAILABLE;
      Courier::Message* msg = COURIER_MESSAGE_NEW(hmibase::gadget::DirectTextureConsumerContentIndicationMsg)(consumerContentStatus, GetInstanceId());
      if (msg)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "DirectTextureConsumer2D: [%25s] post DirectTextureConsumerContentIndicationMsg with content status = %d, InstanceId %d", GetLegacyName(), consumerContentStatus, GetInstanceId()));
         msg->Post();
      }

      if (GetBitmapNode())
      {
         if (GetReplacementNode() == 0)
         {
            GetBitmapNode()->SetRenderingEnabled(true);
         }
         else
         {
            GetBitmapNode()->SetRenderingEnabled((status == CONTENT_AVAILABLE) ? true : false);
         }
         ETG_TRACE_USR4(("DirectTextureConsumer2D: [%25s] %p Node rendering %d %d instanceId %d, gadget content available %u", GetLegacyName(), GetOriginal(), GetBitmapNode()->IsRenderingEnabled(), GetBitmapNode()->IsEffectiveRenderingEnabled(), GetInstanceId(), status));
      }

      if (GetReplacementNode())
      {
         GetReplacementNode()->SetRenderingEnabled((status == CONTENT_AVAILABLE) ? false : true);
         ETG_TRACE_USR4(("DirectTextureConsumer2D: [%25s] %p ReplacementNode rendering %d %d instanceId %d, gadget content available %u", GetLegacyName(), GetOriginal(), GetReplacementNode()->IsRenderingEnabled(), GetReplacementNode()->IsEffectiveRenderingEnabled(), GetInstanceId(), status));
      }
      Invalidate();
   }
}


bool DirectTextureConsumer2D::IsEffectiveRenderingEnabled()
{
   if ((GetBitmapNode() && GetBitmapNode()->IsEffectiveRenderingEnabled()) ||
         (GetReplacementNode() && GetReplacementNode()->IsEffectiveRenderingEnabled()))
   {
      return true;
   }
   else
   {
      return false;
   }
}


Candera::Node2D* DirectTextureConsumer2D::GetBitmapNode() const
{
   if (GetNode() && GetNode()->GetFirstChild())
   {
      return GetNode()->GetFirstChild();
   }

   return GetNode();
}


Candera::Node2D* DirectTextureConsumer2D::GetReplacementNode() const
{
   return (GetBitmapNode() != 0) ? GetBitmapNode()->GetNextSibling() : 0;
}
