/* ***************************************************************************************
* FILE:          TTraverserBase.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TTraverserBase is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#if !defined(TTraverserBase_H)
#define TTraverserBase_H

#include "TraverserBase.h"

/**
 * The template TTraverserBase is a generic traverser implementation.
 * The item has to implement the methods GetFirstChild, GetNextSibling and GetParent (each returning a pointer of type T).
 */
template<typename T>
class TTraverserBase : public TraverserBase
{
   public:
      TTraverserBase() :
         _root(0)
      {
      }

      /**
       *  Call the Traverse method to traverse in a modifying way.
       */
      void Traverse(T& root)
      {
         CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1764, root, const_cast inside SharedTraversalImplementation)
         SharedTraversalImplementation(root, false);
      }

      /**
       *  Call the Traverse method to traverse in a const way.
       */
      void TraverseConst(const T& root)
      {
         SharedTraversalImplementation(root, true);
      }

   protected:
      /**
       * Implement this method to process the item in a modifying way.
       */
      virtual TraverserAction ProcessNode(T& /*node*/)
      {
         return ProceedTraversing;
      }

      /**
       * Implement this method to process the item in a const way.
       */
      virtual TraverserAction ProcessConstNode(const T& /*node*/)
      {
         return ProceedTraversing;
      }

      /**
       * Returns true if the given node is the root node.
       */
      bool IsRootNode(const T* node) const
      {
         return _root == node;
      }

      /**
       * Returns the root node.
       */
      const T* GetRoot() const
      {
         return _root;
      }

   private:
      void SharedTraversalImplementation(const T& root, bool traverseConst)
      {
         _root = &root;
         OnTraverseBegin();
         const T* currentNode = GetRoot();
         TraverserAction action = ProceedTraversing;
         while ((0 != currentNode) && (action != StopTraversing))
         {
            action = (traverseConst) ? ProcessConstNode(*currentNode) : ProcessNode(const_cast<T&>(*currentNode));
            const T* nextChild = (action == ProceedTraversing) ? currentNode->GetFirstChild() : 0;
            if (0 == nextChild)
            {
               action = ProceedTraversing;
               while (0 == nextChild)
               {
                  if (IsRootNode(currentNode))
                  {
                     nextChild = 0;
                     break;
                  }
                  nextChild = currentNode->GetNextSibling();
                  if (0 == nextChild)
                  {
                     action = OnChildProcessing(false);
                     currentNode = currentNode->GetParent();
                  }
               }
            }
            else
            {
               action = OnChildProcessing(true);
            }
            currentNode = nextChild;
         }
         while ((0 != currentNode) && (!IsRootNode(currentNode)))
         {
            static_cast<void>(OnChildProcessing(false));
            currentNode = currentNode->GetParent();
         }
         OnTraverseEnd(action);
      }

      const T* _root;
};


#endif
