/* ***************************************************************************************
* FILE:          TCloneTraverser.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TCloneTraverser is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#if !defined(TCloneTraverser_H)
#define TCloneTraverser_H


#include "TTraverserBase.h"
#include "Courier/Util/Vector.h"
#include "Candera/Engine2D/Cloning/DeepNode2DCloneStrategy.h"

/**
 * The template TTraverserBase is a generic cloning traverser implementation.
 */
template<typename T>
class TCloneTraverser : public TTraverserBase<T>
{
   public:
      TCloneTraverser() :
         _rootClone(0),
         _lastClone(0)
      {
      }

      T* GetRootClone()
      {
         return _rootClone;
      }

   protected:
      virtual ::TraverserBase::TraverserAction OnChildProcessing(bool start)
      {
         if (start)
         {
            if (!_parentClones.Add(_lastClone))
            {
               Clear();
               return ::TraverserBase::StopTraversing;
            }
         }
         else
         {
            FEATSTD_DEBUG_ASSERT(_parentClones.Size() > 0);
            if (_parentClones.Size() == 0)
            {
               Clear();
               return ::TraverserBase::StopTraversing;
            }
            static_cast<void>(_parentClones.Remove(_parentClones.Size() - 1));
         }
         return ::TraverserBase::ProceedTraversing;
      }

      virtual ::TraverserBase::TraverserAction ProcessNode(T& node)
      {
         T* clone = node.Clone();
         _lastClone = clone;
         if (clone == 0)
         {
            Clear();
            return ::TraverserBase::StopTraversing;
         }

         Candera::DeepNode2DCloneStrategy cloneStrategy;
         Candera::DeepNode2DCloneStrategy::Pair pair(&node, clone);
         cloneStrategy.Execute(pair, pair);

         if (this->IsRootNode(&node))
         {
            _rootClone = clone;
         }
         else
         {
            T* parentClone = _parentClones[_parentClones.Size() - 1];
            if (parentClone != 0)
            {
               static_cast<void>(parentClone->AddChild(clone));
            }
         }
         return OnClone(node, *clone);
      }

      /**
       * OnClone will be called for each cloned node.
       */
      virtual ::TraverserBase::TraverserAction OnClone(const T& /*node*/, T& /*clone*/)
      {
         return ::TraverserBase::ProceedTraversing;
      }

      /**
       * OnClear is used to perform a cleanup if an error situation occurs.
       */
      virtual void OnClear()
      {
      }

      /**
       * Performs a complete cleanup of all allocated resources.
       */
      void Clear()
      {
         OnClear();
         if ((0 != _lastClone) && (0 != _lastClone->GetParent()))
         {
            _lastClone->Dispose();
         }
         _lastClone = 0;
         if (0 != _rootClone)
         {
            _rootClone->Dispose();
            _rootClone = 0;
         }
         _parentClones.Clear();
      }

   private:
      T* _rootClone;
      T* _lastClone;
      ::Courier::Vector<T*> _parentClones;
};


#endif
