/* ***************************************************************************************
* FILE:          PoolCloneCachingStrategy.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PoolCloneCachingStrategy is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "ControlTemplate.h"
#include <Courier/Visualization/ViewScene2D.h>

#include "PoolCloneCachingStrategy.h"

#include "hmi_trace_if.h"
#include "Widgets/widget_etg_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/PoolCloneCachingStrategy.cpp.trc.h"
#endif
#include <Widgets/2D/List/ListWidget2D.h>
#include <Widgets/2D/ControlTemplate/CloneEvents.h>
#include "CloneCachingStrategyHelper.h"


PoolCloneCachingStrategy::PoolCloneCachingStrategy() :
   _cacheSize(0)
{
}


PoolCloneCachingStrategy::~PoolCloneCachingStrategy()
{
}


FeatStd::UInt32 PoolCloneCachingStrategy::GetFirstCachedIndex()
{
   ControlTemplateInstanceMap::iterator startIt = _cache.begin();

   return (startIt != _cache.end()) ? startIt->first : 0;
}


FeatStd::UInt32 PoolCloneCachingStrategy::GetLastCachedIndex()
{
   ControlTemplateInstanceMap::reverse_iterator endIt = _cache.rbegin();
   return (endIt != _cache.rend()) ? endIt->first : 0;
}


ControlTemplateInstancePtr PoolCloneCachingStrategy::Retrieve(Candera::Node2D& templateNode, const tSharedPtrIDataItem& source, BaseWidget2D* owner, FeatStd::UInt32 ownerId, FeatStd::UInt32 index, const ListWidget2DBase::TemplateScrollAnimationType& templateAnimations)
{
   ControlTemplateInstancePtr clonedInstance(0);

   if (!source.PointsToNull())
   {
      ControlTemplateInstanceMap::iterator instanceIt = _cache.find(index);
      if (instanceIt != _cache.end())
      {
         clonedInstance = instanceIt->second;
         CachedInstanceEvent event(clonedInstance, templateNode, source, owner);
         CloneCachingStrategyHelper().UpdateControlTemplateInstance(clonedInstance, event, templateNode, source, owner, templateAnimations);
      }
   }

   return clonedInstance;
}


void PoolCloneCachingStrategy::ProcessAfterCloning(ControlTemplateInstancePtr& instance, FeatStd::UInt32 index)
{
   _cache[index] = instance;
}


bool PoolCloneCachingStrategy::ProcessBeforeDisposed(ControlTemplateInstance* instance)
{
   return true;
}


void PoolCloneCachingStrategy::ProcessBeforeFlush(FeatStd::UInt32 firstIndex, FeatStd::UInt32 lastIndex)
{
   while (_cache.size() > _cacheSize)
   {
      ControlTemplateInstancePtr instanceToDispose(0);

      // If cached size exceeds the buffer size dispose the instance furthest away from the visible area
      FeatStd::UInt32 firstCachedIndex = GetFirstCachedIndex();
      FeatStd::UInt32 lastCachedIndex = GetLastCachedIndex();
      if ((firstIndex - firstCachedIndex) > (lastCachedIndex - lastIndex))
      {
         instanceToDispose = GetFirstCachedInstance();
      }
      else
      {
         instanceToDispose = GetLastCachedInstance();
      }

      DisposeInstance(instanceToDispose);
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PoolCloneCachingStrategy::Flush ended"));
}


void PoolCloneCachingStrategy::Finalize()
{
   _cache.clear();
}


ControlTemplateInstancePtr PoolCloneCachingStrategy::GetFirstCachedInstance()
{
   ControlTemplateInstanceMap::iterator it = _cache.begin();
   return (it != _cache.end()) ? it->second : ControlTemplateInstancePtr(0);
}


ControlTemplateInstancePtr PoolCloneCachingStrategy::GetLastCachedInstance()
{
   ControlTemplateInstanceMap::reverse_iterator it = _cache.rbegin();
   return (it != _cache.rend()) ? it->second : ControlTemplateInstancePtr(0);
}


void PoolCloneCachingStrategy::DisposeInstance(ControlTemplateInstancePtr instance)
{
   if (!instance.PointsToNull())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PoolCloneCachingStrategy::DisposeInstance - disposing instance with index: %d\n", instance->GetIndex()));

      _cache.erase(instance->GetIndex());
   }
}
